<?php

namespace Suiterus\Adg\Models\Timekeeping;

use App\Enums\NightDifferentialRange;
use App\Enums\Status;
use Carbon\Carbon;
use DateInterval;
use Suiterus\Adg\Models\SM\Shift;
use Illuminate\Database\Eloquent\Model;
use Suiterus\Adg\Models\SM\ScheduleTemplate;
use Illuminate\Database\Eloquent\SoftDeletes;
use Suiterus\Adg\Models\Approvals\CTO\CTOApplication;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Suiterus\Adg\Models\LeaveManagement\Requests\Leave;
use Suiterus\Adg\Models\Payroll\NightDifferentialPay;
use Suiterus\Adg\Services\Attendance\AttendanceService;
class Attendance extends Model
{
    use HasFactory, softDeletes;

    protected $connection = 'adg_db';

    protected $table = 'attendances';

    protected $fillable = [
        "user_id",
        "date_in",
        "total",
        "remarks",
        "time_in",
        "time_out",
        "break_in",
        "break_out",
        "device_in",
        "device_out",
        'device_break_in',
        'device_break_out',
        'time_in_state',
        'time_out_state',
        'break_in_state',
        'break_out_state',
        'type',
        "schedule_day_id",
        "timetable_id",
        "shift_id",
        "created_by",
        "updated_by"
    ];

    protected $with = [
        'timetable_basis',
        'shift'
    ];

    protected $appends = [
        'formatted_time_in',
        'formatted_time_out',
        'formatted_date_in',
        'formatted_time_in_hours',
        'formatted_time_out_hours',
        'formatted_break_in_hours',
        'formatted_break_out_hours',
        'formatted_date',
        'formatted_day',
        'late_hours',
        'over_time',
        'under_time',
        'half_day',
        'shift_code',
        'night_differential',
        'night_differential_over_time',
        'previous_time_out',
        'scheduled_time_in',
        'scheduled_time_out',
        'scheduled_break_in',
        'scheduled_break_out',
        'scheduled_max_time_in'

    ];

    public function __construct(array $attributes = []){
        $this->table = env('ADG_DB_DATABASE') . '.attendances';
        parent::__construct($attributes);
    }

    public function user()
    {
        return $this->belongsTo('App\Models\User', 'user_id', 'id');
    }

    public function timetable_basis() {
        return $this->belongsTo(Timetable::class, 'timetable_id', 'id');
    }

    public function schedule_template_basis() {
        return $this->belongsTo(ScheduleTemplate::class, 'schedule_day_id', 'id');
    }

    public function shift() {
        return $this->belongsTo(Shift::class,'shift_id','id');
    }

    public function legends() {
        return $this->hasMany(AttendanceLegend::class, 'attendance_id', 'id');
    }

    public function getFormattedTimeInAttribute() {
        $formatted = date("Y-m-d g:i A", strtotime($this->time_in));
        return $formatted;
    }

    public function getFormattedTimeOutAttribute() {
        $formatted = null;
        if($this->time_out != null) {
            $formatted = date("Y-m-d g:i A", strtotime($this->time_out));
        }
        return $formatted;
    }
    
    public function getFormattedDateInAttribute() {
        $formatted = date("F d, Y", strtotime($this->time_in));
        return $formatted;
    }

    public function getFormattedTimeInHoursAttribute() {
        $formatted = date("g:i A", strtotime($this->time_in));

        return $formatted;
    }

    public function getFormattedTimeOutHoursAttribute() {
        $formatted = null;
        if($this->time_out != null) {
            $formatted = date("g:i A", strtotime($this->time_out));
        }
        return $formatted;
    }

    public function getFormattedBreakInHoursAttribute() {
        $formatted = null;
        if($this->break_in != null) {
            $formatted = date("g:i A", strtotime($this->break_in));
        }
        return $formatted;
    }

    public function getFormattedBreakOutHoursAttribute() {
        $formatted = null;
        if($this->break_out != null) {
            $formatted = date("g:i A", strtotime($this->break_out));
        }
        return $formatted;
    }

    public function getFormattedDateAttribute() {
        $formatted = date("d", strtotime($this->time_in));
        return $formatted;
    }

    public function getFormattedDayAttribute() {
        $formatted = date("l", strtotime($this->time_in));
        return $formatted;
    }
    
    public function getShiftCodeAttribute() {
        if($this->schedule_template_basis && $this->timetable_basis){

             $code = $this->schedule_template_basis->schedulable->shift_code;

             $timeIn = date("g:i A", strtotime($this->timetable_basis->time_in));
             $timeOut = date("g:i A", strtotime($this->timetable_basis->time_out));

             return $code ." ". $timeIn. " to ". $timeOut;
        }
    
        elseif($this->shift) {
            return $this->shift != null ? $this->shift->code.' '. $this->shift->formatted_shift_in.' to '. $this->shift->formatted_shift_out : null;
        }
    }

    public function getLateHoursAttribute() {
        if (!$this->time_out || !$this->getShiftCodeAttribute()) {
            return null;
        }

        $time_in = Carbon::parse($this->time_in)->startOfMinute();
        $time_out = Carbon::parse($this->time_out)->startOfMinute();
        $total_minutes =  0;//$this->getDiffMinutes($time_in, $time_out);

        $req_in = Carbon::parse($this->getScheduledTimeInAttribute())->setDateFrom($time_in);
        $req_out = Carbon::parse($this->getScheduledTimeOutAttribute())->setDateFrom($time_in);

        if ($req_out->lt($req_in)) {
            $req_out->addDay();
        }

        if ($this->break_out && ($this->getScheduledBreakInAttribute() && $this->getScheduledBreakOutAttribute())) {
            $req_break_in = Carbon::parse($this->getScheduledBreakInAttribute())->setDateFrom($this->break_out);
            $req_break_out = Carbon::parse($this->getScheduledBreakOutAttribute())->setDateFrom($this->break_out);
            $break_in = Carbon::parse($this->break_in)->startOfMinute();
            $break_out = Carbon::parse($this->break_out)->startOfMinute();

            if ($req_break_out->lt($req_break_in)) {
                $req_break_out->addDay();
            }

            $break_in_range = $this->inBreakRange($this->break_in, $this->break_out);

            if($break_in_range && $break_out->gt($req_break_out)) {
                $break_mins_range = $this->getDiffMinutes($req_break_out, $break_out);
                $total_minutes += $break_mins_range; 
            } 
            if ($break_in->gte($req_break_out) && $break_out->gt($req_break_out) ) {
                $total_break_minutes = $this->getDiffMinutes($break_in, $break_out);
                $total_minutes += $total_break_minutes;
            }
        }

        if ($time_in->gt($req_in)) {
            $late_minutes = $this->getDiffMinutes($req_in, $time_in);
            if ($this->getScheduledBreakOutAttribute()) {
                $req_break_in = Carbon::parse($this->getScheduledBreakInAttribute())->setDateFrom($req_in);
                $max_late = $this->getDiffMinutes($req_in, $req_break_in);
                if ($late_minutes > $max_late) {
                    $late_minutes = $max_late;
                }
            };
            $total_minutes += $late_minutes;
        }
        
        if ($total_minutes > 0) {
            $hours = intdiv($total_minutes, 60);
            $minutes = $total_minutes % 60;
            $hours_str = sprintf("%02d", $hours);
            $minutes_str = sprintf("%02d", $minutes);
            return $hours_str. ':' .$minutes_str;
        }
    }

    public function getTotalHours($time_in, $time_out) {
        if ($time_out->lessThan($time_in)) {
            $time_out->addDay(); // Add a day to the end time if it's smaller than the start time (crossed midnight)
        }
        $totalDuration  = $time_out->diff($time_in);

        $hours = $totalDuration->h;
        $remaining_minutes = $totalDuration->i;

       return Carbon::createFromTime($hours, $remaining_minutes)->format('H:i');
    }

    public function getOverTimeAttribute() {
        if (!$this->time_out) {
            return null;
        }
        $time_in = Carbon::parse($this->getScheduledTimeInAttribute())->setDateFrom($this->time_in);
        $time_out = Carbon::parse($this->time_out)->startOfMinute();
        $total_minutes = 0;

        $req_in = Carbon::parse($this->getScheduledTimeInAttribute())->setDateFrom($this->time_in);
        $req_out = Carbon::parse($this->getScheduledTimeOutAttribute())->setDateFrom($this->time_in);
        
        if ($req_out->lt($req_in)) {
            $req_out->addDay();
        }

        $attendance_service = new AttendanceService($this->user);
        $rest_day = $attendance_service->determineAbsence($this->time_in);
        if ($rest_day['isRestDay'] || (!$this->getShiftCodeAttribute())) {
            $t_in = Carbon::parse($this->time_in)->startOfMinute();
            $t_out =Carbon::parse($this->time_out)->startOfMinute();

            $ot = $t_in->diff($t_out);
            $hours_str = sprintf("%02d", $ot->h);
            $minutes_str = sprintf("%02d", $ot->i);
            return $hours_str. ':' .$minutes_str;
        }

        if ($time_out->gt($req_out)) {
            $ot_minutes = $this->getDiffMinutes($req_out, $time_out);
            $total_minutes += $ot_minutes; 
        }
        if ($total_minutes > 0) {
            $hours = intdiv($total_minutes, 60);
            $minutes = $total_minutes % 60;
            $hours_str = sprintf("%02d", $hours);
            $minutes_str = sprintf("%02d", $minutes);
            return $hours_str. ':' .$minutes_str;
        }   
    }

    public function getUnderTimeAttribute() {
        if (!$this->time_out || !$this->getShiftCodeAttribute()) {
            return null;
        }

        $time_in = Carbon::parse($this->time_in)->startOfMinute();
        $time_out = Carbon::parse($this->time_out)->startOfMinute();
        $total_minutes = 0; //$this->getDiffMinutes($time_in, $time_out);

        $req_in = Carbon::parse($this->getScheduledTimeInAttribute())->setDateFrom($time_in);
        $req_out = Carbon::parse($this->getScheduledTimeOutAttribute())->setDateFrom($time_in);

        if ($req_out->lt($req_in)) {
            $req_out->addDay();
        }

        if ($this->break_out && ($this->getScheduledBreakInAttribute() && $this->getScheduledBreakOutAttribute())) {
            $req_break_in = Carbon::parse($this->getScheduledBreakInAttribute())->setDateFrom($this->break_out);
            $req_break_out = Carbon::parse($this->getScheduledBreakOutAttribute())->setDateFrom($this->break_out);

            $break_in = Carbon::parse($this->break_in)->startOfMinute();
            $break_out = Carbon::parse($this->break_out)->startOfMinute();
            
            if ($req_break_out->lt($req_break_in)) {
                $req_break_out->addDay();
            }

            $break_in_range = $this->inBreakRange($this->break_in, $this->break_out);

            if($break_in_range && $break_in->lt($req_break_in)) {
                $break_mins_range = $this->getDiffMinutes($break_in, $req_break_in);
                $total_minutes += $break_mins_range;
            } 
            if ($break_in->lt($req_break_in) && $break_out->lte($req_break_in) ) {
                $total_break_minutes = $this->getDiffMinutes($break_in, $break_out);
                $total_minutes += $total_break_minutes;
            }
        }

        if ($time_out->lt($req_out)) {
            $undertime_minutes = $this->getDiffMinutes($time_out, $req_out);
            if ($this->getScheduledBreakOutAttribute()) {
                $req_break_out = Carbon::parse($this->getScheduledBreakOutAttribute())->setDateFrom($req_out);
                $max_undertime = $this->getDiffMinutes($req_break_out, $req_out);
                if ($undertime_minutes > $max_undertime) {
                    $undertime_minutes = $max_undertime;
                }
            };
            $total_minutes += $undertime_minutes; 
        }

        if ($total_minutes > 0) {
            $hours = intdiv($total_minutes, 60);
            $minutes = $total_minutes % 60;
            $hours_str = sprintf("%02d", $hours);
            $minutes_str = sprintf("%02d", $minutes);
            return $hours_str. ':' .$minutes_str;
        }
    }


    private function getDiffMinutes($time_in, $time_out) {
        if ($time_out->lessThan($time_in)) {
            $time_out->addDay(); // Add a day to the end time if it's smaller than the start time (crossed midnight)
        }
        $totalDuration  = $time_out->diffInMinutes($time_in);

       return $totalDuration;
    }

    private function inBreakRange($break_in, $break_out) {
        $break_start_time = Carbon::parse($this->getScheduledBreakInAttribute())->setDateFrom($break_in);
        $break_end_time = Carbon::parse($this->getScheduledBreakOutAttribute())->setDateFrom($break_out);

        $start = Carbon::parse($break_in)->startOfMinute();
        $end = Carbon::parse($break_out)->startOfMinute();

        if ($break_end_time->lt($break_start_time)) {
            $break_end_time->addDay(); // add day if given dates are inaccurate where start time is greater than end time
        }

        if ($end->lt($start)) {
            $end->addDay();
        }

        $break_diff_minutes = 0;

        while ($start->lt($end)) {
            $break_diff_start = $start->copy()->max($break_start_time);
            $break_diff_end = $end->copy()->min($break_end_time);

            if ($break_diff_start->lt($break_diff_end)) {
                $break_diff_minutes += $break_diff_start->diffInMinutes($break_diff_end);
            }

            $start->addDay(); // Move to the next day
        }

        if ($break_diff_minutes == 0) {
            return null;
        }
        return $break_diff_minutes;
    }
    /**
     * ? Do we still need to get the half day attribute for the computation of payroll since the output is the same as 
     * ? the under time attribute?
     * 
     */

     public function getHalfDayAttribute() {
        $time_in = Carbon::parse($this->getScheduledTimeInAttribute())->setDateFrom($this->time_in);
        $time_out = Carbon::parse($this->time_out)->startOfMinute();
        $total_hours = $time_out->diffInSeconds($time_in);
        
        $req_in = Carbon::parse($this->getScheduledTimeInAttribute());
        $req_out = Carbon::parse($this->getScheduledTimeOutAttribute());
        $expected_hours = $req_out->diffInSeconds($req_in) / 2;
        
        if ($total_hours < $expected_hours) { 
            $half_day = $expected_hours * 2 - $total_hours;
            
            return gmdate('H:i', $half_day);
        }
    }

    public function getNightDifferentialAttribute() {
        $night_diff = $this->calculateNightDiff($this->time_in,$this->time_out);

        $night_diff_shift = $this->calculateNightDiff($this->getScheduledTimeInAttribute(), $this->getScheduledTimeOutAttribute());

        if (strtotime($night_diff) >= strtotime($night_diff_shift)) {
            return $night_diff_shift;
        }
        return $night_diff;
    }

    /**
     * Gets the actual time-in and out of attendance and calculate the night differential overtime range
     * and substract the hours and minutes based on acetual overtime
     * @return string formatted string of night differential overtime 
     */
    public function getNightDifferentialOverTimeAttribute() {
        $attendance_service = new AttendanceService($this->user);
        $rest_day = $attendance_service->determineAbsence($this->time_in);
        $night_diff = $this->calculateNightDiff($this->time_in, $this->time_out);

        if (!$night_diff) {
            return null;
        }

        if ($rest_day['isRestDay'] || (!$this->getShiftCodeAttribute())) {
            return $night_diff;
        }

        $overtime = $this->getOverTimeAttribute();
        if(!$overtime) {
            return null;
        }
        
        $night_diff_hours_parsed = $this->calculateNightDiffOvertime($night_diff, $overtime);
        return $night_diff_hours_parsed->format('H:i');
        // to-do should return error if night diff exceed 99 hours;   
    }

    public function getPreviousTimeOutAttribute()
    {
        $timeIn = Carbon::parse($this->time_in);
        $timeOut = Carbon::parse($this->time_out);
        
        // Compare the dates separately
        if ($timeOut->copy()->startOfDay()->gt($timeIn->copy()->startOfDay())) {
            $nextDayTimeOut = $timeOut->format('H:i:s');
        } else {
            $nextDayTimeOut = null; // Handle the case where the time out is on the same day
        }

        return $nextDayTimeOut;
    }

    public function getScheduledTimeInAttribute(){
        if ($this->shift) {
            return $this->shift->shift_in;
        }else if ($this->timetable_basis) {
            return  $this->timetable_basis->time_in;
        }
    }
    
    public function getScheduledTimeOutAttribute() {
        if ($this->shift) {
            return $this->shift->shift_out;
        }else if ($this->timetable_basis) {
            return  $this->timetable_basis->time_out;
        } 
    }

    public function getScheduledBreakInAttribute(){
        if ($this->shift) {
            return $this->shift->break_in;
        }else if ($this->timetable_basis) {
            return  $this->timetable_basis->break_in;
        }
    }
    
    public function getScheduledBreakOutAttribute() {
        if ($this->shift) {
            return $this->shift->break_out;
        }else if ($this->timetable_basis) {
            return  $this->timetable_basis->break_out;
        } 
    }

    public function getScheduledMaxTimeInAttribute() {
        if ($this->shift) {
            return $this->shift->shift_in;
        }else if ($this->timetable_basis) {
            return  $this->timetable_basis->max_time;
        }
    }

    /**
     * Calculate the night differential on given range
     * and loops through each day to compile all minutes rendered in night diff range
     * @param \Carbon\Carbon|\DateTimeInterface @see $time_in, $time_out
     *
     * @return string formatted string of night differential
     */
    private function calculateNightDiff($time_in, $time_out) {
        $active_night_diff = NightDifferentialPay::where('status', Status::ACTIVE)->first();

        $night_start_time = Carbon::parse($active_night_diff->range_start_time)->setDateFrom($time_in);
        $night_end_time = Carbon::parse($active_night_diff->range_end_time)->setDateFrom($time_out);

        $start = Carbon::parse($time_in)->startOfMinute();
        $end = Carbon::parse($time_out)->startOfMinute();

        if ($night_end_time->lt($night_start_time)) {
            $night_end_time->addDay(); // add day if given dates are inaccurate where start time is greater than end time
        }

        if ($end->lt($start)) {
            $end->addDay();
        }

        $night_diff_minutes = 0;

        while ($start->lt($end)) {
            $night_diff_start = $start->copy()->max($night_start_time);
            $night_diff_end = $end->copy()->min($night_end_time);

            if ($night_diff_start->lt($night_diff_end)) {
                $night_diff_minutes += $night_diff_start->diffInMinutes($night_diff_end);
            }

            $start->addDay(); // Move to the next day
        }

        if ($night_diff_minutes == 0) {
            return null;
        }

        $hours = intdiv($night_diff_minutes, 60);
        $remaining_minutes = $night_diff_minutes % 60;
        if ($hours > 99) {
            return null;
        } 
        $night_diff_hours = Carbon::createFromTime($hours, $remaining_minutes, 0)->format('H:i');
        return $night_diff_hours;
    }

    /**
     * Calculate the night differential overtime on given overtime and night diff
     * and subtract all hours and minutes and falls out the range of night diff range
     * @param \DateTimeInterface|string @see $night_diff, $overtime
     * @return \Carbon\Carbon formatted string of night differential overtime
     */
    private function calculateNightDiffOvertime($night_diff, $overtime) {
        //subtract hours and minutes in night differential from attendance schedule (timetable || roster)
        //to get actual night differential overtime
        $active_night_diff = NightDifferentialPay::where('status', Status::ACTIVE)->first();

        $night_diff_hours_parsed = Carbon::parse($night_diff);
        $overtime_parsed = $overtime;
        $scheduled_time_in = Carbon::parse($this->getScheduledTimeInAttribute());
        $scheduled_time_out = Carbon::parse($this->getScheduledTimeOutAttribute());
        $night_diff_start = Carbon::parse($active_night_diff->range_start_time);
        $night_diff_end = Carbon::parse($active_night_diff->range_end_time);
        $time_in = Carbon::parse($this->time_in)->setDateFrom(now())->startOfMinute();
        $time_out = Carbon::parse($this->time_out)->setDateFrom(now())->startOfMinute();

        if ($night_diff_end->lt($night_diff_start)) {
            $night_diff_end->addDay();
        }
        if ($scheduled_time_out->lt($scheduled_time_in)) {
            $scheduled_time_out->addDay(); //add day if time_out is not correct date(time-in and time-out moves forward)
        }

        if ($scheduled_time_out > $night_diff_start) {
            $difference = $night_diff_hours_parsed->diffInMinutes($overtime_parsed);
            $night_diff_hours_parsed->subMinutes($difference);
        }

        if ($time_out->lt($time_in)){
            $time_out->addDay();
        }

        if ($time_out->gt($night_diff_end))
        {
            $difference = $time_out->diffInMinutes($night_diff_end);
            $difference1 = $night_diff_hours_parsed->diffInMinutes($overtime_parsed);
            $night_diff_hours_parsed->subMinutes($difference)->addMinutes($difference1);
        }

        return $night_diff_hours_parsed;
    }
}
