<?php

namespace Suiterus\Adg\Models\Payroll;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Suiterus\Adg\Models\SM\Sector;

class PayrollEmployee extends Model
{
    use HasFactory, SoftDeletes;

    protected $connection = 'adg_db';

    protected $table = 'payroll_employees';

    protected $fillable = [
        'payroll_id',
        'user_id',
        'days_worked',
        'absences',
        'lates',
        'undertimes',
        'stored_basic_salary',
        'basic_pay',
        'gross_pay',
        'deductions',
        'total_basic_pay_deduction',
        'total_pera_pay_deduction',
        'net_pay',
        'void_status',
        'salary_grade',
        'salary_step',
        'created_by',
        'updated_by',
    ];

    protected $with = [
        'payroll',
        'withholding_tax'
    ];

    protected $appends = [
        'basic_salary',
        'old_basic_salary'
    ];

    public function __construct(array $attributes = [])
    {
        $this->table = env('ADG_DB_DATABASE') . '.payroll_employees';
        parent::__construct($attributes);
    }

    // Relationships
    public function user()
    {
        return $this->belongsTo('App\Models\User', 'user_id', 'id');
    }

    public function payroll()
    {
        return $this->belongsTo('Suiterus\Adg\Models\Payroll\Payroll', 'payroll_id', 'id');
    }

    public function timekeeping_deductions()
    {
        return $this->hasMany('Suiterus\Adg\Models\Payroll\TimekeepingDeduction', 'payroll_employee_id', 'id');
    }

    public function lwop_deductions()
    {
        return $this->hasMany('Suiterus\Adg\Models\Payroll\LeaveWithoutPayDeduction', 'payroll_employee_id', 'id');
    }

    public function earning_deduction()
    {
        return $this->hasMany('Suiterus\Adg\Models\Payroll\PayrollEarningDeduction', 'payroll_employee_id', 'id');
    }

    public function withholding_tax()
    {
        return $this->hasOne('Suiterus\Adg\Models\Payroll\WithholdingTax', 'payroll_employee_id', 'id');
    }

    // Accessors / Mutators
    public function getBasicPayAttribute($value)
    {
        return number_format($value, 2);
    }
    public function getGrossPayAttribute($value)
    {
        return number_format($value, 2);
    }
    public function getDeductionsAttribute($value)
    {
        return number_format($value, 2);
    }
    public function getNetPayAttribute($value)
    {
        return number_format($value, 2);
    }
    public function getBasicSalaryAttribute()
    {
        $sector = Sector::where('status', 'active')->first();
        $salary = $this->user->salary;
        $salary = $sector->name == 'public' ? $salary->publicSalary->salaryGrade->getRawOriginal('value') : $salary->privateSalary->salary;
        $type = $this->payroll->type;
        $salary = $type == 1 ? ($salary / 4) : $salary;
        return number_format($salary, 2);
    }

    public function getTotalLwopTime()
    {
        $lwop = $this->timekeeping_deductions()->sum('minutes');
        $hours = floor($lwop / 60);
        $minutes = $lwop % 60;
        if ($minutes < 10) {
            $minutes = '0' . $minutes;
        }
        return $hours . ":" . $minutes;
    }

    public function getOldBasicSalaryAttribute()
    {
        return number_format($this->stored_basic_salary, 2);
    }
}
