<?php

namespace Suiterus\Adg\Models\LeaveManagement\Requests;

use App\Models\User;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Suiterus\Adg\Models\LeaveManagement\Crediting\LeaveCreditDeduction;
use Suiterus\Adg\Models\LeaveManagement\LeaveDetail;
use Suiterus\Adg\Models\LeaveManagement\LeaveSubtype;
use Suiterus\Adg\Models\LeaveManagement\Crediting\LeaveWithoutPay;
use Suiterus\Adg\Controllers\LeaveManagement\Crediting\TerminalLeaveBenefit;
use Suiterus\Adg\Models\LeaveManagement\Crediting\MonetizedLeaveBenefit;
use Suiterus\Adg\Models\LeaveManagement\LeaveAllowance;
use Suiterus\Adg\Models\LeaveManagement\LeaveApplication;
use Suiterus\Adg\Models\LeaveManagement\LeaveApprover;
use Suiterus\Adg\Models\LeaveManagement\LeaveAttachment;
use Suiterus\Adg\Models\LeaveManagement\LeaveType;

class Leave extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'user_id',
        'leave_type_id',
        'leave_detail_id',
        'number_of_days',
        'start_date',
        'end_date',
        'date_filed',
        'date_approved',
        'commutation',
        'status',
        'reason_purpose',
        'remarks',
        'file_name',
        'file_path',
        'metadata',
        'created_by',
        'updated_by'
    ];

    protected $appends = [
        'formatted_start_date',
        'formatted_end_date',
        'formatted_date_filed'
    ];

    protected $with = [
        'approvers'
    ];

    protected $casts = [
        'metadata' => 'json'
    ];

    public function __construct(array $attributes = []) {
        $this->table = env('ADG_DB_DATABASE') . '.leaves';
        parent::__construct($attributes);
    }

    public function user() {
        return $this->belongsTo(User::class, 'user_id', 'id');
    }

    public function forCancellation(){
        return $this->hasOne(LeaveForCancellation::class, 'leave_id', 'id')->latest();
    }

    public function leaveAllowances(){
        return $this->hasMany(LeaveAllowance::class, 'leave_id', 'id');
    }

    public function attachments(){
        return $this->hasMany(LeaveAttachment::class, 'leave_id', 'id');
    }

    public function type() {
        return $this->hasManyThrough(LeaveType::class, LeaveSubtype::class, 'leave_subtype_id', 'leave_type_id', 'id', 'id');
    }

    public function approvers() {
        return $this->hasMany(LeaveApprover::class, 'leave_id', 'id');
    }

    public function leaveType() {
        return $this->belongsTo(LeaveType::class, 'leave_type_id', 'id');
    }

    public function detail() {
        return $this->belongsTo(LeaveDetail::class, 'leave_detail_id', 'id');
    }

    public function credit_deductions() {
        return $this->morphOne(LeaveCreditDeduction::class, 'deductable');
    }

    public function leave_without_pay() {
        return $this->hasOne(LeaveWithoutPay::class, 'leave_id', 'id');
    }

    public function monetization() {
        return $this->hasOne(LeaveHasMonetization::class, 'leave_id', 'id');
    }

    public function monetized_leave_benefit() {
        return $this->hasOneThrough(MonetizedLeaveBenefit::class, LeaveHasMonetization::class, 'leave_id', 'leave_monetization_id', 'id', 'id');
    }

    public function terminal_leave_benefit() {
        return $this->hasOne(TerminalLeaveBenefit::class, 'leave_id', 'id');
    }

    public function application() {
        return $this->hasMany(LeaveApplication::class, 'leave_id', 'id');
    }

    public function getReasonPurposeAttribute($value) {
        if($value == '2') {
            $value = 'Days without pay';
        } else if($value == '1') {
            $value = 'Days with pay';
        }
        return $value;
    }

    public function getFormattedStartDateAttribute() {
        return $this->start_date ? date("F d, Y", strtotime($this->start_date)) : null;
    }

    public function getFormattedEndDateAttribute() {
        return $this->end_date ? date("F d, Y", strtotime($this->end_date)) : null;
    }

    public function getFormattedDateFiledAttribute() {
        return $this->date_filed ? date("F d, Y", strtotime($this->date_filed)) : null;
    }

}

