<?php

namespace Suiterus\Adg\Imports;

use App\Enums\Status;
use App\Enums\Log\PayrollLogType;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use PhpOffice\PhpSpreadsheet\Shared\Date;
use Maatwebsite\Excel\Concerns\Importable;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Payroll\SpecialPay;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithStartRow;
use Maatwebsite\Excel\Concerns\SkipsFailures;
use Maatwebsite\Excel\Concerns\SkipsOnFailure;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithValidation;
use Suiterus\Adg\Models\Payroll\SpecialPayFrequency;
use Maatwebsite\Excel\Concerns\WithCalculatedFormulas;

class SpecialPayImport implements ToCollection, WithStartRow, WithValidation, SkipsOnFailure, WithHeadingRow, WithCalculatedFormulas
{
    use HasCustomLogs;
    private $rows = 0;

    use Importable, SkipsFailures;

    public function sheets(): array
    {
        return [
            0 => $this
        ];
    }

    public function collection(Collection $rows)
    {
        foreach ($rows->chunk(300) as $chunk) {
            foreach($chunk as $row) {
                $specialPay = SpecialPay::create([
                    'title' => $row['title'],
                    'description' => $row['description'],
                    'special_pay_frequency_id' => $this->getFrequencyId($row['special_pay_frequency']),
                    'start_date' => Date::excelToDateTimeObject($row['start_date'])->format('Y-m-d'),
                    'end_date' => Date::excelToDateTimeObject($row['end_date'])->format('Y-m-d'),
                    'amount' => floatval($row['amount']),
                    'period' => $row['period'],
                    'amount_per_frequency' => floatval($row['amount_per_frequency']),
                    'created_by' => Auth::id(),
                    'status' => Status::ACTIVE 
                ]);

                $this->logCustomMessage(
                    'upload_special_pay_template',
                    $specialPay,
                    Auth::user()->name . ' uploaded the Special Pay: ' . $row['title'],
                    $specialPay,
                    PayrollLogType::UPLOAD_SPECIAL_PAY_TEMPLATE,
                    new Activity()
                );
            }
        }
    }

    public function startRow(): int
    {
        return 3; 
    }

    public function headingRow(): int
    {
        return 2;
    }

    public function rules(): array
    {
        return [
            '*.2' => 'in:' . implode(',', $this->getFrequencyOptions()),
            '*.3' => 'date_format:Y-m-d',
            '*.4' => 'date_format:Y-m-d',
            '*.5' => 'numeric',
            '*.7' => 'numeric',
        ];
    }

    public function getRowCount(): int
    {
        return $this->rows;
    }

    private function getFrequencyOptions()
    {
        $specialPayFrequency = SpecialPayFrequency::where('status', Status::ACTIVE)
            ->pluck('title')
            ->toArray();

        return $specialPayFrequency;
    }

    private function getFrequencyId($frequencyTitle)
    {
        return SpecialPayFrequency::where('title', $frequencyTitle)->value('id');
    }
}
