<?php

namespace Suiterus\Adg\Imports;

use Carbon\Carbon;
use App\Models\User;
use App\Enums\ScheduleBasis;
use App\Enums\AttendanceType;

use InvalidArgumentException;
use App\Enums\AttendanceLegend;
use Suiterus\Adg\Models\SM\Shift;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use PhpOffice\PhpSpreadsheet\Shared\Date;
use Maatwebsite\Excel\Concerns\Importable;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithStartRow;

use Maatwebsite\Excel\Concerns\SkipsFailures;
use Suiterus\Adg\Models\EMI\EmployeeMetaInfo;
use Maatwebsite\Excel\Concerns\SkipsOnFailure;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithValidation;
use Suiterus\Adg\Models\Timekeeping\Timetable;
use Suiterus\Adg\Models\Timekeeping\Attendance;
use Maatwebsite\Excel\Concerns\WithMultipleSheets;
use Suiterus\Adg\Services\Attendance\AttendanceService;

class AttendanceImport implements ToCollection, WithStartRow, SkipsOnFailure, WithHeadingRow, WithValidation, WithMultipleSheets
{

    private $rows = 0;

    use Importable, SkipsFailures;

    public function sheets(): array
    {
        return [
            0 => $this
        ];
    }

    /**
    * @param Collection $collection
    */
    public function collection(Collection $rows){
        $employees = EmployeeMetaInfo::distinct('employee_id')
                ->select('id','user_id','employee_id')
                ->without(['branch','corporation','division','department','employeeType'])
                ->get();
        $now = now();

        // Get all employees first and get their schedules
        foreach ($rows->chunk(300) as $chunk) {
            $records = array();
            foreach($chunk as $index => $row){
                $time_in = Date::excelToDateTimeObject($row['time_in'])->format('Y-m-d H:i:s');
                $time_out = $row['time_out'] ? Date::excelToDateTimeObject($row['time_out'])->format('Y-m-d H:i:s') : null;
                $total = $time_out ? Carbon::parse($time_out)->diff($time_in)->format('%H:%I') : '0:00';

                $employeeId = explode(" - ",$row['employee_user']);
                $userKey = array_search($employeeId[0], array_column($employees->all(), 'employee_id'));            
                $user = $employees[$userKey]->user;
                $currentdate = Carbon::parse($time_in)->format('Y-m-d');
  
                // Check first if the date_disabled is not null.
                // If not null, then use it as condition                
                $employee_schedule = $user->temporary_schedule_record()
                    ->where(function ($query) use ($time_in) {
                        $date = date('Y-m-d', strtotime($time_in));
                        $query->where('start_date', '<=', $date)
                            ->whereRaw('CASE WHEN date_disabled is not null then date_disabled >= "' . $date . '" ELSE end_date >= "' . $date . '" END');
                    })->first();

                $sched_timetable = null;
                $schedule = null;
                $break_in = null;
                $break_out = null;
                $attendance_service = new AttendanceService($user);
                $roster_shift = $attendance_service->getRosterShift($currentdate); // get roster shift

                if($roster_shift){
                    $break_in = $roster_shift->break_in ? $roster_shift->break_in : null;
                    $break_out = $roster_shift->break_out ? $roster_shift->break_out : null;
                } elseif ($employee_schedule === null) {
                    $employee_schedule = $attendance_service->getActiveSchedule();
                    if($employee_schedule) {
                        if ($employee_schedule->basis === ScheduleBasis::SCHEDULE) {
                            // Get the schedule template day and the timetables associated with the day
                            $day_of_week = date('l', strtotime($time_in));
                            $schedule = $employee_schedule->schedule_template()->where('day', $day_of_week)->first();
                            $sched_timetable = $schedule ? $schedule->timetables()->whereTime('time_out', '!=', date('H:i:s', strtotime($time_in)))->orderBy('time_in', 'asc')->first() : null;
                            if($sched_timetable) {
                                $break_in = $sched_timetable->break_in ? $sched_timetable->break_in : null;
                                $break_out = $sched_timetable->break_out ? $sched_timetable->break_out : null;
                            }
                            $schedule = $schedule ? $schedule->id : null;
                            $sched_timetable = $sched_timetable ? $sched_timetable->id : null;
                        }
                    } 
                }

                if ($row['break_out'] && $row['break_in']) {
                    $break_in = Date::excelToDateTimeObject($row['break_out'])->format('Y-m-d H:i:s');
                    $break_out = Date::excelToDateTimeObject($row['break_in'])->format('Y-m-d H:i:s');
                }

                if ($break_in && $break_out && $time_out) {
                    $hours_diff = Carbon::parse($break_in)->diffInSeconds(Carbon::parse($break_out));
                    $hours_diff = Carbon::parse(gmdate('H:i', $hours_diff));
                    $total = Carbon::parse($total);
                    $total->subHours($hours_diff->format('H'));
                    $total->subMinutes($hours_diff->format('i'));
                    $total = $total->format('G:i');
                }

                $record = array(
                    'user_id'      => $employees[$userKey]['user_id'],
                    'date_in'      => date('Y-m-d'),
                    'time_in'      => $time_in,
                    'time_out'     => $time_out,
                    'break_in'     => $row['break_out'] ? $break_in : null,
                    'break_out'    => $row['break_in'] ? $break_out : null,
                    'total'        => $total,
                    'schedule_day_id' => $schedule,
                    'timetable_id' => $sched_timetable,
                    'shift_id'     => $roster_shift ? $roster_shift->id : null,
                    'type'         => AttendanceType::MANUAL,
                    'remarks'      => $row['remarks'],
                    'created_by'   => Auth::id(),
                    'updated_by'   => Auth::id(),
                    'created_at'   => $now,
                    'updated_at'   => $now,
                );

                $attendance = Attendance::create($record);
                $this->setLegends([$attendance]);
                $this->rows++;
            }
            
        }
    }
    
    /**
     * @return int
     */
    public function startRow(): int
    {
        return 2;
    }

    public function headingRow(): int
    {
        return 1;
    }

    public function rules(): array
    {

        $rules = array(
            'employee_user'  => 'required|not_in:Select Employee, select employee',
            'time_in'    => 'required',
            //'time_out'   => 'required',
        );

        return $rules;
    }


    public function getRowCount(): int
    {
        return $this->rows;
    }

    /**
     * Check for the active schedule of the employee
     */
    private function checkActiveSchedule($user, $attendance) {
        $today = date('Y-m-d', strtotime($attendance));
        $employee_schedule = $user->employeeSchedules()->without('user')->first();

        if($employee_schedule === null) {
            throw new InvalidArgumentException($user->name . " has no active schedule.");
        }
        return $employee_schedule->schedule_template;
    }

    /**
     * Set legends for the inserted records in bulk
     */
    private function setLegends($attendances) {

        foreach($attendances as $attendance) {
            if ($attendance->timetable_basis || $attendance->shift) {
                if(strtotime(date('H:i:s', strtotime($attendance->time_in))) > strtotime($attendance->scheduled_max_time_in)) {
                    $attendance->legends()->create([
                        'legend' => AttendanceLegend::LATE
                    ]);
                }
                if ($attendance->time_out) {
                    if(strtotime(date('H:i:s', strtotime($attendance->time_out))) < strtotime($attendance->scheduled_time_out)) {
                        $attendance->legends()->create([
                            'legend' => AttendanceLegend::UNDERTIME
                        ]);
                    }
                }
            }

            if (($attendance->break_in && $attendance->break_out) && ($attendance->scheduled_break_in && $attendance->scheduled_break_out) ) {
                if(strtotime(date('H:i:s', strtotime($attendance->break_out))) > strtotime($attendance->scheduled_break_out)) {
                    $attendance->legends()->create([
                        'legend' => AttendanceLegend::LATE
                    ]);
                }
                // Check undertime
                if(strtotime(date('H:i:s', strtotime($attendance->break_in))) < strtotime($attendance->scheduled_break_in)) {
                    $attendance->legends()->create([
                        'legend' => AttendanceLegend::UNDERTIME
                    ]);
                }
            }
        }
    }

    private function getActiveRosterShift($user, $currentdate) {
        $shift = Shift::whereHas('employeeShift',function ($query) use ($currentdate, $user) {
            $query->whereHas('rosterDay', function ($query) use ($currentdate) {
                $query->where('date',$currentdate);
            })->whereHas('rosterEmployeePerGroup', function ($query) use ($user){
                $query->where('user_id',$user->id)->whereHas('employeeGroup.roster', function ($query) {
                    $query->where('status', 1);
                });
            });
        })
        ->orWhereHas('headEmployeeShift', function($query) use ($currentdate, $user) {
            $query->whereHas('rosterDay', function ($query) use ($currentdate) {
                $query->where('date',$currentdate);
            })->whereHas('roster', function ($query) use ($user) {
                $query->where('head_nurse_id', $user->id)
                    ->where('status', 1);
            });
        })->first();
    
        return $shift;
    }
}
