<?php

namespace Suiterus\Adg\Controllers\UserSupervisors;

use Exception;
use App\Models\User;
use Illuminate\Http\Request;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Enums\Log\SuperviseeLogType;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\UserSupervisors\Supervisors;
use Suiterus\Adg\Models\UserSupervisors\UserSupervisors;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;

class UserSupervisorController extends Controller
{
    use HasCustomLogs;

    public function create_multiple(Request $req) {

        $validate = Validator::make($req->all(), [
            'data'                              => 'required|array',

        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();

        try {

            foreach($req->data as $data) {
                $record = [
                    'user_id'                   => $data['user_id'],
                    'supervisor_id'             => $data['supervisor_id'],
                    'status'                    => 1,
                    'created_by'                => $req->user()->id,
                    'created_at'                => now(),
                ];

                $supervisee = UserSupervisors::create($record);
                
                $this->logCustomMessage(
                    'create_supervisee',
                    $supervisee,
                    Auth::user()->name . ' assigned ' . $supervisee->supervisee->name . ' as a Supervisee of ' . $supervisee->supervisor->user->name,
                    $supervisee,
                    SuperviseeLogType::CREATE,
                    new Activity()
                );
            }

            DB::commit();

            return response()->json([
                'success'   => true,
                'text'      => 'Record have been successfully created'
            ]);

        } catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => [ 'There was a problem with creating records.' ],
                'message'   => $e->getMessage()
            ]);
        }

    }

    public function init_list_usc(Request $req){
        $paginate = $req->page_count ? intval($req->page_count) : env('DEFAULT_PAGECOUNT');
        return response()->json([
            'data'  =>  UserSupervisors::orderBy('user_id', 'DESC')->with("supervisor")->paginate($paginate)
        ]);
    }

    public function fetch_supervisee(Request $request){
        $supervisorHasSupervisors = User::where('id', $request->user_id)->has('employeeMetaInfo')->with(['user_supervisor'])
            ->without(['permissions', 'roles'])->first();
        $whereClause = array();
        array_push($whereClause, ['id', '!=', $request->user_id]);
        foreach ($supervisorHasSupervisors['user_supervisor'] as $value) {
            array_push($whereClause, ['id', '!=', $value['supervisor']['user_id']]);
        }
        return User::role(['Employee'])->where($whereClause)->with(['employeeMetaInfo' => function($query) {
            $query->without(['corporation', 'branch', 'division', 'department', 'employeeType', 'position', 'itemCode']);
        }])->without(['permissions', 'roles'])->get();
    }

    public function fetch_supervisee_pagination(Request $request){
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return User::role(['Employee'])->whereHas('user_supervisor', function($query) use($request){
            $query->where('supervisor_id', $request->supervisor_id);
        })->where('id', '!=', $request->user_id)->paginate($paginate);
    }

    public function init_list_all_usc(){
        return User::has('user_supervisor')->get();
    }

    public function list_deleted_usc(){
        return UserSupervisors::onlyTrashed()->with("supervisor")->get();
    }

    public function search_deleted_usc(Request $req){
        return response()->json([
            'data'  =>  UserSupervisors::onlyTrashed()->where('id', '=', $req->id)->with("supervisor")->get()
        ]);
    }



    public function restore_deleted_usc(Request $req){
        return response()->json([
            UserSupervisors::onlyTrashed()->where('id', '=', $req->id)->restore()
        ]);
    }

    public function search_usc(Request $req){
        $paginate = $req->page_count ? intval($req->page_count) : env('DEFAULT_PAGECOUNT');
        return response()->json([
            'data'  =>  UserSupervisors::when($req->name != null && isset($req->name) && $req->name != '', function($query) use($req){
                $query->whereHas('supervisee', function($query) use($req){
                    $query->where('name', 'LIKE', '%'.$req->name.'%');
                });
            })->with(['supervisor', 'supervisee'])->paginate($paginate)

        ]);
    }

    public function edit_usc(Request $req){
        $valid = Validator::make($req->all(),[
            'user_id'      =>  'required|integer|unique:adg_db.user_supervisors,user_id,'.$req->id.',id,deleted_at,NULL',

        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try{
            try{
                $perms = UserSupervisors::findOrFail($req->id);
                $perms->user_id = $req->user_id;
                $perms->supervisor_id = $req->supervisor_id;
                $perms->updated_by = $req->user()->id;

                $updated = UserSupervisors::find($req->id);
                $updated->old = collect($perms);
                $updated->attributes = collect($updated);

                $this->logCustomMessage(
                    'update_supervisee',
                    $updated,
                    Auth::user()->name . ' updated the Supervisor of ' . $perms->supervisee->name,
                    $updated,
                    SuperviseeLogType::UPDATE,
                    new Activity()
                );

                $perms->save();
                DB::commit();
                return response()->json([
                    'text'  =>  $req->id.' has been updated.'
                ]);
            }catch(ME $ee){
                DB::rollback();
                return response()->json([
                    'errors'    =>  [ 'ID doesnt exists.' ],
                ],400);
            }
        }catch(Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'There is a problem in updating a record.' ],
                'msg'       =>  $e->getMessage()
            ],500);
        }
    }

    public function delete_usc(Request $req){
        DB::beginTransaction();
        try{
            $perm = UserSupervisors::findOrFail($req->id);

            $this->logCustomMessage(
                'delete_supervisee',
                $perm,
                Auth::user()->name . ' deleted the record of ' . $perm->supervisee->name,
                $perm,
                SuperviseeLogType::DELETE,
                new Activity()
            );

            $perm->delete();
            DB::commit();
            return response()->json([
                'text'  =>  'Record has been deleted.'
            ]);
        }catch(ME $ee){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Record doesnt exists.' ],
            ],400);
        }
    }

}
