<?php

namespace Suiterus\Adg\Controllers\UserSupervisors;

use Exception;
use App\Models\User;
use Illuminate\Http\Request;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Enums\Log\SupervisorLogType;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\UserSupervisors\Supervisors;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;

class SupervisorsController extends Controller
{
    use HasCustomLogs;

    public function create_supervisor(Request $req) {
        DB::beginTransaction();
        try {
            foreach($req->data as $user){
                $records = [
                    'user_id' => $user['user_id'],
                    'created_by' => $req->user()->id
                ];

                $supervisor = Supervisors::create($records);

                $this->logCustomMessage(
                    'create_supervisor',
                    $supervisor,
                    Auth::user()->name . ' assigned ' . $supervisor->user->name . ' as a Supervisor',
                    $supervisor,
                    SupervisorLogType::CREATE,
                    new Activity()
                );
            }

            DB::commit();
            return response()->json([
                'text'  => ' created successfully!',
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01' ],
                'msg'   =>  $e->getMessage()
            ],500);
        }
    }

    public function init_list_supervisors(Request $req){
        $paginate = $req->page_count ? intval($req->page_count) : env('DEFAULT_PAGECOUNT');
        return response()->json([
            'data'  =>  Supervisors::when($req->name != null && isset($req->name) && $req->name != '', function($query) use($req){
                $query->whereHas('user', function($query) use($req){
                    $query->where('name', 'LIKE', '%'.$req->name.'%');
                });
            })->with("user")->paginate($paginate)
        ]);
    }

    public function init_list_all_supervisors(){
        return Supervisors::orderBy('user_id', 'DESC')->with(['user' => function($query) {
            $query->with(['employeeMetaInfo' => function($query) {
                $query->without(['corporation', 'branch', 'division', 'department', 'employeeType', 'position', 'itemCode']);
            }])->without(['currentRole', 'roles']);
        }, 'supervisee'])->get();
    }

    public function list_all_deleted(){
        return Supervisors::onlyTrashed()->with("user")->get();
    }

    public function search_soft_delete(Request $req){
        return response()->json([
            'data'  =>  Supervisors::onlyTrashed()->where('user_id', '=', $req->user_id)->get()
        ]);
    }

    public function list_candidate_supervisor(){
        return User::has('employeeMetaInfo')->with(['employeeMetaInfo' => function($query) {
            $query->without(['corporation', 'branch', 'division', 'department', 'employeeType', 'position', 'itemCode']);
        }])->doesntHave("supervisor")->without(['permissions', 'roles', 'user_supervisor'])->get();
    }


    public function restore_soft_delete(Request $req){
        return response()->json([
            Supervisors::onlyTrashed()->where('id', '=', $req->id)->restore()
        ]);
    }

    public function search(Request $req){
        $paginate = $req->page_count ? intval($req->page_count) : env('DEFAULT_PAGECOUNT');
        return response()->json([
            'data'  =>  Supervisors::whereHas("user",function($query) use ($req){
                $query->where("name","LIKE","%".$req->name."%");
            })->with("user")->paginate($paginate)
        ]);
    }

    public function edit_supervisor(Request $req){
        $valid = Validator::make($req->all(),[
            'user_id'      =>  'required|string|unique:adg_db.supervisors,user_id,'.$req->id.',id,deleted_at,NULL',

        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try{
            try{
                $perms = Supervisors::findOrFail($req->id);
                $perms->user_id = $req->user_id;
                $perms->updated_by = $req->user()->id;

                $perms->save();
                DB::commit();
                return response()->json([
                    'text'  =>  $req->id.' has been updated.'
                ]);
            }catch(ME $ee){
                DB::rollback();
                return response()->json([
                    'errors'    =>  [ 'Position doesnt exists.' ],
                ],400);
            }
        }catch(Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'There is a problem in updating a Position.' ],
                'msg'       =>  $e->getMessage()
            ],500);
        }
    }

    public function delete_supervisor(Request $req){
        DB::beginTransaction();
        try{
            $perm = Supervisors::findOrFail($req->id);

            $this->logCustomMessage(
                'create_supervisor',
                $perm,
                Auth::user()->name . ' removed ' . $perm->user->name . ' as a Supervisor',
                $perm,
                SupervisorLogType::CREATE,
                new Activity()
            );
            $perm->delete();
            DB::commit();
            return response()->json([
                'text'  =>  'Supervisor has been deleted.'
            ]);
        }catch(ME $ee){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Supervisor doesnt exists.' ],
            ],400);
        }
    }

}
