<?php

namespace Suiterus\Adg\Controllers\Training;

use Exception;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use PhpOffice\PhpSpreadsheet\Reader\Xls\RC4;
use Suiterus\Adg\Models\Training\Training;
use Suiterus\Adg\Models\Training\UserTraining;

class UserTrainingController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return User::role(['Employee'])->whereHas('employeeMetaInfo', function($query) use($request){
            $query->when($request->employee_id != null && $request->employee_id != '', function($query) use($request) {
                if(count($request->employee_id) > 0){
                    $query->whereIn('employee_id', $request->employee_id);
                } else{
                    $query->where('employee_id', $request->employee_id);
                }
            });
            $query->when($request->employeeType != null && $request->employeeType != '', function($query) use($request) {
                $query->where('employee_type', $request->employeeType);
            });
            $query->when($request->position != null && $request->position != '', function($query) use($request) {
                $query->where('position_id', $request->position);
            });
            $query->when($request->department != null && $request->department != '', function($query) use($request) {
                $query->where('department_id', $request->department);
            });
        })->when($request->employeeName != null && $request->employeeName != '', function($query) use($request){
            $query->where('name', 'LIKE', '%'.$request->employeeName.'%');
        })->has('user_training')->with('user_training_latest')->paginate($paginate);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();
        try {
            $records = array();
            foreach ($request->all() as $value) {
                array_push($records, [
                    'user_id' => $value['user_id'],
                    'training_id' => $value['training_id'],
                    'status' => 1,
                    'created_by' => Auth::user()->id,
                ]);
            }

            UserTraining::insert($records);
            
            DB::connection(env('ADG_DB_CONNECTION'))->commit();
            return response()->json([
                'text'  =>  'Success'
            ]);
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollback();
            return response()->json([
                'errors'    =>  [ 'Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01' ],
                'msg'   =>  $e->getMessage()
            ],500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        return User::role(['Employee'])->whereDoesntHave('user_training', function($query) use($request){
            $query->where('training_id', $request->training_id);
        })->get();
    }

    public function training_history(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return UserTraining::where('user_id', $request->employeeID)->paginate($paginate);
    }

    public function fetch_edit_training(Request $request)
    {
        return Training::whereDoesntHave('user_training', function($query) use($request) {
            $query->where('user_id', $request->user_id);
        })->orWhere('training_id', $request->old_training_id)->get();
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {
        DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();
        try {
            UserTraining::where([
                ['training_id', $request->old_training_id],
                ['user_id', $request->user_id]
            ])->update([
                'training_id' => $request->new_training_id
            ]);
            
            DB::connection(env('ADG_DB_CONNECTION'))->commit();
            return response()->json([
                'text'  => "Training updated successfully"
            ]);
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollback();
            return response()->json([
                'errors'    =>  [ 'Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01' ],
                'msg'   =>  $e->getMessage()
            ],500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request)
    {
        $valid = Validator::make($request->all(),[  
            'training_id' => 'required',
            'user_id' => 'required',
            'remarks' => 'required|max:50'
        ]);

        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();
        try {
            UserTraining::where([
                ['user_id', $request->user_id],
                ['training_id', $request->training_id]
            ])->update([
                'remarks' => $request->remarks
            ]);
            UserTraining::where([
                ['user_id', $request->user_id],
                ['training_id', $request->training_id]
            ])->delete();
            DB::connection(env('ADG_DB_CONNECTION'))->commit();
            return response()->json([
                'text'  =>  'Success'
            ]);
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollback();
            return response()->json([
                'errors'    =>  [ 'Can`t update your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01' ],
                'msg'   =>  $e->getMessage()
            ],500);
        }
    }
}
