<?php

namespace Suiterus\Adg\Controllers\Training;
use App\Http\Controllers\Controller;
use Suiterus\Adg\Models\SM\TrainingType;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Suiterus\Adg\Models\Training\Training;
use Exception;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;

class TrainingController extends Controller
{

    public function create(Request $req){
        $valid = Validator::make($req->all(),[  
            'training_id' => 'required|string|unique:'.env('ADG_DB_CONNECTION').'.trainings,training_id',
            'title' => 'required',
            'subject' => 'required',
            'date_start' => 'required|date',
            'date_end' => 'required|date',
            'training_type' => 'required',
            'type_of_learning_development' => 'required',
            'service_obligation_month' => 'required|integer',
            'service_obligation_year' => 'required|integer',
            'conducted_sponsored_by' => 'required',
            'message' => 'nullable',
            'is_manual_input' => 'required|boolean'
        ]);

        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();
        try {
            Training::create([
                'training_id' => $req->training_id,
                'title' => $req->title,
                'subject' => $req->subject,
                'date_start' => $req->date_start,
                'date_end' => $req->date_end,
                'training_type' => $req->training_type,
                'type_of_learning_development' => $req->type_of_learning_development,
                'service_obligation_month' => $req->service_obligation_month ?? 0,
                'service_obligation_year' => $req->service_obligation_year ?? 0,
                'conducted_sponsored_by' => $req->conducted_sponsored_by,
                'message' => $req->message,
                'active' => 1,
                'is_manual_input' => $req->is_manual_input,
                'created_by' => Auth::user()->id
            ]);
            DB::connection(env('ADG_DB_CONNECTION'))->commit();
            return response()->json([
                'text'  =>  $req->training_id.' created successfully.'
            ]);
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollback();
            return response()->json([
                'errors'    =>  [ 'Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01' ],
                'msg'   =>  $e->getMessage()
            ],500);
        }
    }

    public function update_training(Request $req){
        $valid = Validator::make($req->all(),[  
            'training_id' => 'required',
            'title' => 'required',
            'subject' => 'required',
            'date_start' => 'required|date',
            'date_end' => 'required|date',
            'training_type' => 'required',
            'type_of_learning_development' => 'required',
            'service_obligation_month' => 'required|integer',
            'service_obligation_year' => 'required|integer',
            'conducted_sponsored_by' => 'required',
            'message' => 'nullable',
            'is_manual_input' => 'required|boolean'
        ]);

        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();
        try {
            Training::where('training_id', $req->training_id)->update([
                'title' => $req->title,
                'subject' => $req->subject,
                'date_start' => $req->date_start,
                'date_end' => $req->date_end,
                'training_type' => $req->training_type, 
                'type_of_learning_development' => $req->type_of_learning_development,
                'service_obligation_month' => $req->service_obligation_month ?? 0,
                'service_obligation_year' => $req->service_obligation_year ?? 0,
                'conducted_sponsored_by' => $req->conducted_sponsored_by,
                'message' => $req->message,
                'is_manual_input' => $req->is_manual_input,
                'updated_by' => Auth::user()->id
            ]);
            DB::connection(env('ADG_DB_CONNECTION'))->commit();
            return response()->json([
                'text'  =>  $req->training_id.' updated successfully.'
            ]);
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollback();
            return response()->json([
                'errors'    =>  [ 'Can`t update your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01' ],
                'msg'   =>  $e->getMessage()
            ],500);
        }
    }

    public function delete(Request $req){
        $valid = Validator::make($req->all(),[  
            'training_id' => 'required',
        ]);

        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();
        try {
            Training::where('training_id', $req->training_id)->delete();
            DB::connection(env('ADG_DB_CONNECTION'))->commit();
            return response()->json([
                'text'  =>  $req->training_id.' deleted successfully.'
            ]);
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollback();
            return response()->json([
                'errors'    =>  [ 'Can`t update your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01' ],
                'msg'   =>  $e->getMessage()
            ],500);
        }
    }

    public function paginate_training(Request $req){
        $paginate = $req->page_count ? intval($req->page_count) : env('DEFAULT_PAGECOUNT');

        return Training::when($req->filter != null && $req->filter != '', function($query) use($req) {
            $query->when($req->filter == 'Today', function($query) use($req){
                $query->whereDate('created_at', Carbon::today());
            });
            $query->when($req->filter == 'This Week', function($query) use($req){
                $now = Carbon::now();
                $startWeek = $now->startOfWeek()->format('Y-m-d H:i');
                $endWeek = $now->endOfWeek()->format('Y-m-d H:i');
                $query->whereBetween('created_at', [$startWeek, $endWeek]);
            });
            $query->when($req->filter == 'Last Week', function($query) use($req){
                $previousWeek = strtotime("-1 week +1 day");
                $startWeek = strtotime("last sunday midnight",$previousWeek);
                $endWeek = strtotime("next saturday",$startWeek);
                $startWeek = date("Y-m-d",$startWeek);
                $endWeek = date("Y-m-d",$endWeek);
                $query->whereBetween('created_at', [$startWeek, $endWeek]);
            });
            $query->when($req->filter == 'This Month', function($query) use($req) {
                $query->whereYear('created_at', Carbon::now()->year)->whereMonth('created_at', Carbon::now()->month);
            });
            $query->when($req->filter == 'Last Month', function($query) use($req) {
                $query->whereYear('created_at', Carbon::now()->year)->whereMonth('created_at', Carbon::now()->subMonth()->month);
            });
        })->latest()->with("training_type")->paginate($paginate);;
    }

    public function list_trainings(){
        return Training::latest()->with("training_type")->get();
    }

    public function init_list_employee(){
        return response()->json([
            'data'  =>  User::role('Employee')->get()
        ]);
    }

    public function generate_training_id(){
        return $this->generate_string_id();
    }

    public function generate_string_id(){
        $i = 0;
        while ($i < 1000) {
            $date = explode('-', now());
            $year = $date[0];
            $timeStamp = strtotime(now());
            $numeric = str_shuffle($timeStamp);
            $numericKey = '';
            $letters = str_shuffle('ABCDEFGHIJKLMNOPQRSTUVWXYZ');
            $lettersKey = '';

            for ($i=0; $i < strlen($numeric) - 4; $i++) { 
                $numericKey .= $numeric[$i];
            }

            for ($i=0; $i < 2; $i++) { 
                $lettersKey .= $letters[$i];
            }
            
            $trainingID = 'TR-' . $year . '-' . $numericKey . '-' . $lettersKey;

            $isTrainingIDExist = Training::where('training_id', $trainingID)->first() == null;

            if ($isTrainingIDExist) {
                return $trainingID;
            }
            $i++;
        }
        return "There was a problem generating training id";
    }

}
