<?php

namespace Suiterus\Adg\Controllers\Timekeeping;

use App\Enums\Log\ScheduleSetupLogType;
use App\Enums\Status;
use App\Traits\Logs\HasCustomLogs;
use Suiterus\Adg\Models\Timekeeping\Timetable;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;

/** Timetable controller for admin/HR control */

class TimetableController extends Controller
{
    use HasCustomLogs;
    private $db;

    public function __construct(){
        $this->db = DB::connection('adg_db');
    }
    
    /**
     * List all timetables as data table with searching
     */
    public function list(Request $request) {

        $paginate = (isset($request->paginate) && $request->paginate !== null) ? $request->paginate : env('DEFAULT_PAGECOUNT');

        try {
            $data = Timetable::when(isset($request->search) && $request->search !== null, function($query) use ($request) {
                $keyword = '%' . $request->search . '%';
                $query->where('name', 'LIKE', $keyword)
                    ->orWhere('description', 'LIKE', $keyword)
                    ->orWhere('time_in', 'LIKE', $keyword)
                    ->orWhere('max_time', 'LIKE', $keyword)
                    ->orWhere('time_out', 'LIKE', $keyword);
            })->paginate($paginate);

            return response()->json([
                'data'  => $data
            ]);
            
        } catch(Exception $e) {
            return response()->json([
                'errors'    => [ __('responses.exception') ],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Fetch a single timetable record
     */
    public function fetch(Request $request) {

        $validate = Validator::make($request->all(),[
            'id'    => 'required|exists:adg_db.timetables,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        try {
            return response()->json([
                'data'  => Timetable::find($request->id)
            ]);
   
        } catch(Exception $e) {
            return response()->json([
                'errors'    => [ __('responses.exception') ],
                'message'   => $e->getMessage()
            ], 500);         
        }

    }

    /**
     * List active timetables without as raw data
     */
    public function fetchAll() {
        return response()->json([
            'data'  => Timetable::where('status', Status::ACTIVE)->get()
        ]);
    }

    /**
     * Create new timetable record
     */
    public function create(Request $request) {
        
        $validate = Validator::make($request->all(), [
            'name'      => 'required|string|unique:adg_db.timetables,name',
            'time_in'   => 'required',
            'max_time'  => 'required',
            'time_out'  => 'required',
            'status'    => 'required|in:1,2',
            'shift_type'      => 'required|in:1,2,3',
            'work_day'  => 'required',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            // Compute the minutes between the time in and out
            $minutes = $this->compute_minutes($request->time_in, $request->time_out);

            $data = Timetable::create([
                'name'          => $request->name,
                'description'   => $request->description,
                'time_in'       => $request->time_in,
                'max_time'      => $request->max_time,
                'time_out'      => $request->time_out,
                'break_in'      => $request->break_in == null ? null : $request->break_in,
                'break_out'     => $request->break_out == null ? null : $request->break_out,
                'minutes'       => $minutes,
                'status'        => $request->status,
                'shift_type'    => $request->shift_type,
                'work_day'      => $request->work_day,
                'created_by'    => $request->user()->id,
                'updated_by'    => $request->user()->id
            ]);

            $this->logCustomMessage(
                'create_timetable',
                $data,
                Auth::user()->name . ' Create timetable',
                $data,
                ScheduleSetupLogType::CREATE,
                new Activity()
            );

            $this->db->commit();
            return response()->json([
                'text'  => __('responses.success.create')
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Update the selected timetable record
     */
    public function update(Request $request) {
        
        $validate = Validator::make($request->all(), [
            'id'        => 'required|exists:adg_db.timetables',
            'name'      => 'required|string|unique:adg_db.timetables,name,'.$request->id.',id,deleted_at,NULL',
            'time_in'   => 'required',
            'max_time'  => 'required',
            'time_out'  => 'required',
            'status'    => 'required|in:1,2',
            'shift_type'=> 'required|in:1,2,3',
            'work_day'  => 'required',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            // Compute the minutes between the time in and out
            $minutes = $this->compute_minutes($request->time_in, $request->time_out);
            $data = Timetable::findOrFail($request->id);
            $old_record = $data;
            $data->update([
                'name'          => $request->name,
                'description'   => $request->description,
                'time_in'       => $request->time_in,
                'max_time'      => $request->max_time,
                'time_out'      => $request->time_out,
                'break_in'      => $request->break_in == null ? null : $request->break_in,
                'break_out'     => $request->break_out == null ? null : $request->break_out,
                'minutes'       => $minutes,
                'status'        => $request->status,
                'shift_type'    => $request->shift_type,
                'work_day'      => $request->work_day,
                'updated_by'    => $request->user()->id
            ]);

            $this->logCustomMessage(
                'update_timetable',
                $old_record,
                Auth::user()->name . ' Update timetable',
                $data,
                ScheduleSetupLogType::UPDATE,
                new Activity()
            );

            $this->db->commit();
            return response()->json([
                'text'  => __('responses.success.update')
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Delete a timetable record
     */
    public function delete(Request $request) {
        
        $validate = Validator::make($request->all(), [
            'id'        => 'required|exists:adg_db.timetables',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $data = Timetable::whereId($request->id)->delete();

            $this->logCustomMessage(
                'delete_timetable',
                $data,
                Auth::user()->name . ' Delete timetable',
                $data,
                ScheduleSetupLogType::DELETE,
                new Activity()
            );
            $this->db->commit();
            return response()->json([
                'text'  => __('responses.success.delete')
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Compute the minutes between the start and end time
     */
    private function compute_minutes($start, $end) {
        $start_time = Carbon::createFromFormat('H:i', $start);
        $end_time = Carbon::createFromFormat('H:i', $end);
        $minutes = $end_time->diffInMinutes($start_time);

        return $minutes;
    }

}
 