<?php

namespace App\Http\Controllers;

use App\Enums\Status;
use App\Models\User;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use InvalidArgumentException;
use Suiterus\Adg\Models\Timekeeping\TemporarySchedule;

/*
    For controlling the temporary shifts/schedules assigned to the employee.
*/

class TemporaryScheduleController extends Controller
{

    private $db;

    public function __construct(){
        $this->db = DB::connection('adg_db');
    }

    /**
     * Create temporary schedule of the employee with the schedule days and 
     * main details such as effective dates and reason for assigning
     * 
     */
    public function create(Request $request) {

        $validate = Validator::make($request->all(), [
            'user_id'       => 'required|exists:mysql.users,id',
            'reason'         => 'required',
            'effective_dates'    => 'required',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors(),
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $user = User::findOrFail($request->user_id);

            /* Check first if the user already has a temporary schedule */
            if($user->temporary_schedule !== null) {
                throw new InvalidArgumentException('The user already has a temporary schedule');
            }

            /*  
                Create the base temporary schedule record where the reason,
                effective dates, and the status are stored.
            */
            $temporary_schedule = $user->temporary_schedule()->create([
                'reason'         => $request->reason,
                'start_date'    => $request->effective_dates[0],
                'end_date'      => $request->effective_dates[1],
                'status'        => Status::ACTIVE
            ]);

            /*
                Loop through the payload schedules and create the records for the
                days and the times of the temporary schedule
            */
            foreach($request->schedules as $schedule) {
                $record = [
                    'day'           => $schedule['day'],
                    'status'        => Status::ACTIVE,
                    'created_by'    => $request->user()->id,
                    'updated_by'    => $request->user()->id,
                ];
                $template = $temporary_schedule->schedule_template()->create($record);
                $template->timetables()->sync($schedule['timetables']);
            }

            $temporary_schedule->save();
            $this->db->commit();

            return response()->json([
                'text'      => 'Temporary Schedule has been created'
            ]);

        } catch(InvalidArgumentException $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [$e->getMessage()],
                'message'   => $e->getMessage()
            ], 400);
        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in creating the temporary schedule.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Edit the temporary shift assigned to the employee. This includes all the details
     * such as the main details and the days/timetables for the schedule
     * 
     */
    public function edit(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'            => 'required|exists:adg_db.temporary_schedules,id',
            'reason'         => 'required',
            'effective_dates'    => 'required',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors(),
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $temporary_schedule = TemporarySchedule::findOrFail($request->id);
              
            // Edit the basic details of the temporary shift
            $temporary_schedule->update([
                'reason'         => $request->reason,
                'start_date'    => $request->effective_dates[0],
                'end_date'      => $request->effective_dates[1],
            ]);

            /**
             *  Delete or detach all the timetables connected from the temporary schedule days/templates
             *  This will delete the days which have constrained values from the pivot tables of the 
             *  schedule template timetables. Delete the schedule template days afterwards
             * 
             */
            foreach($temporary_schedule->schedule_template as $template) {
                $template->timetables()->detach();
            }
            $temporary_schedule->schedule_template()->delete();

            /** 
             *  Attach new schedule template days and  sync new timetables
             *  Deleting the previous and syncing new records reduces the complexity
             */
            foreach($request->schedules as $schedule) {
                $record = [
                    'day'           => $schedule['day'],
                    'status'        => Status::ACTIVE,
                    'created_by'    => $request->user()->id,
                    'updated_by'    => $request->user()->id,
                ];
                $template = $temporary_schedule->schedule_template()->create($record);
                $template->timetables()->sync($schedule['timetables']);
            }

            $temporary_schedule->save();
            $this->db->commit();
            return response()->json([
                'text'      => 'Temporary Schedule has been updated'
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in editing the temporary schedule.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Fetch the temporary schedule by employee - checks if the employee has one, 
     * returns the main schedule if there is no temporary schedule 
     * 
     */
    public function fetchByEmployee(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'    => 'required|exists:mysql.users,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        try {

            $user = User::findOrFail($request->id);
            /*
                Get the temporary schedule first. If no record is found, get the main schedule instead
            */
            $schedule = $user->temporary_schedule()->with('schedule_template')->first();
            if($schedule === null) {
                /* 
                    If there is no main schedule, throw an exception to tell there is no schedule 
                */
                if(count($user->employeeSchedules) == 0) {
                    throw new InvalidArgumentException('The employee currently has no schedule');
                }
                $schedule = $user->employeeSchedules()->with('schedule_template')->first();
            }

            return response()->json([
                'data'  => $schedule
            ]);

        } catch (InvalidArgumentException $e) {
            return response()->json([
                'errors'    => [$e->getMessage()],
                'message'   => $e->getMessage()
            ], 400);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the record'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    // Set temporary schedule as inactive - will revert to using the main schedule
    public function setAsInactive(Request $request) {
        $validate = Validator::make($request->all(), [
            'id'            => 'required|exists:adg_db.temporary_schedules,id',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors(),
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $schedule = TemporarySchedule::findOrFail($request->id);
            $schedule->status = Status::INACTIVE;
            $schedule->date_disabled = date('Y-m-d');
            $schedule->save();

            $this->db->commit();
            return response()->json([
                'text'  => 'Schedule is set as inactive'
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in editing the temporary schedule.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    // Set temporary schedule as active - will revert to using the main schedule
    public function setAsActive(Request $request) {
        $validate = Validator::make($request->all(), [
            'id'            => 'required|exists:adg_db.temporary_schedules,id',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors(),
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $schedule = TemporarySchedule::findOrFail($request->id);
            $schedule->status = Status::ACTIVE;
            $schedule->save();

            $this->db->commit();
            return response()->json([
                'text'  => 'Schedule is set as active'
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in editing the temporary schedule.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    // Fetch employees who are not assigned to a temporary shift
    public function fetchUnassignedEmployees() {
        try {

            return response()->json([
                'data'  => User::without(['roles', 'storage', 'permissions'])
                    ->whereHas('employeeMetaInfo')
                    ->whereHas('employeeSchedules')
                    ->whereDoesntHave('temporary_schedule')
                    ->with(['employeeMetaInfo' => function ($query) {
                        $query->without('branch', 'corporation', 'division', 'department', 'position')
                            ->select('user_id', 'employee_id');
                    }, 'user_supervisor'])
                    ->get()
            ]);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the records.'],
                'message'   => $e->getMessage()
            ]);
        }
    }

}
