<?php

namespace Suiterus\Adg\Controllers\Timekeeping\Roster;

use Exception;
use Carbon\Carbon;
use Illuminate\Http\Request;
use App\Enums\Log\RosterLogType;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use App\Enums\Log\RosterSetupLogType;
use App\Enums\Log\RosterTemplateLogType;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Services\Roster\RosterService;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterTemplate;

class RosterTemplateController extends Controller
{

    use HasCustomLogs;

    public function index()
    {
        return RosterTemplate::where('created_by', Auth::id())->get();
    }

    public function store(Request $request)
    {

        DB::beginTransaction();
        try {
            $template = RosterTemplate::create([
                'name' => $request->template_title,
                'template' => $request->data,
                'created_by' => Auth::id()
            ]);

            $this->logCustomMessage('create_roster_template', $template, Auth::user()->name . ' created roster template', $template, RosterTemplateLogType::CREATE, new Activity());

            DB::commit();

            return response()->json([
                'message' => 'Successfully created roster template',
                'template' => $template
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'message' => 'An error occurred. Please try again later.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function paginate(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return RosterTemplate::where('created_by', Auth::id())->when($request->department_id != '' && $request->department_id != 'null' && isset($request->department_id), function ($query) use ($request) {
            $query->whereJsonContains('template->department_id', $request->department_id);
        })->when($request->division_id != '' && $request->division_id != 'null' && isset($request->division_id), function ($query) use ($request) {
            $query->whereJsonContains('template->division_id', $request->division_id);
        })->when($request->template_name != '' && $request->template_name != 'null' && isset($request->template_name), function ($query) use ($request) {
            $query->where('name', 'LIKE', "%{$request->template_name}%");
        })->when($request->created_by, function ($query) use ($request) {
            $query->where('created_by', $request->created_by);
        })->with(['createdBy' => function ($query) {
            $query->select('id', 'name')->without([
                'roles',
                'permissions',
                'storage',
                'employeeMetaInfo',
                'training_latest',
                'training_filter'
            ]);
        }])->paginate($paginate);
    }

    public function applyTemplate(Request $request)
    {
        $rosterTemplate = RosterTemplate::find($request->template_id);
        $template = $rosterTemplate->template;

        DB::beginTransaction();
        try {
            $rosterService = RosterService::createRoster([
                "office_id" => $template['office_id'],
                "department_id" => $template['department_id'],
                "division_id" => $template['division_id'],
                "section_id" => $template['section_id'],
                "unit_id" => $template['unit_id'],
                "head_nurse_id" => $template['head_nurse_id'],
                "title" => $request->title,
                "description" => $request->description,
                "number_of_weeks" => $request->number_of_weeks,
                "start_date" => $request->start_date,
                "end_date" => Carbon::parse($request->start_date)->addWeeks($request->partition)->subDays(1),
                "partition" => $request->partition,
                "status" => $template['status'],
                "created_by"    =>  Auth::id()
            ])->createGroups($template['employee_groups'])->createPartitions($request->number_of_weeks, $request->partition, $request->type)->createPartitionDays();

            foreach ($template['employee_groups'] as $group) {

                $rosterGroup = $rosterService->getGroups()->firstWhere('group_id', $group['group_id']);

                if ($rosterGroup) {
                    foreach ($group['employee_group_shifts'] as $shift) {
                        $rosterService->createGroupShifts($shift['shift_id'], $rosterGroup->id);
                    }

                    foreach ($group['employees'] as $employee) {
                        $rosterService->addEmployeeToGroup($employee['user_id'], $rosterGroup->id);
                    }
                }
            }

            DB::commit();

            $this->logCustomMessage(
                'create_roster',
                $rosterService->getRoster(),
                Auth::user()->name . ' Create roster',
                $rosterService->getRoster(),
                RosterLogType::CREATE,
                new Activity()
            );

            DB::commit();

            return response()->json([
                'message' => 'Successfully applied template',
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'There is a problem applying template',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function destroy($id)
    {
        $rosterTemplate = RosterTemplate::find($id);
        $this->logCustomMessage('delete_roster_template', $rosterTemplate, Auth::user()->name . ' delete roster template', $rosterTemplate, RosterSetupLogType::DELETE, new Activity());
        $rosterTemplate->delete();
        return response()->json([
            'message' => 'Successfully deleted roster template'
        ], 200);
    }
}
