<?php

namespace Suiterus\Adg\Controllers\Timekeeping\Roster;

use App\Enums\Log\RosterRescheduleLogType;
use App\Enums\RescheduleStatus;
use App\Http\Classes\NotificationMessage;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Http\Request;
use Exception;
use App\Http\Controllers\Controller;

use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Suiterus\Adg\Controllers\Timekeeping\Roster\Employee\RosterEmployeeController;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\SM\Shift;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterDayEmployeeShift;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterRescheduleRequest;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterRescheduleRequestDate;
use Validator;


class RosterRescheduleController extends Controller
{
    use HasCustomLogs;
    public function list_request_schedule(Request $request) {
        $paginate = $request->page_count ? intval($request->page_count) : ENV('DEFAULT_PAGECOUNT');

        $user = User::where('id', Auth::user()->id)->first();
        $actualDesignation = $user->actualDesignation;
        $data = RosterRescheduleRequest::whereHas('rosterDay.rosterPartition.roster', function ($query) use ($user, $actualDesignation) {
            $query->when(isset ($user->actualDesignation), function ($query) use ($actualDesignation) {
                $query->where('head_nurse_id', Auth::id())
                    ->when(isset ($actualDesignation), function ($query) use ($actualDesignation) {
                        $fields = ['office_id', 'department_id', 'division_id', 'section_id', 'unit_id'];
                        foreach ($fields as $field) {
                            $query->when(isset ($actualDesignation->$field), function ($query) use ($field, $actualDesignation) {
                                $query->where($field, $actualDesignation->$field);
                            });
                        }
                    });
            });
        })->when($request->status != null, function ($query) use ($request) {
            $query->where('status', $request->status);
        })->when($request->date_request, function ($query) use ($request) {
            $query->where('request_date', $request->date_request);
        })->whereHas('user', function ($query) use ($request) {
            $query->where('name', 'LIKE', '%' . $request->keyword . '%');
        })->with('user','requestee', 'rosterDay.rosterPartition.roster', 'oldShift', 'newShift')->paginate($paginate);

        
        return response()->json([
            'data'  =>  $data,
        ], 200); 
    }

    public function approve_request(Request $request) {

        $validate = Validator::make($request->all(), [
            'reschedule_id'    => 'required|exists:' . env('ADG_DB_CONNECTION') . '.roster_reschedule_requests,id',
            'status' => 'required|numeric'
        ]); 

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $reschedule = RosterRescheduleRequest::findOrFail($request->reschedule_id);

            if ($request->status == RescheduleStatus::HR_APPROVED) {
                // Find the user schedule
                $userSched = RosterDayEmployeeShift::where([['roster_day_id', $reschedule->request_date],
                    ['shift_id', $reschedule->old_shift_id]])
                ->wherehas('rosterEmployeePerGroup', function ($query) use ($reschedule) {
                    $query->where('user_id', $reschedule->user_id);
                })->firstOrFail();

                // Find the requestee schedule
                $requesteeSched = RosterDayEmployeeShift::where('roster_day_id', $reschedule->request_date)
                ->wherehas('rosterEmployeePerGroup', function ($query) use ($reschedule) {
                    $query->where('user_id', $reschedule->requestee_id);
                })->firstOrFail();

                // Get the shifts only once
                $oldShift = $reschedule->old_shift_id ? Shift::find($reschedule->old_shift_id) : null; //user
                $newShift = $reschedule->new_shift_id ? Shift::find($reschedule->new_shift_id) : null; //requestee

                if (is_null($reschedule->new_shift_id)) {
                    // Rescheduling without a new shift
                    $requesteeSched->update([
                        'remarks' => $oldShift->code,
                        'shift_id' => $reschedule->old_shift_id,
                        'roster_legend_id' => $userSched->roster_legend_id
                    ]);

                    $userSched->update([
                        'remarks' => null,
                        'shift_id' => null,
                        'roster_legend_id' => null,
                    ]);

                } elseif (is_null($reschedule->old_shift_id)) {
                    // Rescheduling without an old shift
                    $userSched->update([
                        'remarks' => $newShift->code,
                        'shift_id' => $reschedule->new_shift_id,
                        'roster_legend_id' => $requesteeSched->roster_legend_id
                    ]);

                    $requesteeSched->update([
                        'remarks' => null,
                        'shift_id' => null,
                        'roster_legend_id' => null,
                    ]);
                } else {
                // Full rescheduling
                    $userOldSched = clone $userSched;
                    
                    $userSched->update([
                        'remarks' => $newShift->code,
                        'shift_id' => $reschedule->new_shift_id,
                        'roster_legend_id' => $requesteeSched->roster_legend_id
                    ]);

                    $requesteeSched->update([
                        'remarks' => $oldShift->code,
                        'shift_id' => $reschedule->old_shift_id,
                        'roster_legend_id' => $userOldSched->roster_legend_id
                    ]);
                }
            }
            
            $reschedule->status = $request->status;
            $reschedule->justification = $request->justification;
            $reschedule->save();
            
            $this->logCustomMessage(
                'set_reschedule_request_status',
                $reschedule,
                Auth::user()->name . ' Set reschedule request status',
                $reschedule,
                RosterRescheduleLogType::SET_STATUS,
                new Activity()
            );

            $request->status == RescheduleStatus::HR_APPROVED ? NotificationMessage::notifySenderAndReceiver('Reschedule Approved', $reschedule->user_id, Auth::id()) :
                NotificationMessage::notifySenderAndReceiver('Reschedule Declined', $reschedule->user_id, Auth::id());
 
            DB::commit();       
            return response()->json([
                'message' => 'Reschedule'. $request->status == RescheduleStatus::HR_APPROVED ? "approved" : "declined" 
            ], 200);

        }catch(Exception $e) {
            DB::rollBack();       
            return response()->json([
                'errors'    => ['The selected record could not be found'],
                'message'   => $e->getMessage(),
                'line'   => $e->getLine()
            ], 500);
        }
    }

    public function requestee_approve_request(Request $request) {

        $validate = Validator::make($request->all(), [
            'reschedule_id'    => 'required|exists:' . env('ADG_DB_CONNECTION') . '.roster_reschedule_requests,id',
            'status' => 'required|numeric'
        ]); 

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $reschedule = RosterRescheduleRequest::findOrFail($request->reschedule_id);
            
            $reschedule->update([
                'status' => $request->status == RescheduleStatus::REQUESTEE_APPROVED ? RescheduleStatus::REQUESTEE_APPROVED : RescheduleStatus::REQUESTEE_DECLINED,
                'justification' => $request->justification
            ]);
            $reschedule->save();

            $request->status == RescheduleStatus::REQUESTEE_APPROVED ? NotificationMessage::notifySenderAndReceiver('Reschedule Approved', $reschedule->user_id, Auth::id()) :
                NotificationMessage::notifySenderAndReceiver('Reschedule Declined', $reschedule->user_id, Auth::id());

            DB::commit();       
            return response()->json([
                'message' => 'Reschedule'. $request->status == RescheduleStatus::REQUESTEE_APPROVED ? "approved" : "declined" 
            ], 200);

        }catch(Exception $e) {
            DB::rollBack();       
            return response()->json([
                'errors'    => ['The selected record could not be found'],
                'message'   => $e->getMessage(),
            ], 500);
        }
    }

    public function fetch_requestor_request() {
        return RosterRescheduleRequest::where('requestee_id', Auth::user()->id)
            ->with(['user','requestee','rosterDay','newShift','oldShift'])->paginate(5);
        
    }
}
