<?php

namespace Suiterus\Adg\Controllers\Timekeeping\Roster;

use Exception;
use Validator;
use Carbon\Carbon;
use App\Models\User;
use Illuminate\Http\Request;
use App\Enums\Log\RosterLogType;
use Suiterus\Adg\Models\SM\Shift;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use Suiterus\Hrjp\Models\Position;
use Suiterus\Adg\Models\SM\Holiday;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Services\Roster\RosterService;
use Suiterus\Adg\Models\Timekeeping\Roster\Roster;
use Suiterus\Adg\Models\Approvals\CTO\CTOApplication;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterDay;
use Suiterus\Adg\Models\LeaveManagement\Requests\Leave;
use Suiterus\Adg\Services\Attendance\AttendanceService;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterLegend;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterPartition;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterStaffPerDay;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterEmployeeGroup;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterDayEmployeeShift;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterEmployeesPerGroup;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterHeadEmployeeShift;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterHeadRequiredHours;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterEmployeeGroupShift;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterEmployeeRequiredHours;

class RosterController extends Controller
{
    use HasCustomLogs;

    public function index()
    {
        return Roster::all();
    }

    public function get_groups($rosterId)
    {
        return RosterEmployeeGroup::where('roster_id', $rosterId)->with(['rosterGroup', 'employees.user'])->get();
    }

    public function search_roster(Request $req)
    {
        $paginate = $req->page_count ? intval($req->page_count) : ENV('DEFAULT_PAGECOUNT');
        $user = User::where('id', Auth::id())->first();
        $actualDesignation = $user->actualDesignation;

        $data = Roster::where('title', 'LIKE', '%' . $req->keyword . '%')
            ->when($user->roles->whereNotIn('name', ['Super Admin', 'Admin', 'Developer']) && $actualDesignation, function ($query) use ($actualDesignation) {
                $fields = ['office_id', 'department_id', 'division_id', 'section_id', 'unit_id'];
                foreach ($fields as $field) {
                    $query->when(isset ($actualDesignation->$field), function ($query) use ($field, $actualDesignation) {
                        $query->where($field, $actualDesignation->$field);
                    });
                }
            })->when($req->division != null && $user->roles->whereIn('name', ['Super Admin', 'Admin', 'Developer']), function ($query) use ($req) {
                $query->whereHas('division', function ($query) use ($req) {
                    $query->whereId($req->division);
                });
            })->when($req->department != null && $user->roles->whereIn('name', ['Super Admin', 'Admin', 'Developer']), function ($query) use ($req) {
                $query->whereHas('department', function ($query) use ($req) {
                    $query->whereId($req->department);
                });
            })->when($req->created_by, function ($query) use ($req) {
                $query->where('created_by', $req->created_by);
            })->with(['user' => function ($query) {
                $query->select('id', 'name')->without([
                    'roles',
                    'permissions',
                    'storage',
                    'employeeMetaInfo',
                    'training_latest',
                    'training_filter'
                ]);
            }])->when($req->date != null, function ($query) use ($req) {
                $startDate = Carbon::parse($req->date[0])->startOfDay();
                $endDate = Carbon::parse($req->date[1])->endOfDay();
                $query->whereDate('start_date', '<=', $startDate)
                    ->whereDate('end_date', '>=', $endDate);
            })->with(
                'partitions.days.employeeShifts',
                'partitions.days.staffPerDays',
                'division',
                'employeeGroups',
                'department',
                'section',
                'unit'
            )
            ->orderBy('created_at', 'desc')
            ->paginate($paginate);

        return response()->json([
            'data'  =>  $data,
        ]);
    }
    public function create_roster(Request $req)
    {

        $valid = Validator::make($req->all(), [
            'title' => 'required|string',
            'roster_description' => 'required|string',
            'number_of_weeks' => 'required|numeric',
            'start_date' => 'required',
            'partition' => 'required',
            'status' => 'required|numeric',
            'type' => 'required|string'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }

        if ($req->number_of_weeks % $req->partition != 0) {
            return response()->json([
                'error'    => "Number of weeks should be divisible by partition"
            ], 400);
        }

        DB::beginTransaction();
        try {

            $groups = array_map(function ($group) {
                return array(
                    'group_id' => $group['id'],
                    'description' => $group['description']
                );
            }, $req->groups);

            $rosterService = RosterService::createRoster([
                'office_id' => $req->office_id,
                'department_id' => $req->department_id,
                'division_id' => $req->division_id,
                'section_id' => $req->section_id,
                'unit_id' => $req->unit_id,
                'head_nurse_id' => $req->head_nurse_id,
                'title' => $req->title,
                'description' => $req->roster_description,
                'number_of_weeks' => $req->number_of_weeks,
                'start_date' => $req->start_date,
                'end_date' => Carbon::parse($req->start_date)->addWeeks($req->number_of_weeks)->subDays(1),
                'partition' => $req->partition,
                'status' => 2,
                'created_by' => Auth::id()
            ])->createGroups($groups)->createPartitions($req->number_of_weeks, $req->partition, $req->type)->createPartitionDays();

            $this->logCustomMessage(
                'create_roster',
                $rosterService->getRoster(),
                Auth::user()->name . ' Create roster',
                $rosterService->getRoster(),
                RosterLogType::CREATE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'message' => 'Successfully created Roster',
                'data' => $rosterService->getRoster()
            ], 200);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in creating roster schedule.'],
                'msg'       =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function create_initial_head_emp_shift($partitions, $data)
    {
        foreach ($partitions as $partition) {
            RosterHeadRequiredHours::create([
                'roster_partition_id' => $partition->id,
                'roster_id' => $partition->roster_id
            ]);
            foreach ($partition['days'] as $day) {
                list($remarks, $legend) = $this->set_remarks_and_legends($data->head_nurse_id, $day['date']);
                RosterHeadEmployeeShift::create([
                    'roster_id' => $partition->roster_id,
                    'roster_day_id' => $day['id'],
                    'remarks' => $remarks,
                    'roster_legend_id' => $legend
                ]);
            }
        }
    }

    public function fetch_head_employee(Request $request)
    {
        $employee = User::whereHas('employeeMetaInfo', function ($query) use ($request) {
            $query->where([
                ['department_id', $request->department_id],
                ['division_id', $request->division_id]
            ])->whereHas('position', function ($query) {
                $query->whereHas('rosterGroup', function ($query) {
                    $query->where('name', 'Head Employees');
                });
            });
        })->select('id', 'name')->without('roles', 'permissions', 'storage', 'employeeMetaInfo', 'supervisor', 'exitInterview', 'userProfilePicture', 'profileBasicInfo')->get();

        return response()->json(['data' => $employee], 200);
    }

    public function assign_head_employee(Request $request)
    {
        DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();
        try {
            $roster = Roster::where('id', $request->roster_id)->first();
            $oldRoster = clone $roster;

            $roster->update([
                'head_nurse_id' => $request->employee
            ]);

            $roster->old = collect($oldRoster);
            $roster->attributes = collect($roster);

            $this->logCustomMessage(
                'assign_head_employee',
                $roster,
                Auth::user()->name . " assign {$roster->headNurse->name} as head employee of {$roster->title}",
                $roster,
                'Assign head employee',
                new Activity()
            );
            DB::connection(env('ADG_DB_CONNECTION'))->commit();
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in assigning head employee.'],
                'msg'       =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function remove_head_employee(Request $request)
    {
        DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();
        try {
            $roster = Roster::where('id', $request->roster_id)->first();
            $oldRoster = clone $roster;

            $roster->update([
                'head_nurse_id' => null
            ]);

            $roster->old = collect($oldRoster);
            $roster->attributes = collect($roster);

            $this->logCustomMessage(
                'remove_head_employee',
                $roster,
                Auth::user()->name . " remove {$oldRoster->headNurse->name} as head employee of {$roster->title}",
                $roster,
                'Remove head employee',
                new Activity()
            );
            DB::connection(env('ADG_DB_CONNECTION'))->commit();
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in removing head employee.'],
                'msg'       =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function assign_head_employee_shift(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'roster_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.rosters,id',
            'shift_id.id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.shifts,id',
            'roster_day_id' => 'required|array|exists:' . env('ADG_DB_CONNECTION') . '.roster_days,id'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $roster = Roster::findOrFail($request->roster_id);
            foreach ($request->roster_day_id as $day_id) {
                $rosterDay = RosterDay::findOrFail($day_id);

                list($leave, $cto, $holiday) = $this->get_cto_leave_holiday($roster->head_nurse_id, $rosterDay->date);
                if (empty($leave) && empty($cto)) {
                    $shift = Shift::findOrFail($request->shift_id['id']);
                    RosterHeadEmployeeShift::updateOrCreate(
                        [
                            'roster_id' => $request->roster_id,
                            'roster_day_id' => $day_id,
                        ],
                        [
                            'shift_id' => $shift->id,
                            'remarks' => $shift->code
                        ]
                    );
                }
            }

            $user = User::whereId($roster->head_nurse_id)->first();
            $attendance_service = new AttendanceService($user);
            foreach ($request->roster_day_id as $day_id) {
                $rosterDay = RosterDay::findOrFail($day_id);
                $attendance_service->reEvaluateAttendance($rosterDay->date);
            }

            $this->update_head_required_hours_per_partition($request->roster_id);

            $this->logCustomMessage(
                'assign_head_employee_shift',
                null,
                Auth::user()->name . ' Assign head employee shift ',
                null,
                RosterLogType::ASSIGN,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'message' => 'Successfully assigned Shift/s',
            ], 200);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem assigning shifts of head employee.'],
                'msg'       =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    //* for assigning of details for group in a roster
    public function set_details(Request $request)
    {

        $valid = Validator::make($request->all(), [
            'roster_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.rosters,id',
            'roster_group_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.roster_employee_groups,id',
            'shifts' => 'array|required',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $roster = Roster::whereId($request->roster_id)->first();
            $group = RosterEmployeeGroup::findOrFail($request->roster_group_id);

            foreach ($request->shifts as $shift) {
                $group_shift = RosterEmployeeGroupShift::updateOrCreate([
                    'roster_employee_group_id' => $group->id,
                    'shift_id' => $shift['shift_id']
                ], [
                    'roster_employee_group_id' => $group->id,
                    'shift_id' => $shift['shift_id']
                ]);

                $this->add_partitions($roster, $group_shift);
            }
            foreach ($request->employee_id as $id) {
                $employee = RosterEmployeesPerGroup::create([
                    'user_id'   => $id,
                    'roster_employee_group_id'  =>  $group->id
                ]);

                foreach ($roster->partitions as $partition) {
                    RosterEmployeeRequiredHours::create([
                        'roster_partition_id' => $partition->id,
                        'roster_employee_id' => $employee->id
                    ]);
                    foreach ($partition['days'] as $day) {

                        list($remarks, $legend) = $this->set_remarks_and_legends($id, $day['date']);
                        $employee->dayEmployeeShifts()->create([
                            'roster_day_id' => $day['id'],
                            'remarks' => $remarks,
                            'roster_legend_id' => $legend
                        ]);
                    }
                }
            }

            $this->logCustomMessage(
                'set_details_roster',
                null,
                Auth::user()->name . ' Set details roster',
                null,
                RosterLogType::SET_DETAILS,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'message' => 'Successfully assigned Shift/s'
            ], 200);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem assigning shifts per group.'],
                'msg'       =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function fetch_roster_day_per_group(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'roster_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.rosters,id'
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $days = RosterDay::whereHas('rosterPartition', function ($query) use ($request) {
                $query->whereHas('roster', function ($query) use ($request) {
                    $query->whereId($request->roster_id);
                });
            })->get();

            DB::commit();
            return response()->json([
                'days' => $days,
            ], 200);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in fetching roster days.'],
                'msg'       =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function assign_number_of_staff_per_day(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'group_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.roster_employee_groups,id',
            'selected_dates' => 'required|array',
            'selected_dates.*.id' => 'required|numeric|exists:' . env('ADG_DB_CONNECTION') . '.roster_days,id',
            'shift_id' => 'required|array|exists:' . env('ADG_DB_CONNECTION') . '.roster_employee_group_shifts,id'
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            foreach ($request->shift_id as $shift) {
                foreach ($request->selected_dates as $day) {
                    RosterStaffPerDay::updateOrCreate([
                        'roster_employee_group_shift_id' => $shift,
                        'roster_day_id' => $day['id'],
                    ]);
                }
            }

            DB::commit();
            return response()->json([
                "message" => ['Succesfully assigned number of staff per shift and group'],
            ], 200);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in assigning number of staff'],
                'msg'       =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function assign_employee_shifts(Request $request)
    {

        $valid = Validator::make($request->all(), [
            'roster_day_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.roster_days,id',
            'employee_group_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.roster_employee_groups,id',
            'employee_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.roster_employees_per_groups,id',
            'shift_id.shift.id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.shifts,id',
            'legend_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.roster_legends,id',
            'remarks' => 'required',
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $roster = Roster::whereHas('employeeGroups', function ($query) use ($request) {
                $query->whereId($request->employee_group_id);
            })->first();
            $rosterDay = RosterDay::findOrFail($request->roster_day_id);
            $user_id = RosterEmployeesPerGroup::whereId($request->employee_id)->first();
            list($leave, $cto, $holiday) = $this->get_cto_leave_holiday($user_id->user_id, $rosterDay->date);
            $legend = RosterLegend::where('key', 'leave')->value('id');
            if (empty($leave) && empty($cto)) {
                RosterDayEmployeeShift::updateOrCreate([
                    'roster_employees_per_group_id' => $request->employee_id,
                    'roster_day_id' => $request->roster_day_id
                ], [
                    'shift_id' => $request->shift_id['shift_id'],
                    'roster_legend_id' => empty($holiday) ? $request->legend_id : (!empty($legend) ? $legend : null),
                    'remarks' => $request->remarks
                ]);
            }

            $user_id = RosterEmployeesPerGroup::whereId($request->employee_id)->pluck('user_id');

            $user = User::whereId($user_id)->first();
            $attendance_service = new AttendanceService($user);
            $attendance_service->reEvaluateAttendance($rosterDay->date);

            $this->update_required_hours_per_partition($user_id, $roster->id);
            $this->update_min_staff_per_days($request->employee_group_id);

            $this->logCustomMessage(
                'assign_employee_shift',
                null,
                Auth::user()->name . ' Assign employee shift',
                null,
                RosterLogType::ASSIGN,
                new Activity()
            );

            DB::commit();
            return response()->json([
                "message" => ['Succesfully assigned employee shift on roster'],
            ], 200);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in assigning employee shift'],
                'msg'       =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function update_head_employee_shift(Request $request)
    {
        foreach ($request->selected_head_employee_shifts as $head_employee_shift) {
            $head_employee_shift = RosterHeadEmployeeShift::where('id', $head_employee_shift)->first();

            $head_employee_shift->update([
                'shift_id' => $request->shift_id,
                'roster_legend_id' => $request->legend_id,
                'remarks' => $request->remarks
            ]);

            $head_nurse_id = $head_employee_shift->roster->head_nurse_id;

            $user = User::whereId($head_nurse_id)->first();
            $attendance_service = new AttendanceService($user);
            $date = $head_employee_shift->roster_day_id;

            $roster_day = RosterDay::findOrFail($date);
            $attendance_service->reEvaluateAttendance($roster_day->date);

            $this->update_head_required_hours_per_partition($head_employee_shift->roster_id);

            $updated = RosterHeadEmployeeShift::where('id', $head_employee_shift->id)->first();
            $updated->old = collect($head_employee_shift);
            $updated->attributes = collect($updated);

            $this->logCustomMessage(
                'update_head_employee_shift',
                $updated,
                Auth::user()->name . ' updated the shift: ' . $updated->rosterDay->date .  ' of ' . $user->name,
                $updated,
                RosterLogType::UPDATE,
                new Activity()
            );
        }

        return response()->json([
            'message' => 'Updated the head employee shift'
        ], 200);
    }

    public function update_employee_shift(Request $request)
    {
        foreach ($request->selected_employee_shifts as $employee_shift) {
            $employee_shift = RosterDayEmployeeShift::where('id', $employee_shift)->first();
            $previous_shift_id = $employee_shift->shift_id;
            $existing_group_shifts = $employee_shift->rosterEmployeePerGroup->employeeGroup->employeeGroupShifts->pluck('shift_id')->toArray();

            if (!in_array($request->shift_id, $existing_group_shifts)) {
                $group_shift = RosterEmployeeGroupShift::create([
                    'roster_employee_group_id' => $employee_shift->rosterEmployeePerGroup->employeeGroup->id,
                    'shift_id' => $request->shift_id
                ]);

                $roster = $employee_shift->rosterEmployeePerGroup->employeeGroup->roster;

                $this->add_partitions($roster, $group_shift);
            }

            $employee_shift->update([
                'shift_id' => $request->shift_id,
                'roster_legend_id' => $request->legend_id,
                'remarks' => $request->remarks
            ]);

            $shift = RosterDayEmployeeShift::where([
                ['shift_id', $previous_shift_id]
            ])->whereHas('rosterEmployeePerGroup', function ($query) use ($employee_shift) {
                $query->where('roster_employee_group_id', $employee_shift->rosterEmployeePerGroup->roster_employee_group_id);
            })->count();
            if ($shift == 0) {
                $roster_employee_group_shift = RosterEmployeeGroupShift::where([
                    ['roster_employee_group_id', $employee_shift->rosterEmployeePerGroup->employeeGroup->id],
                    ['shift_id', $previous_shift_id]
                ])->first();

                if ($roster_employee_group_shift) {
                    RosterStaffPerDay::where('roster_employee_group_shift_id', $roster_employee_group_shift->id)->delete();
                    $roster_employee_group_shift->delete();
                }
            }

            $user_id = $employee_shift->rosterEmployeePerGroup->user_id;
            $date = $employee_shift->rosterDay()->first()->date;
            $roster_id = $employee_shift->rosterEmployeePerGroup->employeeGroup->roster_id;

            $user = User::whereId($user_id)->first();
            $attendance_service = new AttendanceService($user);
            $attendance_service->reEvaluateAttendance($date);

            $this->update_required_hours_per_partition([$user_id], $roster_id);
            $this->update_min_staff_per_days($employee_shift->rosterEmployeePerGroup->employeeGroup->id);

            $updated = RosterDayEmployeeShift::where('id', $employee_shift->id)->first();
            $updated->old = collect($employee_shift);
            $updated->attributes = collect($updated);

            $this->logCustomMessage(
                'update_employee_shift',
                $updated,
                Auth::user()->name . ' updated the shift: ' . $updated->rosterDay->date .  ' of ' . $user->name,
                $updated,
                RosterLogType::UPDATE,
                new Activity()
            );
        }

        return response()->json([
            'message' => 'Updated the employee shift'
        ], 200);
    }

    public function update_head_employee_roster_legend(Request $request)
    {
        foreach ($request->selected_head_employee_shifts as $head_employee_shift) {
            $head_employee_shift = RosterHeadEmployeeShift::where('id', $head_employee_shift)->first();
            $head_employee_shift->update([
                'roster_legend_id' => $request->legend_id
            ]);

            $updated = RosterHeadEmployeeShift::where('id', $head_employee_shift->id)->first();
            $updated->old = collect($head_employee_shift);
            $updated->attributes = collect($updated);

            $this->logCustomMessage(
                'update_head_employee_roster_legend',
                $updated,
                Auth::user()->name . ' updated the roster legend of: ' . $updated->roster->headNurse->name,
                $updated,
                RosterLogType::UPDATE,
                new Activity()
            );
        }

        return response()->json([
            'message' => 'Updated the head employee roster legend'
        ], 200);
    }

    public function update_employee_roster_legend(Request $request)
    {
        foreach ($request->selected_employee_shifts as $employee_shift) {
            $employee_shift = RosterDayEmployeeShift::where('id', $employee_shift)->first();
            $employee_shift->update([
                'roster_legend_id' => $request->legend_id
            ]);

            $user_id = $employee_shift->rosterEmployeePerGroup->user_id;

            $updated = RosterDayEmployeeShift::where('id', $employee_shift->id)->first();
            $updated->old = collect($employee_shift);
            $updated->attributes = collect($updated);

            $this->logCustomMessage(
                'update_employee_roster_legend',
                $updated,
                Auth::user()->name . ' updated the roster legend of: ' . $employee_shift->rosterEmployeePerGroup->user->name,
                $updated,
                RosterLogType::UPDATE,
                new Activity()
            );
        }

        return response()->json([
            'message' => 'Updated the employee roster legend'
        ], 200);
    }

    public function delete_head_employee_shift(Request $request)
    {
        foreach ($request->selected_head_employee_shifts as $head_employee_shift) {
            $head_employee_shift = RosterHeadEmployeeShift::where('id', $head_employee_shift)->first();

            $head_employee_shift->update([
                'shift_id' => null,
                'roster_legend_id' => null,
                'remarks' => ''
            ]);

            $head_nurse_id = $head_employee_shift->roster->head_nurse_id;

            $user = User::whereId($head_nurse_id)->first();
            $attendance_service = new AttendanceService($user);
            $date = $head_employee_shift->roster_day_id;

            $roster_day = RosterDay::findOrFail($date);
            $attendance_service->reEvaluateAttendance($roster_day->date);

            $this->update_head_required_hours_per_partition($head_employee_shift->roster_id);

            $updated = RosterHeadEmployeeShift::where('id', $head_employee_shift->id)->first();
            $updated->old = collect($head_employee_shift);
            $updated->attributes = collect($updated);

            $this->logCustomMessage(
                'remove_head_employee_shift',
                $updated,
                Auth::user()->name . ' updated the shift: ' . $head_employee_shift->rosterDay->date .  ' of ' . $user->name,
                $updated,
                RosterLogType::REMOVE_SHIFT,
                new Activity()
            );
        }

        return response()->json([
            'message' => 'Deleted the head employee shift'
        ], 200);
    }

    public function delete_employee_shift(Request $request)
    {
        foreach ($request->selected_employee_shifts as $employee_shift) {
            $employee_shift = RosterDayEmployeeShift::where('id', $employee_shift)->first();
            $shift_id = $employee_shift->shift_id;

            $roster_employee_per_group = RosterEmployeesPerGroup::where('id', $employee_shift->roster_employees_per_group_id)->first();

            $employee_shift->update([
                'shift_id' => null,
                'roster_legend_id' => null,
                'remarks' => ''
            ]);

            $shift = RosterDayEmployeeShift::where([
                ['shift_id', $shift_id]
            ])->whereHas('rosterEmployeePerGroup', function ($query) use ($employee_shift) {
                $query->where('roster_employee_group_id', $employee_shift->rosterEmployeePerGroup->roster_employee_group_id);
            })->count();

            if ($shift == 0) {
                $roster_employee_group_shift = RosterEmployeeGroupShift::where([
                    ['roster_employee_group_id', $roster_employee_per_group->roster_employee_group_id],
                    ['shift_id', $shift_id]
                ])->first();

                if ($roster_employee_group_shift) {
                    RosterStaffPerDay::where('roster_employee_group_shift_id', $roster_employee_group_shift->id)->delete();
                    $roster_employee_group_shift->delete();
                }
            }

            $user_id = $employee_shift->rosterEmployeePerGroup->user_id;
            $date = $employee_shift->rosterDay()->first()->date;
            $roster_id = $employee_shift->rosterEmployeePerGroup->employeeGroup->roster_id;

            $user = User::whereId($user_id)->first();
            $attendance_service = new AttendanceService($user);
            $attendance_service->reEvaluateAttendance($date);

            $this->update_required_hours_per_partition([$user_id], $roster_id);
            $this->update_min_staff_per_days($employee_shift->rosterEmployeePerGroup->employeeGroup->id);

            $updated = RosterDayEmployeeShift::where('id', $employee_shift->id)->first();
            $updated->old = collect($employee_shift);
            $updated->attributes = collect($updated);

            $this->logCustomMessage(
                'remove_employee_shift',
                $updated,
                Auth::user()->name . ' removed the shift: ' . $employee_shift->rosterDay->date .  ' of ' . $user->name,
                $updated,
                RosterLogType::REMOVE_SHIFT,
                new Activity()
            );
        }

        return response()->json([
            'message' => 'Delete employee shift',
        ], 200);
    }

    public function fetch_group_employees(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'roster_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.rosters,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {
            $employees = RosterEmployeeGroup::where('roster_id', $request->roster_id)
                ->with('employeeGroupShifts.shift', 'rosterGroup', 'employees.user')->get();
            DB::commit();
            return response()->json([
                "data" => $employees
            ], 200);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem fetching group employees'],
                'msg'       =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function fetch_roster_legend()
    {
        return response()->json([
            'legends' => RosterLegend::orderBy('id', 'desc')->get()
        ]);
    }

    public function fetch_employees(Request $request)
    {
        $currentUser = User::where('id', Auth::id())->first();
        $actualDesignation = $currentUser->actualDesignation;

        return User::when($actualDesignation, function ($query) use ($actualDesignation) {
            $query->whereHas('actualDesignation', function ($query) use ($actualDesignation) {
                $fields = ['office_id', 'department_id', 'division_id', 'section_id', 'unit_id'];
                foreach ($fields as $field) {
                    $query->when(isset ($actualDesignation->$field), function ($query) use ($field, $actualDesignation) {
                        $query->where($field, $actualDesignation->$field);
                    });
                }
            });
        })
        ->where('name', 'LIKE', '%' . $request->employeeName . '%')
        ->has('actualDesignation')
        ->with([
            'actualDesignation.department',
            'actualDesignation.division',
        ])
        ->without([
            'currentRole',
            'roles',
            'permissions',
            'storage',
            'employeeMetaInfo',
            'supervisor',
            'user_supervisor',
            'exitInterview',
            'userProfilePicture',
            'profileBasicInfo'
        ])
        ->get();

    }

    public function fetch_roster_schedule(Request $request)
    {

        $valid = Validator::make($request->all(), [
            'roster_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.rosters,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $data = Roster::whereId($request->roster_id)
                ->with([
                    'partitions.days.staffPerDays',
                    'partitions.days',
                    'employeeGroups.rosterGroup',
                    'employeeGroups.employees.dayEmployeeShifts.rosterDay',
                    'employeeGroups.employees.dayEmployeeShifts.rosterLegend',
                    'employeeGroups.employees.dayEmployeeShifts.shift',
                    'employeeGroups.employeeGroupShifts.staffPerShift',
                    'employeeGroups.employeeGroupShifts.shift',
                    'employeeGroups.employees.requiredHours',
                    'employeeGroups.employees.user',
                    'employeeGroups.employees.oldTransferRemarks',
                    'headRequiredHours',
                    'department',
                    'division',
                    'section',
                    'unit',
                    'headNurse',
                    'headEmployeeShifts.shift',
                    'headEmployeeShifts.rosterDay',
                    'headEmployeeShifts.legend'
                ])
                ->first();

            DB::commit();
            return response()->json([
                'data' => $data
            ], 200);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a fetching roster schedule'],
                'msg'       =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }
    public function set_roster_status(Request $request)
    {

        $valid = Validator::make($request->all(), [
            'roster_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.rosters,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $data = Roster::whereId($request->roster_id)->with('employeeGroups.employees')->first();

            $activeRoster = Roster::where([['department_id', $data->department_id], ['division_id', $data->division_id], ['office_id', $data->office_id], ['section_id', $data->section_id], ['unit_id', $data->unit_id], ['status', 1]])
            ->where('id', '!=', $data->id)->exists();

            if ($activeRoster) {
                return response()->json(['message' => 'This department & division has existing active roster'], 400);
            }

            $data->status = ($data->status == 1) ? 2 : 1;
            $data->save();

            $this->logCustomMessage(
                'set_status_roster',
                $data,
                Auth::user()->name . ' Set status roster',
                $data,
                RosterLogType::SET_STATUS,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'data' => $data
            ], 200);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is an error changing status of roster'],
                'message'       =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function assign_bulk_employee_shift(Request $request)
    {
        $valid = Validator::make($request->all(), [
            '*.employee_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.roster_employees_per_groups,id',
            '*.dayShift.*.date.*' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.roster_days,id',
            '*.dayShift.*.shift.shift_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.shifts,id',
            '*.dayShift.*.legend.id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.roster_legends,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            foreach ($request->all() as $req) {
                $roster = Roster::whereHas('employeeGroups.employees', function ($query) use ($req) {
                    $query->whereId($req['employee_id']);
                })->first();
                foreach ($req['dayShift'] as $day) {
                    foreach ($day['date'] as $day_id) {
                        $rosterDay = RosterDay::findOrFail($day_id);
                        $user_id = RosterEmployeesPerGroup::whereId($req['employee_id'])->first();
                        list($leave, $cto, $holiday) = $this->get_cto_leave_holiday($user_id->user_id, $rosterDay->date);
                        $legend = RosterLegend::where('key', 'leave')->value('id');
                        if (empty($leave) && empty($cto)) {
                            RosterDayEmployeeShift::updateOrCreate([
                                'roster_employees_per_group_id' => $req['employee_id'],
                                'roster_day_id' => $day_id
                            ], [
                                'shift_id' => $day['shift']['shift_id'],
                                'roster_legend_id' => empty($holiday) ? $day['legend']['id'] : (!empty($legend) ? $legend : null),
                                'remarks' => $day['shift']['shift']['code']
                            ]);
                        }
                    }
                }

                $this->update_required_hours_per_partition([$req['employee']['user_id']], $roster->id);
                $this->update_min_staff_per_days($req['group_id']);
            }

            $user_id = RosterEmployeesPerGroup::whereId($request[0]['employee_id'])->pluck('user_id');
            $user = User::whereId($user_id)->first();
            $attendance_service = new AttendanceService($user);

            foreach ($request[0]['dayShift'][0]['dates'] as $dates) {
                $attendance_service->reEvaluateAttendance($dates['date']);
            }

            $this->logCustomMessage(
                'assign_bulk_employee_shift',
                null,
                Auth::user()->name . ' Assign bulk employee shift',
                null,
                RosterLogType::ASSIGN,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'data' => 'Successfully assign bulk employee shift'
            ], 200);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem assign bulk employee shift'],
                'message'       =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function get_cto_leave_holiday($id, $date)
    {
        $leave = Leave::where([['user_id', $id], ['status', 1]])
            ->whereDate('start_date', '<=', $date)
            ->whereDate('end_date', '>=', $date)
            ->with('leaveType')
            ->latest('updated_at')
            ->first();
        $cto = CTOApplication::where([['user_id', $id], ['status', 1]])
            ->whereDate('start_date', '<=', $date)
            ->whereDate('end_date', '>=', $date)
            ->latest('updated_at')
            ->first();
        $holiday = Holiday::whereDate('date', $date)
            ->where('status', 1)
            ->first();

        return [$leave, $cto, $holiday];
    }

    public function fetch_partition(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'roster_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.rosters,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $data = RosterPartition::where('roster_id', $request->roster_id)->get();

            DB::commit();
            return response()->json([
                "data" => $data
            ], 200);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in fetching partitions'],
                'msg'       =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function set_employee_required_hours(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'partition_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.roster_partitions,id',
            'employee_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.roster_employees_per_groups,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            foreach ($request->employee_id as $emp_id) {
                foreach ($request->partition_id as $partition_id) {
                    RosterEmployeeRequiredHours::updateOrCreate([
                        'roster_partition_id' => $partition_id,
                        'roster_employee_id' => $emp_id
                    ]);
                }
            }

            DB::commit();
            return response()->json([
                "message" => 'Successfully created required work hours'
            ], 200);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in assigning required hours'],
                'msg'       =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function set_head_required_hours(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'partition_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.roster_partitions,id',
            'roster_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.rosters,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            foreach ($request->partition_id as $id) {
                RosterHeadRequiredHours::updateOrCreate([
                    'roster_partition_id' => $id,
                    'roster_id' => $request->roster_id
                ]);
            }


            DB::commit();
            return response()->json([
                "message" => 'Successfully created head employee required hours'
            ], 200);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in assigning required hours'],
                'msg'       =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function update_roster_schedule(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'roster_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.rosters,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {
            $roster = Roster::findOrFail($request->roster_id);
            $roster->title = $request->title;
            $roster->office_id = $request->office_id;
            $roster->department_id = $request->department_id;
            $roster->division_id = $request->division_id;
            $roster->section_id = $request->section_id;
            $roster->unit_id = $request->unit_id;
            $roster->save();

            DB::commit();
            return response()->json([
                "message" => 'Successfully updated Roster'
            ], 200);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in updating roster name'],
                'msg'       =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function remove_employee_shifts(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'roster_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.rosters,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $emp_shifts = RosterDayEmployeeShift::whereHas('rosterEmployeePerGroup', function ($query) use ($request) {
                $query->whereIn('user_id', $request->user_id)->whereHas('employeeGroup.roster', function ($query) use ($request) {
                    $query->where('id', $request->roster_id);
                });
            })->whereHas('rosterDay', function ($query) use ($request) {
                $query->whereIn('date', $request->dates);
            })->with('rosterEmployeePerGroup', 'rosterDay', 'shift')->get();

            foreach ($emp_shifts as $shift) {
                list($remarks, $legend) = $this->set_remarks_and_legends($shift->rosterEmployeePerGroup->user_id, $shift->rosterDay->date);
                $shift->shift_id = null;
                $shift->roster_legend_id = $legend;
                $shift->remarks = $remarks;

                $this->logCustomMessage(
                    'remove_employee_shift',
                    $shift,
                    Auth::user()->name . ' Remove employee shift',
                    $shift,
                    RosterLogType::REMOVE_SHIFT,
                    new Activity()
                );

                $shift->save();
            }

            foreach ($request->user_id as $user_id) {
                $user = User::whereId($user_id)->first();
                $attendance_service = new AttendanceService($user);
                foreach ($emp_shifts as $shift) {
                    $attendance_service->reEvaluateAttendance($shift->rosterDay->date);
                }
            }

            $group = RosterEmployeeGroup::whereHas('employees', function ($query) use ($request) {
                $query->whereIn('user_id', $request->user_id);
            })->whereHas('roster', function ($query) use ($request) {
                $query->whereId($request->roster_id);
            })->first();

            $this->update_min_staff_per_days($group->id);

            $this->update_required_hours_per_partition($request->user_id, $request->roster_id);

            DB::commit();
            return response()->json([
                "message" => 'Successfully remove employee shift'
            ], 200);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors' => ['There is a problem in removing employee shift'],
                'msg' => $e->getMessage(),
                'line' => $e->getLine()
            ], 500);
        }
    }

    public function remove_head_employee_shifts(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'roster_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.rosters,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $head_emp_shifts = RosterHeadEmployeeShift::where('roster_id', $request->roster_id)
                ->whereHas('rosterDay', function ($query) use ($request) {
                    $query->whereIn('date', $request->dates);
                })->with('roster', 'rosterDay', 'shift')->get();

            foreach ($head_emp_shifts as $shift) {
                list($remarks, $legend) = $this->set_remarks_and_legends($shift->roster->head_nurse_id, $shift->rosterDay->date);
                $shift->shift_id = null;
                $shift->roster_legend_id = $legend;
                $shift->remarks = $remarks;

                $this->logCustomMessage(
                    'remove_head_employee_shift',
                    $shift,
                    Auth::user()->name . ' Remove head employee shift',
                    $shift,
                    RosterLogType::REMOVE_SHIFT,
                    new Activity()
                );

                $shift->save();
            }

            $roster_id = Roster::whereId($request->roster_id)->pluck('head_nurse_id')->first();
            $user = User::whereId($roster_id)->first();
            $attendance_service = new AttendanceService($user);
            foreach ($head_emp_shifts as $shift) {
                $attendance_service->reEvaluateAttendance($shift->rosterDay->date);
            }

            $this->update_head_required_hours_per_partition($request->roster_id);

            DB::commit();
            return response()->json([
                "message" => 'Successfully remove head employee shift'
            ], 200);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors' => ['There is a problem in removing employee shift'],
                'msg' => $e->getMessage(),
                'line' => $e->getLine()
            ], 500);
        }
    }

    private function update_required_hours_per_partition($user_ids, $roster_id)
    {
        $roster = Roster::whereId($roster_id)->with('partitions')->first();
        foreach ($user_ids as $user_id) {
            foreach ($roster->partitions as $partition) {
                $shifts = RosterDayEmployeeShift::whereHas('rosterEmployeePerGroup', function ($query) use ($user_id, $roster_id) {
                    $query->where('user_id', $user_id)
                        ->whereHas('employeeGroup.roster', function ($query) use ($roster_id) {
                            $query->where('id', $roster_id);
                        });
                })->whereHas('rosterDay.rosterPartition', function ($query) use ($partition) {
                    $query->whereId($partition->id);
                })->with('shift')->get();

                $total_hours = 0;
                foreach ($shifts->pluck('shift') as $shift) {
                    if (!$shift) continue;
                    if (!$shift->break_in || !$shift->break_out) {
                        $break_hours = 0;
                    } else {
                        $break_in = strtotime($shift->break_in);
                        $break_out = strtotime($shift->break_out);
                        $break_hours = ($break_out - $break_in) / 3600;
                    };
                    $total_hours += ($shift->work_hours  - $break_hours);
                }

                $req_hours = RosterEmployeeRequiredHours::where([
                    'roster_partition_id' => $partition->id,
                    'roster_employee_id' => $shifts[0]->roster_employees_per_group_id,
                ])->first();
                $req_hours->required_hours = $total_hours;
                $req_hours->save();
            }
        }
    }

    private function update_head_required_hours_per_partition($roster_id)
    {
        $roster = Roster::whereId($roster_id)->with('partitions')->first();
        foreach ($roster->partitions as $partition) {
            $shifts = RosterHeadEmployeeShift::where('roster_id', $roster_id)
                ->whereHas('rosterDay.rosterPartition', function ($query) use ($partition) {
                    $query->whereId($partition->id);
                })->with('shift')->get();

            $total_hours = 0;
            foreach ($shifts->pluck('shift') as $shift) {
                if (!$shift) continue;
                if (!$shift->break_in || !$shift->break_out) {
                    $break_hours = 0;
                } else {
                    $break_in = strtotime($shift->break_in);
                    $break_out = strtotime($shift->break_out);
                    $break_hours = ($break_out - $break_in) / 3600;
                };
                $total_hours += ($shift->work_hours  - $break_hours);
            }
            $total_hours;
            $req_hours = RosterHeadRequiredHours::where([
                'roster_partition_id' => $partition->id,
                'roster_id' => $roster->id,
            ])->first();
            $req_hours->required_hours = $total_hours;
            $req_hours->save();
        }
    }

    public function update_min_staff_per_days($roster_employee_group_id)
    {
        $staff_per_days = RosterStaffPerDay::whereHas('shift.rosterEmployeeGroup', function ($query) use ($roster_employee_group_id) {
            $query->whereId($roster_employee_group_id);
        })->with('rosterDay', 'shift.shift')->get();

        foreach ($staff_per_days as $staff_per_day) {
            $count = 0;
            $count = RosterDayEmployeeShift::where('roster_day_id', $staff_per_day->rosterDay->id)
                ->where('shift_id', $staff_per_day->shift->shift->id)
                ->whereHas('rosterEmployeePerGroup.employeeGroup', function ($query) use ($staff_per_day) {
                    $query->whereId($staff_per_day->shift->roster_employee_group_id);
                })
                ->count();

            $staff_per_day->number_of_staff = $count;
            $staff_per_day->save();
        }
    }
    private function set_remarks_and_legends($user_id, $date)
    {

        list($leave, $cto, $holiday) = $this->get_cto_leave_holiday($user_id, $date);
        $legend = RosterLegend::where('key', 'leave')->value('id');
        if (!empty($leave)) {
            $remarks = $leave->leaveType->remark;
        } elseif (!empty($cto)) {
            $remarks = $cto->remarks;
            $legend = null;
        } elseif (!empty($holiday)) {
            $remarks = '‎ '; //insert this remarks for frontend to see blank with color
        } else {
            $remarks = ' ';
            $legend = null;
        }

        return [$remarks, $legend];
    }

    public function remove_employees(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'roster_employee_per_group_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.roster_employees_per_groups,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {

            $employee = RosterEmployeesPerGroup::with('employeeGroup')->find($request->roster_employee_per_group_id);

            $service = new RosterService();
            $service->removeEmployees($employee);

            $this->logCustomMessage(
                'remove_employee',
                $employee,
                Auth::user()->name . ' Remove employee',
                $employee,
                RosterLogType::REMOVE_EMPLOYEE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                "message" => 'Successfully remove employee'
            ], 200);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors' => ['There is a problem in removing employee'],
                'msg' => $e->getMessage(),
                'line' => $e->getLine()
            ], 500);
        }
    }

    private function add_partitions($roster, $group_shift)
    {
        foreach ($roster->partitions as $partition) {
            foreach ($partition['days'] as $day) {
                $group_shift->staffPerShift()->updateOrCreate([
                    'roster_day_id' => $day['id'],
                ], [
                    'number_of_staff' => 0
                ]);
            }
        }
    }

    public function get_users_shift(Request $request, RosterService $rosterService)
    {
        return $rosterService->getUsersShift($request->roster_id, $request->start_date, $request->end_date, $request->user_ids);
    }

    public function remove_group(Request $request) {

        $valid = Validator::make($request->all(), [
            'id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.roster_employee_groups,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $group = RosterEmployeeGroup::whereId($request->id)->with('employees')->first();
            if($group){
                $employees = $group->employees;

                $service = new RosterService();

                foreach($employees as $employee){
                    $service->removeEmployees($employee);
                }

                $group->delete();
            }
            DB::commit();
            return response()->json([
                "message" => 'Successfully remove group'
            ], 200);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors' => ['There is a problem in removing group'],
                'msg' => $e->getMessage(),
                'line' => $e->getLine()
            ], 500);
        }
    }
}
