<?php

namespace Suiterus\Adg\Controllers\Timekeeping\Roster\Employee;

use App\Enums\LeaveStatus;
use App\Enums\RescheduleStatus;
use App\Http\Controllers\Controller;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Suiterus\Adg\Models\Approvals\CTO\CTOApplication;
use Suiterus\Adg\Models\LeaveManagement\LeaveApprover;
use Suiterus\Adg\Models\LeaveManagement\Requests\Leave;
use Suiterus\Adg\Models\SM\Holiday;
use Suiterus\Adg\Models\Timekeeping\Roster\Roster;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterDay;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterDayEmployeeShift;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterEmployeeGroupShift;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterHeadEmployeeShift;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterRescheduleRequest;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterRescheduleRequestDate;
use Validator;

class RosterEmployeeController extends Controller
{
    public function search_roster(Request $req) {
        $paginate = $req->page_count ? intval($req->page_count) : ENV('DEFAULT_PAGECOUNT');


        $data = Roster::where('title','LIKE', '%' . $req->keyword . '%')
        ->whereHas('employeeGroups.employees', function ($query) {
            $query->where('user_id', Auth::id());
        })->orWhere('head_nurse_id', Auth::id())
        ->when($req->date != null, function ($query) use ($req){
            $startDate = Carbon::parse($req->date[0])->startOfDay();
            $endDate = Carbon::parse($req->date[1])->endOfDay();
            $query->whereDate('start_date', '<=', $startDate)
                ->whereDate('end_date', '>=', $endDate);
        })
        ->orderBy('created_at','desc')
        ->with('partitions.days.employeeShifts','partitions.days.staffPerDays','division','employeeGroups','department')
        ->paginate($paginate);

        return response()->json([
            'data'  =>  $data,
        ]);
    }

    public function fetch_available_dates(Request $request)
    { 
        DB::beginTransaction();
        try {
            $user = User::findOrFail(Auth::id());
            $shift = RosterEmployeeGroupShift::whereHas('rosterEmployeeGroup.roster', function ($query) use($user, $request) {
                $query->whereId($request->roster_id)->where([['department_id',$user->actualDesignation->department_id],
                    ['division_id',$user->actualDesignation->division_id]
                ]);
            })->whereHas('rosterEmployeeGroup.employees', function ($query) {
                $query->where('user_id', Auth::id());
            })->with('shift')->get();

            $current_schedule = RosterDayEmployeeShift::where('roster_day_id', $request->roster_day_id)->whereHas('rosterEmployeePerGroup', function ($query) {
                $query->where('user_id', Auth::id());
            })->whereHas('RosterDay.rosterPartition.roster', function($query) use ($user, $request){
                $query->whereId($request->roster_id)->where([['department_id',$user->actualDesignation->department_id],
                    ['division_id',$user->actualDesignation->division_id]
                ]);
            })->with('rosterDay','shift','rosterLegend')->first();

            DB::commit();
            return response()->json([
                'current_schedule' => $current_schedule,
                'shifts' => $shift
            ], 200);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem fetching available dates.'],
                'msg'       =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }
    public function fetch_current_schedule(Request $request) {

        $paginate = $request->page_count ? intval($request->page_count) : ENV('DEFAULT_PAGECOUNT');
        $roster = Roster::whereId($request->roster_id)->where('head_nurse_id', Auth::id())->first();

        if($roster){
            $current_schedule = RosterHeadEmployeeShift::whereHas('roster', function($query) use ($roster){
                $query->whereId($roster->id);
            })->whereHas('rosterDay', function ($query) use ($request) {
                $query->whereMonth('date', $request->month)
                ->whereYear('date', $request->year);
            })->with('rosterDay','shift','legend')->get();
        }else{
            $current_schedule = RosterDayEmployeeShift::whereHas('rosterEmployeePerGroup', function ($query) {
                    $query->where('user_id', Auth::id());
                })->whereHas('RosterDay.rosterPartition.roster', function($query) use ($request){
                    $query->whereId($request->roster_id);
                })->whereHas('RosterDay', function ($query) use ($request) {
                    $query->whereMonth('date', $request->month)
                    ->whereYear('date', $request->year);
                })
                ->with('rosterDay','shift','rosterLegend')->get();
        }
        return response()->json([
            'data' => $current_schedule
        ], 200);
    }

    public function fetchHolidayByMonthAndYear(Request $request)
    {
        return response()->json([
            'data' => Holiday::whereMonth('date', $request->month)
                ->whereYear('date', $request->year)->get()
        ]);
    }

    public function fetchLeaveByMonthAndYear(Request $request)
    {
        return response()->json([
            'data' => Leave::whereMonth('start_date', $request->month)
                ->whereYear('start_date', $request->year)
                ->orWhere(function ($query) use ($request) {
                    $query->whereMonth('end_date', $request->month)
                        ->whereYear('end_date', $request->year);
                })
                ->where('status', LeaveStatus::APPROVED)
                ->where('user_id', Auth::id())
                ->with('leaveType')
                ->get()
        ]);
    }

    public function fetchCTOByMonthAndYear(Request $request)
    {
        $statusApproved = 1;
        return response()->json([
            'data' => CTOApplication::whereMonth('start_date', $request->month)
                ->whereYear('start_date', $request->year)
                ->orWhere(function ($query) use ($request) {
                    $query->whereMonth('end_date', $request->month)
                        ->whereYear('end_date', $request->year);
                })
                ->where('status', $statusApproved)
                ->where('user_id', Auth::id())
                ->get()
        ]);
    }


    public function fetch_reschedule_request(Request $request) {
        $paginate = $request->page_count ? intval($request->page_count) : ENV('DEFAULT_PAGECOUNT');

        $user = User::findOrFail(Auth::id());
        $data = RosterRescheduleRequest::where('user_id',Auth::id())
            ->whereHas('rosterDay.rosterPartition.roster', function ($query) use ($user) {
                $query->where([['department_id',$user->actualDesignation->department_id],
                    ['division_id',$user->actualDesignation->division_id]
                ]);
            })->with('requestee','rosterDay','newShift','oldShift')->paginate($paginate);
        return response()->json([
            'data'  => $data
        ]);
    }

    public function create_reschedule_request(Request $request) {

        $valid = Validator::make($request->all(), [
            'requestee_id' => 'required|exists:' . env('DB_CONNECTION') . '.users,id',
            'roster_day_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.roster_days,id'
        ]);

        if($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            RosterRescheduleRequest::create([
                'user_id'   =>  Auth::id(),
                'requestee_id'  =>  $request->requestee_id,
                'request_date' => $request->roster_day_id,
                'new_shift_id'  =>  $request->new_shift_id,
                'old_shift_id' => $request->old_shift_id,
                'reason'    =>  $request->reason,
                'status'    =>  RescheduleStatus::PENDING
            ]);

            DB::commit();
            return response()->json([
                'message' => 'Successfully created reschedule request'
            ], 200);

        } catch (Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in creating a reschedule request.'],
                'msg'       =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function update_reschedule_request(Request $request) {

        $valid = Validator::make($request->all(), [
            'reschedule_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.roster_reschedule_requests,id',
            'requestee_id' => 'required|exists:' . env('DB_CONNECTION') . '.users,id',
            'roster_day_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.roster_days,id'
        ]);

        if($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $reschedule = RosterRescheduleRequest::firstOrFail($request->reschedule_id);
            $reschedule->update([
                'requestee_id'  =>  $request->requestee_id,
                'requestee_id'  =>  $request->requestee_id,
                'request_date' => $request->roster_day_id,
                'new_shift_id'  =>  $request->new_shift_id,
                'old_shift_id' => $request->old_shift_id,
                'reason'    =>  $request->reason
            ]);

            DB::commit();
            return response()->json([
                'message' => 'Successfully updated reschedule request'
            ], 200);

        } catch (Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in updating the reschedule request.'],
                'msg'       =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function fetch_roster_schedule(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'roster_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.rosters,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $data = Roster::where(function ($query) use ($request) {
                $query->where('id', $request->roster_id)->where('head_nurse_id', Auth::id());
            })->orWhere(function ($query) use ($request) {
                $query->where('id', $request->roster_id)->whereHas('employeeGroups.employees', function ($query) {
                    $query->where('user_id', Auth::id());
                });
            })->with([
                    'partitions.days.staffPerDays',
                    'partitions.days',
                    'employeeGroups.rosterGroup',
                    'employeeGroups.employees.dayEmployeeShifts.rosterDay',
                    'employeeGroups.employees.dayEmployeeShifts.rosterLegend',
                    'employeeGroups.employees.dayEmployeeShifts.shift',
                    'employeeGroups.employeeGroupShifts.staffPerShift',
                    'employeeGroups.employeeGroupShifts.shift',
                    'employeeGroups.employees.requiredHours',
                    'employeeGroups.employees.user',
                    'employeeGroups.employees.oldTransferRemarks',
                    'headRequiredHours',
                    'department',
                    'division',
                    'section',
                    'unit',
                    'headNurse',
                    'headEmployeeShifts.shift',
                    'headEmployeeShifts.rosterDay',
                    'headEmployeeShifts.legend'
                ])->orderBy('id', 'desc')->first();

            DB::commit();
            return response()->json([
                'data' => $data
            ], 200);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem fetching roster schedule'],
                'msg'       =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function fetch_roster_table(){
        return Roster::whereHas('employeeGroups.employees', function ($query) {
            $query->where('user_id', Auth::id());
        })->orWhere('head_nurse_id', Auth::id())
        ->orderBy('created_at','desc')
        ->with('partitions.days.employeeShifts','partitions.days.staffPerDays','division','employeeGroups','department')
        ->get();
    }
}
