<?php

namespace Suiterus\Adg\Controllers\Timekeeping;

use App\Enums\Biometrics\EnabledStatus;
use App\Enums\Log\BiometricsLogType;
use Suiterus\Adg\Models\Timekeeping\rfid;
use Illuminate\Http\Request;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use App\Http\Controllers\Controller;
use App\Models\User;
use Exception;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use App\Enums\Status;
use App\Exceptions\BiometricConnectException;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\Auth;
use InvalidArgumentException;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Timekeeping\BiometricDevice;
use Suiterus\Adg\Services\ZKTecoService;

class RFIDController extends Controller
{
    use HasCustomLogs;
    /**
     * Create new RFID record - Creates user in the local database
     */
    public function create_rfid(Request $req)
    {

        $validate = Validator::make($req->all(), [
            'data'                => 'required|array',

        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            foreach($req->data as $data){
                $user = User::find($data['user_id']);
                $devices = BiometricDevice::where('status', Status::ACTIVE)->get();
                if($devices === null) {
                    throw new BiometricConnectException('The device for the user does not exist.');
                }
                foreach($devices as $device){
                    $zk = new ZKTecoService($device->host, $device->port);
                    if(!$zk->checkDeviceConnection()) continue;
                    $zk->disable();
                    $zk->setUser($user, true, $data['card_number'], null, $data['device_id']);
                    $zk->enable();
                    $zk->disconnect();
                }
                
            }

            $this->logCustomMessage(
                'create_rfid_user',
                null,
                Auth::user()->name . ' Create RFID user',
                null,
                BiometricsLogType::CREATE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'success'   => true,
                'text'      => 'RFID have been successfully created.',
            ]);
        } catch(BiometricConnectException $e) {
            return response()->json([
                'errors'    => [__('responses.zkteco.connection-failed')],
                'message'   => $e->getMessage()
            ], 400);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem with creating RFID.'],
                'message'   => $e->getMessage()
            ]);
        }
    }

    /**
     * Synchronize card from device to system
     */
    public function syncAllCard() {

        DB::beginTransaction();
        try {
            $users = User::whereHas('biometric_rfid_record.card')->without(
                    'roles',
                    'permissions',
                    'storage',
                    'employeeMetaInfo',
                    'supervisor',
                    'exitInterview',
                    'userProfilePicture',
                    'profileBasicInfo')
                ->get();

            $devices = BiometricDevice::where('status', Status::ACTIVE)->get();
            if($devices === null) {
                throw new BiometricConnectException('The device for the user does not exist.');
            }
            foreach($devices as $device){
                $zk = new ZKTecoService($device->host, $device->port);
                if(!$zk->checkDeviceConnection()) continue;
                $zk->disable();
                foreach ($users as $user) {
                    $zk->syncCard($user);
                }
                $zk->enable();
                $zk->disconnect();
            }

            $this->logCustomMessage(
                'sync_all_rfid_card_user',
                null,
                Auth::user()->name . ' Sync all RFID card user',
                null,
                BiometricsLogType::SYNC,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => 'All card has been synced.'
            ]);

        } catch(BiometricConnectException $e) {
            return response()->json([
                'errors'    => [__('responses.zkteco.connection-failed')],
                'message'   => $e->getMessage()
            ], 400);
        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem with syncing the card'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    public function syncCard(Request $request) {

        DB::beginTransaction();
        try {

            $user = User::find($request->id);

            $devices = BiometricDevice::where('status', Status::ACTIVE)->get();
            if($devices === null) {
                throw new BiometricConnectException('The device for the user does not exist.');
            }
            foreach($devices as $device){
                $zk = new ZKTecoService($device->host, $device->port);
                if(!$zk->checkDeviceConnection()) continue;
                $zk->disable();
                $zk->syncCard($user);
                $zk->enable();
                $zk->disconnect();
            }

            $this->logCustomMessage(
                'sync_rfid_card_user',
                null,
                Auth::user()->name . ' Sync RFID Card user',
                null,
                BiometricsLogType::SYNC,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => 'Card has been synced.'
            ]);

        } catch(BiometricConnectException $e) {
            return response()->json([
                'errors'    => [__('responses.zkteco.connection-failed')],
                'message'   => $e->getMessage()
            ], 400);
        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem with syncing the card'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    //SEARCH FUNCTION
    public function search_employee(Request $req)
    {
        $search = $req->input('name');
        $valid = Validator::make($req->all(), [
            'name' => 'required|string', 
            'page_count' => 'nullable'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }

        $paginate = $req->page_count ? intval($req->page_count) : env('DEFAULT_PAGECOUNT');

        try {
            $search_query = user:: where('name', 'LIKE', "%{$search}%")
                ->paginate($paginate);

            if ($search_query->isEmpty()) {
                DB::rollback();

                return response()->json(
                    [
                        'message' => 'Employee does not exist.',
                    ],
                    400
                );
            } else {
                return response()->json([
                    'data' => $search_query,
                ]);
            }
        } catch (Exception $e) {
            DB::rollback();

            return response()->json(
                [
                    'errors' => ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : BLOG-0x03'],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }

    //filter
    public function filter_rfid(Request $req)
    {
        
        $paginate = $req->paginate ? intval($req->paginate) : env('DEFAULT_PAGECOUNT');

        $records = RFID::with(['device_record.user' => function ($query) {

            $query->select('id', 'name')->without(['roles', 'permission', 'storage', 'employeeMetaInfo', 'supervisor', 'exitInterview']);
        }])->when($req->date != null, function ($query) use ($req) {
            $query->whereDate('date_issued', $req->date);
        })->when($req->status != null, function ($query) use ($req) {
            $query->where('status', $req->status);
        })->whereHas("device_record.user", function ($query) use ($req) {
            $query->when($req->name != null, function ($query) use ($req) {
                $query->where('name', "LIKE", '%' . $req->name . '%');
            })->when(isset($req->employees) && count($req->employees) > 0, function($query) use($req){
                $query->whereIn('id', $req->employees);
            });
        })->orderBy('created_at', 'asc')->paginate($paginate);

        return response()->json([
            'data' => $records
        ]);
    }

    //UPDATE FUNCTION
    public function update_rfid(Request $req, rfid $rfid)
    {
        $valid = Validator::make($req->all(), [
            'id' => 'required|exists:adg_db.rfid,id'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        try {
            RFID::where('user_id', $req->user_id)->update([
                'user_id' => $req->user_id,
                'rfid' => $req->rfid,
                'card_number' => $req->card_number,
                'status' => $req->status,
                'date_issued' => $req->date_issued,
                'created_by' => 1,
                'updated_by' => 1,
            ]);

            return response()->json([
                'message' => "Updated successfully!",
            ]);
        } catch (Exception $e) {

            DB::rollback();

            return response()->json([
                'errors'    =>  ['The request could not be process.'],
                'msg'   =>  $e->getMessage()
            ], 500);
        
        }
    }

    //FETCH ALL FUNCTION
    public function fetch_all_rfid(Request $req)
    {
        $paginate = $req->page_count ? intval($req->pageinate) : env('DEFAULT_PAGECOUNT');

        $data = RFID::with(['device_record.user' => function ($q) {
            $q->select('id', 'name')->without('roles', 'permissions', 'storage', 'employeeMetaInfo', 'supervisor', 'exitInterview');
        }])->paginate($paginate);

        return response()->json([
            'data' => $data
        ]);
    }

    // DELETE FUNCTION
    public function delete_rfid(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'id' => 'required|exists:adg_db.rfid,id'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {

            $delete = rfid::findOrFail($req->id);
            $delete->delete();
            DB::commit();

            return response()->json([
                'text'  =>  'RFID has been deleted.',

            ]);
        } catch (Exception $e) {

            DB::rollback();

            return response()->json([
                'errors'    =>  ['The request could not be process.'],
                'msg'   =>  $e->getMessage()
            ], 500);
        }
    }

    public function restore_deleted_rfid(Request $req)
    {
        try {
            try {
                RFID::onlyTrashed()
                    ->findOrFail($req->id)
                    ->restore();
                DB::commit();
                return response()->json([
                    'text' => 'Rfid has been restored.',
                ]);
            } catch (ME $e) {
                DB::rollback();
                return response()->json(
                    [
                        'errors' => ['Rfid not found!.'],
                    ],
                    404
                );
            }
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Something went wrong while processing your request. Error Code : '],
                    'message' => $e->getMessage(),
                ],
                500
            );
        }
    }

    public function init_list_deleted_rfid()
    {
        $paginate = 5;
        $deleted_rfid = RFID::onlyTrashed()
            ->orderBy('deleted_at', 'desc')
            ->paginate($paginate);
        return response()->json([
            'text' => 'fetch successful.',
            'data' => $deleted_rfid,
        ]);
    }

    // APPROVE
    public function approve(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'id'    => 'required|exists:adg_db.rfid,id'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'  => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $rfid = RFID::find($request->id);

            if ($rfid->status == Status::ACTIVE) {
                throw new InvalidArgumentException('The RFID is already approved');
            }

            $rfid->status = Status::ACTIVE;

            $user = $rfid->device_record->user;

            $devices = BiometricDevice::where('status', EnabledStatus::ENABLED)->get();
            if($devices === null) {
                throw new BiometricConnectException('The device for the user does not exist.');
            }
            foreach($devices as $device){
                $zk = new ZKTecoService($device->host, $device->port);
                if(!$zk->checkDeviceConnection()) continue;
                $zk->disable();
                $zk->setCardActive($user);
                $zk->enable();
                $zk->disconnect();
            }
            

            $rfid->save();

            $this->logCustomMessage(
                'approve_rfid_user',
                $rfid,
                Auth::user()->name . ' Approve RFID user',
                $rfid,
                BiometricsLogType::APPROVE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => 'RFID approved.'
            ]);
        } catch(BiometricConnectException $e) {
            return response()->json([
                'errors'    => [__('responses.zkteco.connection-failed')],
                'message'   => $e->getMessage()
            ], 400);
        } catch (InvalidArgumentException $e) {
            DB::connection('adg_db')->rollBack();
            return response()->json([
                'errors'    => [$e->getMessage()],
                'message'   => $e->getMessage()
            ], 500);
        } catch (Exception $e) {
            DB::connection('adg_db')->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in approving the record'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    // DECLINE
    public function decline(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'id'    => 'required|exists:adg_db.rfid,id'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'  => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $rfid = RFID::find($request->id);

            if ($rfid->status == Status::INACTIVE) {
                throw new InvalidArgumentException('The RFID is already declined');
            }

            $user = $rfid->device_record->user;
            
            $devices = BiometricDevice::where('status', EnabledStatus::ENABLED)->get();
            if($devices === null) {
                throw new BiometricConnectException('The device for the user does not exist.');
            }
            foreach($devices as $device) {
                $zk = new ZKTecoService($device->host, $device->port);
                if(!$zk->checkDeviceConnection()) continue;
                $zk->disable();
                $rfid->status = Status::INACTIVE;
                $zk->setCardInactive($user);
                $zk->enable();
                $zk->disconnect();
            }
            

            $rfid->save();

            $this->logCustomMessage(
                'decline_rfid_user',
                $rfid,
                Auth::user()->name . ' Decline RFID user',
                $rfid,
                BiometricsLogType::DECLINE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => 'RFID declined.'
            ]);
        } catch(BiometricConnectException $e) {
            return response()->json([
                'errors'    => [__('responses.zkteco.connection-failed')],
                'message'   => $e->getMessage()
            ], 400);
        } catch (InvalidArgumentException $e) {
            DB::connection('adg_db')->rollBack();
            return response()->json([
                'errors'    => $e->getMessage(),
                'message'   => $e->getMessage()
            ], 500);
        } catch (Exception $e) {
            DB::connection('adg_db')->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in declining the record'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Fetch users/employees without and RFID record
     */
    public function employees() {

        try {

            return response()->json([
                'data'  => User::whereDoesntHave('biometric_rfid_record.card')
                    ->without(['roles', 'storage', 'permissions'])
                    ->whereHas('employeeMetaInfo')
                    ->with(['employeeMetaInfo' => function ($query) {
                        $query->without('branch', 'corporation', 'division', 'department', 'position')
                            ->select('user_id', 'employee_id');
                    }, 'user_supervisor'])    
                    ->get() 
            ]);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in getting the request.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }
}