<?php

namespace Suiterus\Adg\Controllers\Timekeeping;

use App\Enums\Log\ScheduleLogType;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use Suiterus\Adg\Models\Timekeeping\EmployeeSchedule;
use Suiterus\Adg\Models\SM\ScheduleTitle;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Suiterus\Adg\Models\SM\Branch;
use Suiterus\Adg\Models\SM\Corporation;
use Suiterus\Adg\Models\SM\Department;
use Suiterus\Adg\Models\SM\Division;
use App\Models\User;
use App\Traits\Logs\HasCustomLogs;
use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\SM\EmployeeType;
use Suiterus\Hrjp\Models\Position;

class EmployeeScheduleController extends Controller
{
    use HasCustomLogs;
    /** 
     * Initialize list of employees to be displayed on the main page of the Schedules Page in Timekeeping
     * 
     */
    public function init_list_employees(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : ENV('DEFAULT_PAGECOUNT');
        $user = User::where('id', Auth::id())->first();
        $actualDesignation = $user->actualDesignation;

        $data = User::when($user->roles->whereNotIn('name', ['Super Admin', 'Admin', 'Developer']) && $actualDesignation, function ($query) use ($actualDesignation) {
            $query->whereHas('actualDesignation', function ($query) use ($actualDesignation) {
                $fields = ['office_id', 'department_id', 'division_id', 'section_id', 'unit_id'];
                foreach ($fields as $field) {
                    $query->when(isset ($actualDesignation->$field), function ($query) use ($field, $actualDesignation) {
                        $query->where($field, $actualDesignation->$field);
                    });
                }
            });
        })
        ->with(['employeeSchedules' => function($query) {
            $query->with(['schedule_template' => function($query) { 
                $query->with('schedule_template'); }]);
        },
        'temporary_schedule' => function($query) {
            $query->with('schedule_template');
        }, 
        'actualDesignation.department', 
        'actualDesignation.division',
        'actualDesignation.section',
        'actualDesignation.unit'
        ])
        ->whereHas('employeeMetaInfo')
        ->orderBy('created_at', 'asc')->paginate($paginate);

        return response()->json([
            'data' => $data
        ]);
    }

    /**
     * Fetch all schedule templates from ScheduleTitle model where the basis is schedule-based 
     * and the schedule has days for the template.
     * 
     */
    public function fetch_schedule_templates()
    {
        $schedule = ScheduleTitle::where('status', 1)->get();

        $schedules = [];
        foreach($schedule as $sched) {
            if($sched->basis == 1) {
                if($sched->hasSchedule == 1) { array_push($schedules, $sched); }
            } else {
                array_push($schedules, $sched);
            }
        }
        return response()->json([
            'data' => $schedules
        ]); 
    }

    /**
     * Assign a specific schedule template for each employee.
     * Different from bulk schedule, as each employee in assign individual 
     * has a specific schedule template assigned
     * 
     */
    public function assign_individual_schedule(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'schedules' => 'required|array',
            'schedules.*.employee' => 'required|unique:adg_db.employee_schedules,user_id,NULL,id,deleted_at,NULL',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors(),
            ], 400);
        }
        
        DB::beginTransaction();
        try {
            $schedules = [];            
            foreach ($request->schedules as $schedule) {
                $record = [
                    'user_id'               => $schedule['employee'],
                    'schedule_template_id'  => $schedule['schedule_template'],
                    'created_by'            => $request->user()->id,
                    'updated_by'            => $request->user()->id
                ];
                $inserted = EmployeeSchedule::create($record);
                if($inserted) {
                    array_push($schedules, $inserted);
                    if($inserted->schedule_template->basis == 2 && $schedule['hours'] != null) {
                        $inserted->hours()->create([
                            'hours'         => $schedule['hours'],
                            'created_by'    => $request->user()->id,
                            'updated_by'    => $request->user()->id,
                        ]);
                    }
                }
            }

            $this->logCustomMessage(
                'assign_invididual_schedule',
                null,
                Auth::user()->name . ' assign individual schedule',
                null,
                ScheduleLogType::ASSIGN_INDIVIDUAL,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text' => 'Employee Schedules have been successfully created for '.count($schedules).' employees.',
            ]);

        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    =>  ['There was a problem with creating schedules to the employee.'],
                'message'   =>  $e->getMessage()
            ], 400);
        }
    }

    /**
     * Assigning schedule to bulk employees.
     * The same schedule is assigned for all employees selected, 
     * unlike  from the assign individual where each employee 
     * has a specific schedule
     * 
     */
    public function assign_bulk_schedules(Request $request) {

        $validate = Validator::make($request->all(), [
            'template_id'       => 'required|integer',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();

        try {

            $template = ScheduleTitle::findOrFail($request->template_id);
            $employees = $request->employees;

            $schedules = [];
            if($template->basis == 1) {
                foreach($employees as $employee) {
                    $record = [
                        'user_id'               => $employee,
                        'schedule_template_id'  => $template->id,
                        'created_by'            => $request->user()->id,
                        'updated_by'            => $request->user()->id,
                        'created_at'            => now(),
                        'updated_at'            => now()
                    ];
                    if(count(User::findOrFail($employee)->employeeSchedules) == 0){
                        array_push($schedules, $record);
                    }
                }
    
                EmployeeSchedule::insert($schedules);
            } else {
                $date = date('Y-m-d');
                foreach($employees as $employee) {
                    $record = [
                        'user_id'               => $employee,
                        'schedule_template_id'  => $template->id,
                        'created_by'            => $request->user()->id,
                        'updated_by'            => $request->user()->id,
                        'created_at'            => now(),
                        'updated_at'            => now()
                    ];
                    $inserted = EmployeeSchedule::create($record);
                    array_push($schedules);
                    $inserted->hours()->create([
                        'hours'         => $request->hours,
                        'created_by'    => $request->user()->id,
                        'updated_by'    => $request->user()->id,
                    ]);
                }
            }

            $this->logCustomMessage(
                'assingn_bulk_schedule',
                null,
                Auth::user()->name . ' assign bulk schedule',
                null,
                ScheduleLogType::ASSIGN_BULK,
                new Activity()
            );
            
            DB::commit();
            return response()->json([
                'text'  => $template->title . ' has been assigned to ' . count($schedules) . ' employees.'
            ]);     

        } catch(ME $me) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['The selected employees or schedules could not be found.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch(Exception $e){
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in assigning the schedules.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Fetch employees with or without a schedule assigned at the main/index page
     * Filter parameters such as branch, corporation, division, and department are also used
     * for searching particular employees
     * 
     */
    public function filter_employees_main(Request $request) {

        try {

            $paginate = isset($request->paginate) && $request->paginate !== null ? $request->paginate : env('DEFAULT_PAGECOUNT');

            $users = User::whereHas('employeeMetaInfo', function($query) use ($request) {
                            $query->when($request->branch !== null, function($query) use ($request) {
                                return $query->where('branch_id', $request->branch);
                            })
                            ->when($request->corporation !== null, function($query) use ($request) {
                                return $query->where('corp_id', $request->corporation);
                            });
                        })
                        ->whereHas('actualDesignation', function($query) use ($request) {
                            $query->when($request->division !== null, function($query) use ($request) {
                                return $query->where('division_id', $request->division);
                            })
                            ->when($request->department !== null, function($query) use ($request) {
                                return $query->where('department_id', $request->department);
                            });
                        })
                        ->with(['employeeSchedules' => function($query) {
                                $query->with(['schedule_template' => function($query) { $query->with('schedule_template'); }]);
                            }, 
                            'roles' => function($query){
                                $query->whereNotIn('name', ['Super Admin', 'Admin', 'Developer']);
                            },
                            'temporary_schedule' => function($query) {
                                $query->with('schedule_template');
                            }, 
                            'actualDesignation.department', 
                            'actualDesignation.division',
                            'actualDesignation.section',
                            'actualDesignation.unit'
                            ])
                        ->where('name', 'LIKE', '%'.$request->employee_name.'%');

            if(count($request->employees) > 0) {
                $users = $users->whereIn('id', $request->employees);
            }
            return response()->json([
                'data' => $users
                        ->orderBy('created_at', 'asc')
                        ->paginate($paginate)
            ]);

        } catch (ME $me) {
            return response()->json([
                'errors'    => ['The selected filters could not be found.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the records.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Fetch employees who don't have any schedules yet to be displayed
     * at the bulk schedule assigning. 
     * 
     */
    public function filter_employees_bulk(Request $request) {

        try {
            $paginate = $request->page_count ? intval($request->page_count) : ENV('DEFAULT_PAGECOUNT');
            $users = User::doesntHave('employeeSchedules')
                    ->whereHas('employeeMetaInfo', function($query) use ($request) {
                        $query->when($request->branch !== null, function($query) use ($request) {
                            return $query->where('branch_id', $request->branch);
                        })
                        ->when($request->corporation !== null, function($query) use ($request) {
                            return $query->where('corp_id', $request->corporation);
                        })
                        ->when($request->position !== null, function($query) use ($request) {
                            return $query->where('position_id', $request->position);
                        })
                        ->when($request->employee_type !== null, function($query) use ($request) {
                            return $query->where('employee_type', $request->employee_type);
                        });
                    })
                    ->whereHas('actualDesignation', function($query) use ($request) {
                        $query->when($request->division !== null, function($query) use ($request) {
                            return $query->where('division_id', $request->division);
                        })
                        ->when($request->department !== null, function($query) use ($request) {
                            return $query->where('department_id', $request->department);
                        });
                    })
                    ->with(['roles' => function($query){
                        $query->whereNotIn('name', ['Super Admin', 'Admin', 'Developer']);
                    },
                    'actualDesignation.department', 
                    'actualDesignation.division',
                    'actualDesignation.section',
                    'actualDesignation.unit'
                    ])
                    ->orderBy('created_at', 'asc')
                    ->paginate($paginate);
            return response()->json([
                'data' => $users
            ]);

        } catch (ME $me) {
            return response()->json([
                'errors'    => ['The selected filters could not be found.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the records.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Fetch the branches and corporations for the filter parameters of the index page
     * 
     */
    public function fetch_branch_corps() {
        try {

            $branches = Branch::where('status', 1)->get();
            $corporations = Corporation::where('status', 1)->get();
            $departments = Department::where('status', 1)->get();
            $divisions = Division::where('status', 1)->get();

            return response()->json([
                'data' => [
                    'branches'      => $branches,
                    'corporations'  => $corporations,
                    'departments'   => $departments,
                    'divisions'     => $divisions,
                ]
            ]);

        } catch (ME $me) {
            return response()->json([
                'errors'    => ['The branches and corporations could not be found.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the records.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Edit a particular schedule assigned for the employee.
     * The only fields to be changed is the schedule template used
     * of the schedule for the employee and the effective dates
     */
    public function edit(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'                    => 'required|integer',
            'schedule_template_id'  => 'required|integer',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400); 
        }

        DB::beginTransaction();
        try {
            $schedule = EmployeeSchedule::findOrFail($request->id);
            $prev = $schedule->schedule_template;
            $new = ScheduleTitle::find($request->schedule_template_id);
            $old_record = $schedule;

            $schedule->update([
                'schedule_template_id'  => $request->schedule_template_id,
                'updated_by'            => $request->user()->id
            ]);

            if($new->basis == 2) {
                if($schedule->hours == null) {
                    $schedule->hours()->create([
                        'hours'         => $request->hours ? $request->hours : 0,
                        'created_by'    => $request->user()->id,
                        'updated_by'    => $request->user()->id
                    ]);
                } else {
                    $schedule->hours()->update([
                        'hours'         => $request->hours ? $request->hours : 0,
                        'updated_by'    => $request->user()->id
                    ]);
                }
            } else {
                if($prev->basis == 2) {
                    if($schedule->hours != null) $schedule->hours()->delete();
                }
            }
            $schedule->save();

            $this->logCustomMessage(
                'update_employee_schedule',
                $old_record,
                Auth::user()->name . ' update employee schedule',
                $schedule,
                ScheduleLogType::UPDATE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => 'Employee Schedule has been updated.'
            ]);

        } catch (ME $me) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['The target employee schedule could not be found.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in editing the record.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Delete a schedule record for the employee
     * 
     */
    public function delete(Request $request) {
        $validate = Validator::make($request->all(), [
            'id.*' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.employee_schedules,id',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400); 
        }

        DB::beginTransaction(); 

        try {

            foreach ($request->id as $id) {
                $record = EmployeeSchedule::with('user')->findOrFail($id);
                if($record->has('hours')) {
                    $record->hours()->delete();
                }

                $this->logCustomMessage(
                    'delete_employee_schedule',
                    $record,
                    Auth::user()->name . ' deleted schedule a for employee '. $record->user->name,
                    $record,
                    ScheduleLogType::DELETE,
                    new Activity()
                );

                $record->delete();
            } 

            DB::commit();
            return response()->json([
                'text'  => 'Employee Schedule has been deleted.'
            ]);

        } catch (ME $me) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['The target employee schedule could not be found.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in editing the record.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    public function fetch_employees_without_schedule() {
        return response()->json([
            'data' => User::role(['Employee'])->whereHas('employeeMetaInfo')
            ->with('actualDesignation', 'actualDesignation.department', 'actualDesignation.division', 'actualDesignation.section', 'actualDesignation.unit')
            ->doesntHave('employeeSchedules')->get()
        ]);
    }

    public function paginate_employees_without_schedule(Request $request) {
        $paginate = $request->page_count ? intval($request->page_count) : ENV('DEFAULT_PAGECOUNT');
        return response()->json([
            'data' => User::role(['Employee'])->whereHas('employeeMetaInfo')
            ->with('actualDesignation', 'actualDesignation.department', 'actualDesignation.division', 'actualDesignation.section', 'actualDesignation.unit')
            ->doesntHave('employeeSchedules')->paginate($paginate)
        ]);
    }

    public function fetch_employees_schedule() {
        $user = User::where('id', Auth::id())->first();
        $actualDesignation = $user->actualDesignation;

        if ($actualDesignation) {
            return response()->json([
                'data' => EmployeeSchedule::whereHas('user', function ($query) use ($actualDesignation) {
                    $query->whereHas('actualDesignation', function ($query) use ($actualDesignation) {
                        $fields = ['office_id', 'department_id', 'division_id', 'section_id', 'unit_id'];
                        foreach ($fields as $field) {
                            $query->when(isset ($actualDesignation->$field), function ($query) use ($field, $actualDesignation) {
                                $query->where($field, $actualDesignation->$field);
                            });
                        }
                    });
                })->get()
            ]);
        }
        
        return response()->json([
            'data' => EmployeeSchedule::get()
        ]);
    }

}
