<?php

namespace Suiterus\Adg\Controllers\Timekeeping\Employee;

use Exception;
use App\Models\User;
use App\Enums\Status;
use App\Enums\ScheduleBasis;
use Illuminate\Http\Request;
use App\Enums\AttendanceType;
use InvalidArgumentException;
use Illuminate\Support\Carbon;
use App\Enums\AttendanceLegend;
use Suiterus\Adg\Models\SM\Shift;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Suiterus\Adg\Models\SM\Suspension;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Services\Time\TimeService;
use Suiterus\Adg\Models\SM\ScheduleTemplate;
use Suiterus\Adg\Models\Timekeeping\Attendance;
use Suiterus\Adg\Models\Approvals\CTO\CTOApplication;
use Suiterus\Adg\Models\Timekeeping\EmployeeSchedule;
use Suiterus\Adg\Models\LeaveManagement\Requests\Leave;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Suiterus\Adg\Controllers\LeaveManagement\Table\Credits;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterDayEmployeeShift;
use Suiterus\Adg\Services\Attendance\AttendanceService;

/** Attendance controller used for the functions in the employee portal */

class AttendanceController extends Controller
{



    /**
     * Fetch the attendance records of a specific employee to be displayed in their dashboard
     *
     */
    public function fetch_employee_attendance(Request $req)
    {
        try {
            $user = User::findOrFail($req->user_id);
            date_default_timezone_set('UTC');

            $attendances = Attendance::where('user_id', $req->user_id)->when($req->year != null, function ($query) use ($req) {
                $query->whereMonth('time_in', $req->month)->whereYear('time_in', $req->year);
            })->get();


            $currentDay = Carbon::create($req->year, $req->month, 1)->startOfDay();
            $lastDay = $currentDay->copy()->endOfMonth()->endOfDay();

            $attendanceService = new AttendanceService($user);
            $totalOvertime = $attendanceService->totalOvertime($attendances);
            $totalUndertime = $attendanceService->totalUndertime($attendances);
            $totalLates = $attendanceService->totalLates($attendances);
            $totalNightDiff = $attendanceService->totalNightDifferential($attendances);
            $totalNightDiffOverTime = $attendanceService->totalNightDifferentialOvertime($attendances);
            $timeCount = $attendanceService->getTimeInCount($attendances); //working days
            $sumHours = $attendanceService->computeSumHours($attendanceService->computeTotalMinutes($attendances));
            $undertimeFrequency = $attendanceService->totalUnderTimeFrequency($attendances);
            $lateFrequency = $attendanceService->totalLateFrequency($attendances);

            $totalTardiness = TimeService::calculateTotalTime([$totalLates, $totalUndertime]);

            $totalFrequency = $undertimeFrequency + $lateFrequency;


            return response()->json([
                'data' => $attendanceService->getTimekeepingAttendanceDisplay($currentDay, $lastDay),
                'Tardiness' => $totalTardiness,
                'Overtime' => $totalOvertime,
                'Undertime' => $totalUndertime,
                'LateHours' => $totalLates,
                'NightDifferential' => $totalNightDiff,
                'NightDifferentialOverTime' => $totalNightDiffOverTime,
                'TimeInCount' => $timeCount,
                'TotalHours' => $sumHours,
                'FrequencyLate' => $lateFrequency,
                'FrequencyUnderTime' => $undertimeFrequency,
                'TotalFrequency' => $totalFrequency,
            ]);
        } catch (ME $me) {
            return response()->json([
                'errors'    => ['The selected filters could not be found.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the records.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }

    public function search_attendance_date(Request $req)
    {
        try {
            $user = User::where('id', $req->user()->id)->first();
            $year = $req->year ? (string) $req->year : Carbon::parse($user->created_at)->format('Y');
            $data = [];
            if($user) {
                for ($i = 1; $i < 13; $i++){
                    if((Carbon::now()->year == $year && Carbon::now()->month < $i) || Carbon::now()->year < $year) continue;
                    $data[] = [
                        'month' =>  $i,
                        'year'  => $year
                    ];
                    
                }
            }
            $dates = [
                'data' => $data
            ];

            return response()->json([
                'data' => $dates,
                'user' => $user

            ]);
        } catch (ME $me) {
            return response()->json([
                'errors'    => ['The selected filters could not be found.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the records.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }

    /**
     * Filter the attendance records of the employee by the range of dates
     *
     */
    public function filter_employee_attendance(Request $req)
    {
        $paginate = isset($req->paginate) && $req->paginate !== null ? $req->paginate : env('DEFAULT_PAGECOUNT');
        try {

            $user = Attendance::whereDate('time_in', ">=", $req->filter_dates[0])
                ->whereDate('time_in', "<=", $req->filter_dates[1])
                ->where('user_id', $req->user()->id);

            return response()->json([
                'data' => $user->paginate($paginate)
            ]);
        } catch (ME $me) {
            return response()->json([
                'errors'    => ['The selected filters could not be found.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the records.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Time-in function of the employee. The function first checks
     * If the employee has a temporary shift. If there is no temporary shift, an active schedule is checked.
     * If there is no active schedule, the employee will not be timed in
     *
     */
    public function create_check_in(Request $request)
    {
        DB::beginTransaction();
        try {

            $user = User::findOrFail($request->user()->id);

            $compare = true;

            // Check for temporary schedule
            $employee_schedule = $user->temporary_schedule;

            $currentdate = Carbon::now()->format('Y-m-d');
            $currenttime_check_in = Carbon::now()->format('H:i:s');
            $roster_shift = $this->getActiveRosterShift($user, $currentdate);
            // Get the schedule and timetable record for the attendance of the employee
            // Compare the dates to check which in the schedule template is the current date

            // If no temporary schedule, check if has active main schedule
            if ($roster_shift) {
                $compare = false;
            } elseif ($employee_schedule === null) {
                $employee_schedule = $this->checkActiveSchedule($user);

                // If active main schedule is not schedule-based, don't continue in comparing dates and checking late
                if ($employee_schedule->basis !== ScheduleBasis::SCHEDULE) {
                    $compare = false;
                }
            }

            $day_of_week = date('l');
            $schedule = $employee_schedule === null ? null : $employee_schedule->schedule_template()->where('day', $day_of_week)->first();

            // Get the timetable where the time-in is within the timetable range
            $sched_timetable = $schedule !== null ? ($schedule->timetables()->whereTime('time_out', '>=', $currenttime_check_in)->orderBy('time_in', 'asc')->first()) : null;


            $attendance = Attendance::create([
                "user_id"       => $request->user()->id,
                "date_in"       => $currentdate,
                "total"         => 0,
                "time_in"       => now(),
                'schedule_day_id' => $schedule !== null ? $schedule->id : null,
                'timetable_id'  => $sched_timetable !== null ? $sched_timetable->id : null,
                'shift_id'      => $roster_shift !== null ? $roster_shift->id : null,
                "type"          => AttendanceType::EMPLOYEE,
                "created_by"    => $request->user()->id,
                "updated_by"    => $request->user()->id,
            ]);

            if ($compare) {
                // TEMPORARY - timetable-based checking of attendance
                if ($sched_timetable !== null && (strtotime($currenttime_check_in) > strtotime($sched_timetable->max_time))) {
                    // Make late legend - to continue
                    $attendance->legends()->create([
                        'legend'    => AttendanceLegend::LATE
                    ]);
                }
            }

            //check roster shift if user has active roster shift for checking attendance if late
            if ($roster_shift && (strtotime($currenttime_check_in) > strtotime($roster_shift->shift_in))) {
                $attendance->legends()->create([
                    'legend'    => AttendanceLegend::LATE
                ]);
            }
            DB::commit();
            return response()->json([
                'data' => $attendance,
                'text' => 'Time in confirmed.',
            ]);
        } catch (InvalidArgumentException $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => [$e->getMessage()]
            ], 400);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(
                [
                    'errors' => ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : AM-comp-0x05'],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }

    /**
     * Time-out function of the employee.
     * Checks for the undertime and overtime of the employee
     *
     */
    public function update_check_out(Request $request)
    {

        $validate = Validator::make($request->all(), [
            'id'                    => 'required|integer'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            // Find an attendance record for the day
            $attendance = Attendance::findOrFail($request->id);
            $user = $attendance->user;
            $currenttime_check_out = Carbon::now();

            // Compute for the total time between the in and out
            $total = Carbon::parse($currenttime_check_out)->diff($attendance->time_in)->format('%H:%I');

            $compare = true;
            $break_in = null;
            $break_out = null;

            // Get and check temporary schedule and main schedule (same as time-in function)
            $employee_schedule = $user->temporary_schedule;
            $roster_shift = $this->getActiveRosterShift($user, $attendance->date_in);
            if ($roster_shift) {
                $compare = false;
                $break_in = $roster_shift->break_in != null ? $roster_shift->break_in : null;
                $break_out = $roster_shift->break_out != null ? $roster_shift->break_out : null;
            } elseif ($employee_schedule === null) {
                $employee_schedule = $this->checkActiveSchedule($user);
                if ($employee_schedule->basis !== ScheduleBasis::SCHEDULE) {
                    $compare = false;
                }
            }

            // Fetch suspensions to check if there are suspension the day prior
            $suspensions = Suspension::where('status', Status::ACTIVE)->get();
            if ($compare) {
                // Check if user is undertime
                $schedule_timetable = $attendance->timetable_basis;
                if ($schedule_timetable) {
                    //set time for break_in & out
                    $break_in = $schedule_timetable->break_in !== null ? $schedule_timetable->break_in : null;
                    $break_out = $schedule_timetable->break_out !== null ? $schedule_timetable->break_out : null;
                }
                if ($schedule_timetable !== null && (strtotime($currenttime_check_out) < strtotime($schedule_timetable->time_end))) {
                    //check if date of time in has current suspension created, if true, do not record as undertime
                    if ($suspensions->contains('date', $attendance->date_in)) {
                    } else {
                        $attendance->legends()->create([
                            'legend' => AttendanceLegend::UNDERTIME
                        ]);
                    }
                }
            }

            if ($roster_shift && (strtotime($currenttime_check_out) < strtotime($roster_shift->shift_out))) {
                $attendance->legends()->create([
                    'legend'    => AttendanceLegend::UNDERTIME
                ]);
            }

            if ($break_in && $break_out) {
                $hours_diff = Carbon::parse($break_in)->diffInSeconds(Carbon::parse($break_out));
                $hours_diff = Carbon::parse(gmdate('H:i', $hours_diff));
                $total = Carbon::parse($total);
                $total->subHours($hours_diff->format('H'));
                $total->subMinutes($hours_diff->format('i'));
                $total = $total->format('G:i');
            }

            $attendance->update([
                'total'         => $total,
                'time_out'      => now(),
                'break_in'      => $break_in,
                'break_out'     => $break_out,
                'updated_by'    => $request->user()->id,
            ]);

            $attendance->save();
            DB::commit();
            return response()->json([
                'text'  => 'Time out confirmed.'
            ]);
        } catch (ME $me) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['Schedule Temlate does not exist.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in updating the Employee Attendance'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Fetch the current schedule status of the employee
     * If no active schedule is found, returns no active schedule and cannot time in
     * If no attendance record is found, can time-in
     */
    public function fetch_status(Request $request)
    {
        $user = User::find($request->user()->id);
        $hasSchedule = $user->temporary_schedule;
        $type = 'temporary';
        if ($hasSchedule === null) {
            $hasSchedule = EmployeeSchedule::where('user_id', $request->user()->id)->first()->schedule_template;
            $type = 'main';
        }
        if ($hasSchedule == null) {
            return response()->json([
                'hasSchedule'   => false,
                'text'  => 'You are not assigned to a schedule yet.'
            ]);
        }

        // Check if the user as a time-in
        $attendance = Attendance::where('user_id', $request->user()->id)
            ->where('time_out', null)
            ->first();
        return response()->json([
            'hasSchedule'   => true,
            'template'  => $hasSchedule,
            'type' => $type,
            'data' => $attendance
        ]);
    }

    /**
     * Fetch attendance by date type: Last hour, today, this week etc.
     */
    public function filter_by_date_type(Request $request)
    {
        $now = now();
        $paginate = (isset($request->paginate) && $request->paginate !== null) ? intval($request->paginate) : env('DEFAULT_PAGECOUNT');

        switch (strtolower($request->date_filter)) {
            case 'last hour':
                $start_date = Carbon::parse($now)->subHour();
                break;
            case 'today':
                $start_date = Carbon::parse($now)->startOfDay();
                break;
            case 'this week':
                $start_date = Carbon::parse($now)->startOfWeek();
                break;
            case 'this month':
                $start_date = Carbon::parse($now)->startOfMonth();
                break;
            case 'this year':
                $start_date = Carbon::parse($now)->startOfYear();
                break;
            default:
                $start_date = Carbon::createFromTimeStamp(0);
                break;
        }

        try {

            $start_date = $start_date->toDateTimeString();
            $now = $now->toDateTimeString();
            $start = date('Y-m-d', strtotime($start_date));
            $end = date('Y-m-d', strtotime($now));

            $data = Attendance::when($request->date_filter !== 'all', function ($query) use ($start, $end) {
                $query->where(function ($query) use ($start, $end) {
                    $query->whereDate('time_in', '>=', $start)
                        ->where(function ($query) use ($end) {
                            $query->where('time_out', NULL)
                                ->orWhereDate('time_out', '<=', $end);
                        });
                });
            })
                ->where('user_id', $request->user()->id)
                ->with('user')
                ->orderBy('time_in', 'desc')
                ->paginate($paginate);

            return response()->json([
                'data'  => $data,
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in searching the records.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ]);
        }
    }

    /**
     * internal function that checks if the employee has an active schedule
     */
    private function checkActiveSchedule($user)
    {
        $today = date('Y-m-d');
        $employee_schedule = $user->employeeSchedules()->without('user')->first();

        if ($employee_schedule === null) {
            throw new InvalidArgumentException("You don't have an active schedule.");
        }
        return $employee_schedule->schedule_template;
    }

    private function getActiveRosterShift($user, $currentdate)
    {
        $shift = Shift::whereHas('employeeShift', function ($query) use ($currentdate, $user) {
            $query->whereHas('rosterDay', function ($query) use ($currentdate) {
                $query->where('date', $currentdate);
            })->whereHas('rosterEmployeePerGroup', function ($query) use ($user) {
                $query->where('user_id', $user->id)->whereHas('employeeGroup.roster', function ($query) {
                    $query->where('status', 1);
                });
            });
        })
            ->orWhereHas('headEmployeeShift', function ($query) use ($currentdate, $user) {
                $query->whereHas('rosterDay', function ($query) use ($currentdate) {
                    $query->where('date', $currentdate);
                })->whereHas('roster', function ($query) use ($user) {
                    $query->where('head_nurse_id', $user->id)
                        ->where('status', 1);
                });
            })->first();

        return $shift;
    }

    public function get_cto_leave_holiday($id, $date)
    {
        $leaveStatusApproved = 1;
        $leave = Leave::where([['user_id', $id], ['status', $leaveStatusApproved]])
            ->whereDate('start_date', '<=', $date)
            ->whereDate('end_date', '>=', $date)
            ->with('leaveType')
            ->with('credit_deductions')
            ->latest('updated_at')
            ->first();

        $ctoStatusApproved = 1;
        $cto = CTOApplication::where([['user_id', $id], ['status', $ctoStatusApproved]])
            ->whereDate('start_date', '<=', $date)
            ->whereDate('end_date', '>=', $date)
            ->latest('updated_at')
            ->first();
        return [$leave, $cto];
    }

    public function is_day_off($user_id, $day)
    {
        $user = User::where('id', $user_id)->with(['employeeSchedules' => function ($query) {
            $query->with(['schedule_template' => function ($query) {
                $query->with('schedule_template');
            }]);
        }, 'temporary_schedule' => function ($query) {
            $query->with('schedule_template');
        }])
            ->whereHas('employeeMetaInfo')
            ->first();
        $day = date("l", strtotime($day));

        if (!$user->employeeSchedules) {
            throw new Exception('Employee has no schedule');
        }

        $schedules = $user->employeeSchedules[0]['schedule_template']['schedule_template'];
        $days = [];
        foreach ($schedules as $scheduled) {
            $days[] = $scheduled['day'];
        }

        return !in_array($day, $days);
    }
}
