<?php

namespace Suiterus\Adg\Controllers\Timekeeping;

use App\Enums\Biometrics\EnabledStatus;
use App\Enums\Log\BiometricsLogType;
use App\Http\Controllers\Controller;
use App\Enums\Status;
use App\Exceptions\BiometricConnectException;
use Suiterus\Adg\Models\Timekeeping\BiometricRfidUser;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use App\Models\User;
use App\Models\Biometrics;
use App\Traits\Logs\HasCustomLogs;
use ErrorException;
use Illuminate\Http\Request;
use InvalidArgumentException;
use Suiterus\Adg\Models\Timekeeping\Biometrics as BIO;
use Illuminate\Support\Facades\Validator;
use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Timekeeping\BiometricDevice;
use Suiterus\Adg\Models\Timekeeping\Fingerprint;
use Suiterus\Adg\Services\ZKTecoService;

class BiometricsController extends Controller
{
    use HasCustomLogs;
    private $db;
    private $zk;

    public function __construct(){
        $this->db = DB::connection('adg_db');
    }

    /**
     * Fetch user device accounts - users that are registered in the device
     */
    public function index(Request $request) {

        $paginate = isset($request->paginate) && $request->paginate !== null ? $request->paginate : env('DEFAULT_PAGECOUNT');

        try {

            $data = BiometricRfidUser::with(['user' => function($query) {
                $query->without(['roles', 'storage', 'permissions'])
                ->with(['employeeMetaInfo' => function ($query) {
                    $query->without('branch', 'corporation', 'division', 'department', 'position')
                        ->select('user_id', 'employee_id');
                }, 'user_supervisor']);
            }, 'biometrics.fingerprints' => function($query) {
                $query->select('id', 'biometric_id', 'finger_number', 'updated_at');
            }])->whereHas('user', function($query) use($request) {
                $query->when($request->name != null, function ($query) use ($request) {
                    $query->where('name', "LIKE", '%' . $request->name . '%');
                })->when(isset($request->employees) && count($request->employees) > 0, function($query) use($request){
                    $query->whereIn('id', $request->employees);
                });
            })->paginate($paginate);

            return response()->json([
                'data' => $data  
            ]);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the data'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Check if the system can connect to the device
     */
    public function checkConnection() {

        try {

            return response()->json([
                'data'  => $this->zk->checkDeviceConnection()
            ]);          

        } catch(BiometricConnectException | ErrorException | Exception $e) {
            return response()->json([
                'errors'    => [__('responses.zkteco.connection-failed')],
                'message'   => $e->getMessage()
            ], 400);
        }

    }

    /**
     * Create zkteco device user
     */
    public function createDeviceUser(Request $request) {

        $validate = Validator::make($request->all(), [
            'data'             => 'required|array'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            foreach($request->data as $data){
                $user = User::find($data['user_id']);
                $devices = BiometricDevice::where('status', EnabledStatus::ENABLED)->get();
                if($devices === null) {
                    throw new BiometricConnectException('The device for the user does not exist.');
                }
                foreach($devices as $device){
                    $zk = new ZKTecoService($device->host, $device->port);
                    if(!$zk->checkDeviceConnection()) continue;
                    $zk->disable();
                    $zk->setUser($user, true, 0, null, $data['device_id']);
                    $zk->enable();
                    $zk->disconnect();
                }
            }
            
            $this->logCustomMessage(
                'create_biometrics_user',
                null,
                Auth::user()->name . ' Create Biometrics user',
                null,
                BiometricsLogType::CREATE,
                new Activity()
            );

            $this->db->commit();
            return response()->json([
                'text'  => 'ZKTeco users created.'
            ]);
        } catch(BiometricConnectException | ErrorException $e) {
            return response()->json([
                'errors'    => [__('responses.zkteco.connection-failed')],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 400);
        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in creating the device user'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    public function syncAllFingerprints(Request $request) {
        $this->db->beginTransaction();
        try {
            $token = $request->header('Authorization');           
            ZKTecoService::syncAllFingerprints($token);

            //This function is for future use when getting the templates of finger is optimized for now it will be commented
            // $devices = BiometricDevice::where('status', EnabledStatus::ENABLED)->get();
            // if($devices === null) {
            //     throw new BiometricConnectException('The device for the user does not exist.');
            // }
            // foreach($devices as $device){
            //     $zk = new ZKTecoService($device->host, $device->port);
            //     if (!$zk->checkDeviceConnection()) continue;
            //     $zk->disable();
            //     foreach ($users as $user) {  
            //         $zk->getUserFingerprints($user);
            //     }
            //     $zk->enable();
            //     $zk->disconnect();
            // }

            $this->logCustomMessage(
                'sync_all_fingerprints',
                null,
                Auth::user()->name . ' Sync all fingerprints',
                null,
                BiometricsLogType::SYNC,
                new Activity()
            );

            $this->db->commit();
            return response()->json([
                'text'  => 'All fingerprints synced.'
            ]);
        } catch(BiometricConnectException | ErrorException $e) {
            return response()->json([
                'errors'    => [__('responses.zkteco.connection-failed')],
                'message'   => $e->getMessage()
            ], 400);
        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in syncing the data'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }
    /**
     * Synchronize fingerprint data of user
     */
    public function syncFingerprints(Request $request) {

        $validate = Validator::make($request->all(), [
            'id' => 'required|exists:mysql.users,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {
            $token = $request->header('Authorization');
            $user = User::find($request->id);

            ZKTecoService::syncFingerprints($user,$token);

            $devices = BiometricDevice::where('status', EnabledStatus::ENABLED)->get();
            if($devices === null) {
                throw new BiometricConnectException('The device for the user does not exist.');
            }
            foreach($devices as $device){
                $zk = new ZKTecoService($device->host, $device->port);
                if(!$zk->checkDeviceConnection()) continue;
                $zk->disable();
                $zk->getUserFingerprints($user);
                $zk->enable();
                $zk->disconnect();
            }

            $this->logCustomMessage(
                'sync_fingerprint_user',
                null,
                Auth::user()->name . ' Sync user fingerprints',
                null,
                BiometricsLogType::SYNC,
                new Activity()
            );

            $this->db->commit();
            return response()->json([
                'text'  => 'Fingerprints synced.'
            ]);
        } catch(BiometricConnectException | ErrorException $e) {
            return response()->json([
                'errors'    => [__('responses.zkteco.connection-failed')],
                'message'   => $e->getMessage()
            ], 400);
        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in syncing the data'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    //create
    public function create_biometrics(Request $req)
    {
        $validate = Validator::make($req->all(), [
            'data'             => 'required|array'
            
        ]);
        
        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();

        try{
            $records = array();
            foreach($req->data as $data){
                $record = [
                    'user_id'           => $data['user_id'],
                    'type'              => $data['type'],
                    'biometric_number'  => $data['biometric_number'],
                    'status'            => Status::INACTIVE,
                    'date_issued'       => date('Y-m-d'),
                    'created_by'        => $req->user()->id,
                    'created_at'        => now(),
                ];
                array_push($records, $record);
            }

            BIO::insert($records);

            DB::commit();

            return response()->json([
                'text' => 'New biometric record has been created.',
            ]);
            
            } catch(BiometricConnectException | ErrorException $e) {
                return response()->json([
                    'errors'    => [__('responses.zkteco.connection-failed')],
                    'message'   => $e->getMessage()
                ], 400);
            } catch(Exception $e) {
                DB::connection(env('ADG_DB_CONNECTION'))->rollBack();
                return response()->json([
                    'errors'    => ['The request could not be processed.'],
                    'message'   => $e->getMessage(),
                ], 500);
            } 
    }   

    //fetchall
    public function fetch_all_biometrics(Request $req){
        $paginate = $req->paginate ? intval($req->paginate) : env('DEFAULT_PAGECOUNT');

        $data = BIO::with(['user' => function ($q){
            $q->select('id','name')->without('roles', 'permissions', 'storage', 'employeeMetaInfo', 'supervisor', 'exitInterview');
        
        }])->paginate($paginate);

        return response()->json([
            'data' => $data
        ]);

    }

    //fetch single bio
    public function fetch_single_biometrics(Request $req){
        
        $valid = Validator::make($req->all(), [

            'id' => 'required|exists:adg_db.biometrics,id'
        ]);

        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }

        try{
            $single_bio = BIO::where('id', $req->id)->first();
            
            return response()->json([
                'message' => 'Fetch successful.',
                'data'    => $single_bio,
            ]);

        }catch (Exception $e) {
            
            return response()->json([
                'error' => ['Can`t fetch your biometric as of now.'],
                'msg' => $e->getMessage(),
            ]);
        }

    }

    //update
    public function update_biometrics(Request $req){
        $validate = Validator::make($req->all(), [
            'id'    =>'required|exists:adg_db.biometrics,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }
        DB::beginTransaction();

        try{
    
            BIO::where('id', $req->id)
            ->update([
                'user_id'           => $req->user_id,
                'type'              => $req->type,
                'biometric_number'  => $req->biometric_number,
                'status'            => $req->status,
                'date_issued'       => $req->date_issued
            
            ]);

        DB::commit();
        return response()->json([
                'message' => "Biometics has been successfully updated!"
            ]);

        
        DB::rollback();
        }catch(Exception $e){
            return $e->getmessage();
        }
    }

    //delete
    public function delete_biometrics(Request $req){
    
        $valid = Validator::make($req->all(), [
            'id' => 'required|exists:adg_db.biometrics,id'
        ]);
        
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }

        DB::beginTransaction();
        try{

            $delete = BIO::findOrFail($req->id);
            $delete->delete();
            DB::commit();

            return response()->json([
                'text'  =>  'Biometric has been deleted.',
                
            ]);

        }catch(Exception $e){

            DB::rollback();
                
            return response()->json([
                'errors'    =>  [ 'The request could not be process.' ],
                'msg'   =>  $e->getMessage()
            ],500);

        }
        
    }

    //filter
    public function filter_biometrics(Request $req){
        $paginate = $req->paginate ? intval($req->paginate) : env('DEFAULT_PAGECOUNT');
    
        $records = BIO::with(['user' => function($query){
    
            $query->select('id', 'name') -> without(['roles', 'permission', 'storage', 'employeeMetaInfo', 'supervisor', 'exitInterview']);
    
        }])->when($req->date!=null, function($query) use ($req){
            $query->whereDate('date_issued', $req->date);
    
        })->when($req->type!=null, function($query) use ($req){
            $query->where('type', $req->type);

        })->when($req->status!=null, function($query) use ($req){
            $query->where('status', $req->status);
    
        })->whereHas("user", function($query) use ($req){
            $query->when($req->name!=null, function($query) use ($req) {
                $query->where('name', "LIKE", '%'. $req->name . '%');
            });
    
        })->orderBy('created_at', 'asc')->paginate($paginate);
    
        return response()->json([
            'data' => $records
        ]);
    
    }

    //aprove
    public function approve(Request $request)
    {

        $validate = Validator::make($request->all(), [
            'id'    => 'required|exists:adg_db.biometrics,id'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'  => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $biometrics = BIO::find($request->id);

            if ($biometrics->status == Status::ACTIVE) {
                throw new InvalidArgumentException('The record is already approved');
            }

            $biometrics->status = Status::ACTIVE;

            $biometrics->save();
            DB::commit();
            return response()->json([
                'text'  => 'Biometric approved.'
            ]);
        } catch (InvalidArgumentException $e) {
            DB::connection('adg_db')->rollBack();
            return response()->json([
                'errors'    => $e->getMessage(),
                'message'   => $e->getMessage()
            ], 500);
        } catch (Exception $e) {
            DB::connection('adg_db')->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in approving the records'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }


    //decline
    public function decline(Request $request)
    {

        $validate = Validator::make($request->all(), [
            'id'    => 'required|exists:adg_db.biometrics,id'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'  => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $biometrics = BIO::find($request->id);

            if ($biometrics->status == Status::INACTIVE) {
                throw new InvalidArgumentException('The record is already declined');
            }

            $biometrics->status = Status::INACTIVE;

            $biometrics->save();
            DB::commit();
            return response()->json([
                'text'  => 'Biometric declined.'
            ]);
        } catch (InvalidArgumentException $e) {
            DB::connection('adg_db')->rollBack();
            return response()->json([
                'errors'    => $e->getMessage(),
                'message'   => $e->getMessage()
            ], 500);
        } catch (Exception $e) {
            DB::connection('adg_db')->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in declining the records'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Fetch users/employees without a Biometrics record
     */
    public function employees() {

        try {

            return response()->json([
                'data'  => User::whereDoesntHave('biometric_rfid_record.biometrics')
                    ->without(['roles', 'storage', 'permissions'])
                    ->whereHas('employeeMetaInfo')
                    ->with(['employeeMetaInfo' => function ($query) {
                        $query->without('branch', 'corporation', 'division', 'department', 'position')
                            ->select('user_id', 'employee_id');
                    }, 'user_supervisor'])    
                    ->get() 
            ]);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in getting the request.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

}