<?php

namespace Suiterus\Adg\Controllers\Timekeeping;

use Exception;
use Validator;
use Carbon\Carbon;
use App\Models\User;
use App\Enums\ScheduleBasis;
use Illuminate\Http\Request;
use App\Enums\AttendanceType;
use InvalidArgumentException;
use App\Enums\AttendanceLegend;
use App\Enums\Log\AttendanceLogType;
use Suiterus\Adg\Models\SM\Shift;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Traits\Logs\HasCustomLogs;
use Suiterus\Adg\Services\ZKTecoService;
use Suiterus\Adg\Models\Timekeeping\Timetable;
use Suiterus\Adg\Models\Timekeeping\Attendance;
use Suiterus\Adg\Models\Payroll\TimekeepingDeduction;
use Suiterus\Adg\Services\Attendance\AttendanceService;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Timekeeping\AttendanceLegend as TimekeepingAttendanceLegend;

/** 
 * Attendance controller used for the functions in the admin portal - admin functionalities
 *  
 */


class AttendanceController extends Controller
{
    use HasCustomLogs;
    private $db;
    private $zk;

    public function __construct(){
        $this->db = DB::connection('adg_db');
        $this->zk = new ZKTecoService();
    }

    /**
     * Create new manual attendance records per employee
     */
    public function create(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'attendances.*.employeeSelected'    => 'required|integer',
            'attendances.*.time_in'             => 'required',
            //'attendances.*.time_out'            => 'required',
            'attendances.*.total'               => 'required',
        ]);
        if ($valid->fails()) {
            return response()->json(
                [
                    'errors' => $valid->errors(),
                ],
                400
            );
        }
        DB::beginTransaction();
        try {

            // Loop through the attendance records in the request
            $records = 0;
            foreach($req->attendances as $attendance) {
                $user = User::findOrFail($attendance['employeeSelected']);
                $employee_schedule = $user->temporary_schedule_record()
                    ->where(function($query) use ($attendance) {
                        $date = date('Y-m-d', strtotime($attendance['time_in']));
                        $query->where('start_date', '<=', $date)
                            ->whereRaw('CASE WHEN date_disabled is not null then date_disabled >= "' . $date .'" ELSE end_date >= "' . $date . '" END' );
                    })->first();
                $sched_timetable = null;
                $schedule = null;
                $legends = [];
                $break_in = null;
                $break_out = null;
                // Format time_in and time_out
                $time_in = date('Y-m-d H:i:s', strtotime($attendance['time_in']));
                $time_out = $attendance['time_out'] ? date('Y-m-d H:i:s', strtotime($attendance['time_out'])) : null;
                $total = $time_out ? Carbon::parse($time_out)->diff($time_in)->format('%H:%I') : '0:00';
                $attendance_service = new AttendanceService($user);

                $roster_shift = $attendance_service->getRosterShift(Carbon::parse($time_in)->format('Y-m-d'));
                // $this->getActiveRosterShift($user, Carbon::parse($time_in)->format('Y-m-d')); use this code get roster active shift
                // if has employee schedule, continue to getting schedule tempalate day and timetable details
                if($roster_shift) {

                    $break_in = $roster_shift->break_in ? $roster_shift->break_in : null;
                    $break_out = $roster_shift->break_out ? $roster_shift->break_out : null;

                      // Check Late
                    if(strtotime(date('H:i:s', strtotime($attendance['time_in']))) > strtotime($roster_shift->shift_in)) {
                        array_push($legends, AttendanceLegend::LATE);
                    }

                    // Check undertime
                    if(strtotime(date('H:i:s', strtotime($attendance['time_out']))) < strtotime($roster_shift->shift_out)) {
                        array_push($legends, AttendanceLegend::UNDERTIME);
                    }

                } elseif ($employee_schedule === null) {

                    $employee_schedule = $attendance_service->getActiveSchedule();
                    if ($employee_schedule) {
                        if ($employee_schedule->basis === ScheduleBasis::SCHEDULE) {
                            // Get the schedule template day and the timetables associated with the day
                            $day_of_week = date('l', strtotime($time_in));
                            /**
                             * Getting the timetable for the current time in recorded
                             * To get the timetable, check if the time-in time is greater than the time in of the timetables
                             * if there are two timetables, there will be 2 results. Only the first result is fetched and
                             * compared which is the latest (if morning and afternoon and the time in is greater than the time in of both, get the afternoon), 
                             * since if the time is greater than the morning only, then it automatically means that the schedule used will not be the afternoon session 
                             */
                            $schedule = $employee_schedule->schedule_template()->where('day', $day_of_week)->first();
                            $sched_timetable = $schedule ? ($schedule->timetables()->whereTime('time_out', '!=', date('H:i:s', strtotime($time_in)))->orderBy('time_in', 'asc')->first()) : null;

                            // If timetable is found
                            if ($sched_timetable) {
                                //set time for break_in & out
                                $break_in = $sched_timetable->break_in ? $sched_timetable->break_in : null;
                                $break_out = $sched_timetable->break_out ? $sched_timetable->break_out : null;
                                // Check Late
                                if (strtotime(date('H:i:s', strtotime($attendance['time_in']))) > strtotime($sched_timetable->max_time)) {
                                    array_push($legends, AttendanceLegend::LATE);
                                }

                                // Check undertime
                                if (strtotime(date('H:i:s', strtotime($attendance['time_out']))) < strtotime($sched_timetable->time_out)) {
                                    array_push($legends, AttendanceLegend::UNDERTIME);
                                }
                            }

                            $schedule = $schedule ? $schedule->id : null;
                            $sched_timetable = $sched_timetable ? $sched_timetable->id : null;
                            $records++;
                        }
                    }
                }

                if ($attendance['break_in'] && $attendance['break_out']) {
                    $break_in = $attendance['break_in'];
                    $break_out = $attendance['break_out'];
                }
                //sub break time to total hours if exist
                if($break_in && $break_out && $time_out) {
                    $hours_diff = Carbon::parse($break_in)->diffInSeconds(Carbon::parse($break_out));
                    $hours_diff = Carbon::parse(gmdate('H:i', $hours_diff));
                    $total = Carbon::parse($attendance['total']);
                    $total->subHours($hours_diff->format('H'));
                    $total->subMinutes($hours_diff->format('i'));
                    $total = $total->format('G:i');
                }

                // Create the attendance record
                $record = Attendance::create([
                    'time_in'       => $time_in,
                    'time_out'      => $time_out,
                    'break_in'      => $attendance['break_in'] ? $break_in : null,
                    'break_out'     => $attendance['break_out'] ? $break_out : null,
                    'user_id'       => $attendance['employeeSelected'],
                    'total'         => $total,
                    'schedule_day_id' => $schedule,
                    'timetable_id'  => $sched_timetable,
                    'shift_id'      => $roster_shift ? $roster_shift->id : null,
                    'remarks'       => $attendance['remarks'],
                    'type'          => AttendanceType::MANUAL,
                    'created_by'    => $req->user()->id,
                    'updated_by'    => $req->user()->id,
                ]);

                if (($record->break_in && $record->break_out) && ($record->scheduled_break_in && $record->scheduled_break_out) ) {
                    if (strtotime(date('H:i:s', strtotime($record->break_in))) < strtotime($record->scheduled_break_in)) {
                        array_push($legends, AttendanceLegend::UNDERTIME);
                    }
                    if (strtotime(date('H:i:s', strtotime($record->break_out))) > strtotime($record->scheduled_break_out)) {
                        array_push($legends, AttendanceLegend::LATE);
                    }
                }

                // record the legends
                foreach($legends as $legend) {
                    $record->legends()->create([
                        'legend' => $legend,
                    ]);
                }

                $records++;
            }

            $this->logCustomMessage(
                'create_manual_attendance',
                null,
                Auth::user()->name . ' Create manual attendance',
                null,
                AttendanceLogType::CREATE_MANUAL,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text' => 'Attendance for ' . $records . ' employees have been created.',
            ]);
        } catch(InvalidArgumentException $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => [$e->getMessage()]
            ], 403);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json(
                [
                    'errors' => ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : AM-comp-0x05'],
                    'msg' => $e->getMessage(),
                ],
                500
            );
        }
    }

    /**
     * Synchronize ZKTeco Attendance records from device to System
     * [Dec 2022] Previously used for manual synchronization
     */
    public function syncBiometricsAttendance() {

        $this->db->beginTransaction();
        try {

            $this->zk->syncAttendance();

            $this->db->commit();
            return response()->json([
                'text'  => 'Attendance synced'
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in syncing the attendance records.'],
                'message'   => $e->getMessage()
            ]);
        }

    }

    /**
     * Fetch employees with schedules for attendance
     */
    public function fetch_name()
    {
        return response()->json([
            'data' => User::whereHas('employeeMetaInfo')->orderBy('created_at', 'asc')->get()
        ]);
    }

    /**
     * Fetch attendances with filtered parameters  
     */
    public function filter_attendance(Request $req){

        $paginate = isset($req->paginate) && $req->paginate !== null ? $req->paginate : env('DEFAULT_PAGECOUNT');
        try{
            $users = Attendance::when(count($req->filter_dates)>1, function($q) use ($req){
                    $q->whereDate('time_in', '>=', $req->filter_dates[0])
                    ->whereDate('time_in', '<=', $req->filter_dates[1]);
                })->when(count($req->filter_dates)==1, function($q) use ($req){
                    $q->whereDate('time_in', $req->filter_dates[0]);
                })->when(count($req->filter_time)==1, function($q) use ($req){
                    $q->whereTime('time_in', '>=', $req->filter_time[0]);
                })->when(count($req->filter_time)>1, function($q) use ($req){
                    $q->whereTime('time_in', '>=', $req->filter_time[0])
                    ->whereTime('time_in', '<=', $req->filter_time[1])
                    ->whereTime('time_out', '>=', $req->filter_time[0])
                    ->whereTime('time_out', '<=', $req->filter_time[1]);
                })
                ->when($req->employee_name !== null && $req->employee_name !== '', function($query) use ($req) {
                    $query->whereHas('user', function($query) use ($req) {
                        $keyword = '%' . $req->employee_name . '%';
                        $query->where('name', 'LIKE', $keyword)
                            ->orWhereHas('employeeMetaInfo', function($query) use($keyword) {
                                $query->whereHas('branch', function($query) use ($keyword) {
                                    $query->where('name', 'LIKE', $keyword);
                                })->orWhereHas('corporation', function($query) use ($keyword) {
                                    $query->where('name', 'LIKE', $keyword);
                                })->orWhereHas('department', function($query) use ($keyword) {
                                    $query->where('name', 'LIKE', $keyword);
                                })->orWhereHas('division', function($query) use ($keyword) {
                                    $query->where('name', 'LIKE', $keyword);
                                })
                                ;
                            });
                    });
                })->with(['user', 'legends'])
                ->orderBy('updated_at', 'DESC');

            if(count($req->employees) > 0) {
                $users = $users->whereHas('user', function($query) use ($req) {
                    $query->whereIn('id', $req->employees);
                });
            }
            
            return response()->json([
                'data' => $users->orderBy('updated_at', 'asc')->paginate($paginate)
            ]);
        
        } 
        catch (ME $me) {
            return response()->json([
                'errors'    => ['The selected filters could not be found.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in fetching the records.'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Edit the attendance record - only the remarks and legends are updatable
     */
    public function edit(Request $request) {

        $valid = Validator::make($request->all(), [
            'id'                  => 'required|exists:adg_db.attendances,id',
        ]);

        if($valid->fails()) {
            return response()->json([
                'errors'    => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $record = Attendance::findOrFail($request->id);
            $old_record = $record;

            $record->update([
                'time_in'    => $request->time_in,
                'time_out'   => $request->time_out,
                'break_in'    => $request->break_in,
                'break_out'   => $request->break_out,
                'total'      => $request->total,
                'remarks'    => $request->remarks,
                'updated_at' => now(),
            ]); 

            $user = User::whereId($record->user_id)->first();
            $attendance_service = new AttendanceService($user);
            $attendance_service->reEvaluateAttendance(date('Y-m-d H:i:s', strtotime($record->time_in)), $record);

            $this->logCustomMessage(
                'update_attendance',
                $old_record,
                Auth::user()->name . ' Update attendance',
                $record,
                AttendanceLogType::UPDATE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => $record->user->name . "'s record has been updated."
            ]);

        } catch(InvalidArgumentException $e) {
            return response()->json([
                'errors'    => [$e->getMessage()],
            ], 500);
        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in editing the record.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Delete the attendance record for the employee (soft delete)
     */
    public function delete(Request $request) {

        DB::beginTransaction();
        try  {

            $record = Attendance::findOrFail($request->id);
            $name = $record->user->name;
            $record->delete();

            $this->logCustomMessage(
                'delete_attendance',
                $record,
                Auth::user()->name . ' delete attendance',
                $record,
                AttendanceLogType::DELETE,
                new Activity()
            );

            DB::commit();

            return response()->json([
                'text'  => $name . "'s has been deleted."
            ]); 

        } catch(ME $e) {
            return response()->json([
                'errors'    => ['The selected record could not be found.'],
                'message'   => $e->getMessage()
            ], 500);
        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in deleting the record.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Check for the active schedule of the employee
     */
    private function checkActiveSchedule($user, $attendance) {
        $today = date('Y-m-d', strtotime($attendance));
        $employee_schedule = $user->employeeSchedules()->without('user')->first();
        if($employee_schedule === null) {
            throw new InvalidArgumentException("Employee don't have an active schedule.");
        }
        return $employee_schedule->schedule_template;
    }

    private function getActiveRosterShift($user, $currentdate) {
        $shift = Shift::whereHas('employeeShift',function ($query) use ($currentdate, $user) {
            $query->whereHas('rosterDay', function ($query) use ($currentdate) {
                $query->where('date',$currentdate);
            })->whereHas('rosterEmployeePerGroup', function ($query) use ($user){
                $query->where('user_id',$user->id)->whereHas('employeeGroup.roster', function ($query) {
                    $query->where('status', 1);
                });
            });
        })
        ->orWhereHas('headEmployeeShift', function($query) use ($currentdate, $user) {
            $query->whereHas('rosterDay', function ($query) use ($currentdate) {
                $query->where('date',$currentdate);
            })->whereHas('roster', function ($query) use ($user) {
                $query->where('head_nurse_id', $user->id)
                    ->where('status', 1);
            });
        })->first();
    
        return $shift;
    }

}
