<?php

namespace Suiterus\Adg\Controllers\SuccessionManagement;

use Carbon\Carbon;
use App\Models\Successor;
use Illuminate\Http\Request;
use Suiterus\Adg\Models\SM\Sector;
use App\Http\Controllers\Controller;
use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Suiterus\Hrjp\Models\Position_history;
use Suiterus\Adg\Models\PDS\PDSPerPosition;
use Suiterus\Hrjp\Models\Position_has_salary;
use Suiterus\Adg\Models\Salary\EmployeeAllowance;
use Suiterus\Adg\Models\ServiceRecord\ServiceRecord;
use Suiterus\Adg\Models\EMI\EmployeeExtraFieldColumn;

class SuccessorController extends Controller
{

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        foreach ($request->successors as $successor) {
            Successor::create([
                'critical_position_id' => $request->critical_position_id,
                'successor_id' => $successor
            ]);
        }

        return response()->json([
            'message' => 'Successor created successfully'
        ]);
    }

    public function paginate(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return Successor::whereHas('successor', function ($query) use ($request) {
            $query->whereHas('employeeMetaInfo', function ($query) use ($request) {
                $query->when($request->position_id != null && isset($request->position_id) && $request->position_id != '', function ($query) use ($request) {
                    $query->where('position_id', $request->position_id);
                })->when($request->department_id != null && isset($request->department_id) && $request->department_id != '', function ($query) use ($request) {
                    $query->where('department_id', $request->department_id);
                })->when($request->division_id != null && isset($request->division_id) && $request->division_id != '', function ($query) use ($request) {
                    $query->where('division_id', $request->division_id);
                })->when($request->item_code != null && isset($request->item_code) && $request->item_code != '', function ($query) use ($request) {
                    $query->whereHas('itemCode', function ($query) use ($request) {
                        $query->where('item_code', 'LIKE', '%' . $request->item_code . '%');
                    });
                });
            })->when($request->successor_name != null && isset($request->successor_name) && $request->successor_name != '', function($query) use($request){
                $query->where('name', 'LIKE', '%'.$request->successor_name .'%');
            });
        })->when($request->number_of_promotion != null && isset($request->number_of_promotion) && $request->number_of_promotion != '', function($query) use($request){
            $query->has('successorPromotions', $request->number_of_promotion);
        })->where('critical_position_id', $request->critical_position_id)->with('successor')->paginate($paginate);
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Successor  $successoSuccessor
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        return Successor::where('id', $request->successor_id)->with([
            'successor.elementary',
            'successor.secondary',
            'successor.vocational',
            'successor.college',
            'successor.graduate',
            'successor.civilService',
            'successor.workExperience',
            'successor.voluntaryWork',
            'successor.learningDevelopment',
            'successor.skillHobby',
            'successor.evaluation',
            'successor.salary.publicSalary',
            'successor.salary.privateSalary'
        ])->first();
    }

    public function promote(Request $request)
    {

        DB::beginTransaction();
        try {
            $activeSector = Sector::where('status', 'active')->pluck('name')->first();

            $successor = Successor::where('id', $request->successor_id)->first();

            $criticalPosition = $successor->criticalPosition;

            if (!$criticalPosition->currentIncumbent) {
                return response()->json([
                    'message' => 'Critical position has no incumbent'
                ], 409);
            }

            $currentIncumbent = $criticalPosition->currentIncumbent()->with(['salary.publicSalary', 'salary.privateSalary'])->first();

            $currentIncumbentItemCode = $currentIncumbent->employeeMetaInfo->itemCode;
            $successorItemCode = $successor->successor->employeeMetaInfo->itemCode;

            if ($currentIncumbentItemCode->position_id == $successorItemCode->position_id) {
                return response()->json([
                    'message' => 'Successor have the same position as incumbent'
                ], 409);
            }

            $successorSalaryGrade = $successor->successor->salary->publicSalary->salary_grade_id;
            $currentIncumbentSalaryGrade = $currentIncumbent->salary->publicSalary->salary_grade_id;

            $currentIncumbentSalary = (float) str_replace(',', '', $currentIncumbent->salary->publicSalary->salaryGrade->value);
            $currentIncumbentAnnualSalary = $currentIncumbentSalary * 12;

            $successor->successor->employeeMetaInfo->update([
                'item_code_id' => $currentIncumbentItemCode->id,
                'position_id' => $currentIncumbentItemCode->position_id
            ]);

            if ($activeSector == 'public') {
                $currentIncumbentSalaryGradeId = $currentIncumbent->salary->publicSalary->salary_grade_id;

                $successor->successor->salary->publicSalary->update([
                    'salary_grade_id' => $currentIncumbentSalaryGradeId
                ]);
            }

            $successor->update([
                'status' => 2
            ]);

            $successor->criticalPosition->update([
                'status' => 2,
                'updated_by' => Auth::id()
            ]);

            if ($currentIncumbentItemCode->position_id != $successorItemCode->position_id) {
                Position_history::where([
                    ['user_id', $successor->successor_id],
                    ['item_code_id', $successorItemCode->id],
                    ['position_id', $successorItemCode->position_id]
                ])->update([
                    'position_status' => 2,
                    'updated_by' => Auth::id()
                ]);

                $position_history = Position_history::create([
                    'user_id' => $successor->successor_id,
                    'item_code_id' => $currentIncumbentItemCode->id,
                    'position_id' => $currentIncumbentItemCode->position_id,
                    'appointment_date' => Carbon::now(),
                    'date_of_effectivity' => Carbon::now(),
                    'status' => 1,
                    'position_status' => 1,
                    'employment_status' => 0,
                    'created_by' => Auth::id(),
                ]);

                $employeeExtraFieldColumn = EmployeeExtraFieldColumn::whereHas('employeeExtraField', function ($query) use ($successor) {
                    $query->where('user_id', $successor->successor_id);
                })->get();

                foreach ($employeeExtraFieldColumn as $pds) {
                    PDSPerPosition::create([
                        'position_history_id'   =>    $position_history->id,
                        'eefc_id'               =>    $pds->id,
                        'field_name'            =>    $pds->field_name,
                        'field_value'           =>    $pds->field_value,
                        'created_by'            =>    Auth::id()
                    ]);
                }
            }

            $oldServiceRecord = ServiceRecord::where('user_id', $successor->successor_id)->whereNull('end_date');
            $oldServiceRecord->update([
                'end_date'  => Carbon::now()
            ]);

            $sucessorAllowances = EmployeeAllowance::where('user_id', $successor->successor_id)->get();

            $allowanceSum = 0;

            foreach ($sucessorAllowances as $allowances) {
                $allowanceSum += $allowances->amount;
            }

            if ($currentIncumbentSalaryGrade != $successorSalaryGrade) {
                ServiceRecord::create([
                    'user_id'           =>  $successor->successor_id,
                    'start_date'        =>  Carbon::now(),
                    'position_id'       =>  $currentIncumbentItemCode->position_id,
                    'salary'            =>  $currentIncumbentAnnualSalary,
                    'allowance'         =>  $allowanceSum,
                    'employee_type_id'  =>  $successor->successor->employeeMetaInfo->employee_type,
                    'division_id'       =>  $successor->successor->employeeMetaInfo->division_id,
                    'remark_id'         =>  20,  //* Step Increment
                    'created_by'        =>  Auth::id()
                ]);
            }

            if ($currentIncumbentItemCode->position_id != $successorItemCode->position_id) {
                ServiceRecord::create([
                    'user_id'           =>  $successor->successor_id,
                    'start_date'        =>  Carbon::now(),
                    'position_id'       =>  $currentIncumbentItemCode->position_id,
                    'salary'            =>  $currentIncumbentAnnualSalary,
                    'allowance'         =>  $allowanceSum,
                    'employee_type_id'  =>  $successor->successor->employeeMetaInfo->employee_type,
                    'division_id'       =>  $successor->successor->employeeMetaInfo->division_id,
                    'remark_id'         =>  2,  //* PROMOTED
                    'created_by'        =>  Auth::id()
                ]);
            }

            DB::commit();

            return response()->json([
                'message' => 'Successor promoted successfully.',
            ], 200);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'An error has occurred. Please try again later.',
            ], 500);
        }
    }
}
