<?php

namespace Suiterus\Adg\Controllers\SuccessionManagement;

use Illuminate\Http\Request;

use Suiterus\Hrjp\Models\ItemCode;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\EMI\EmployeeMetaInfo;
use Suiterus\Adg\Models\SuccessionManagement\CriticalPosition;

class CriticalPositionController extends Controller
{

    public function index(){
        return CriticalPosition::where('status', 1)->get();
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {

        $itemCodeId = ItemCode::where([
            ['position_id', $request->position_id],
            ['item_code', $request->item_code]
        ])->pluck('id')->first();

        $criticalPosition = CriticalPosition::where([
            ['item_code_id', $itemCodeId],
            ['status', 1]
        ])->first();

        if ($criticalPosition) {
            return response()->json([
                'message' => 'Critical position already exists.'
            ], 409);
        }

        $currentIncumbent = EmployeeMetaInfo::without([
            'corporation',
            'branch',
            'division',
            'department',
            'employeeType',
            'position',
            'itemCode'
        ])->where('item_code_id', $itemCodeId)->first();

        if (!$itemCodeId) {
            return response()->json([
                'message' => 'Item code not found.'
            ], 404);
        }

        $criticalPosition = CriticalPosition::create([
            'item_code_id' => $itemCodeId,
            'vacancy_start' => $request->vacancy_start,
            'current_incumbent_id' => $currentIncumbent ? $currentIncumbent->user->id : null,
            'reason_for_replacement_id' => $request->reason_for_replacement_id,
            'created_by' => Auth::id(),
        ]);

        return response()->json([
            'data' => $criticalPosition,
            'message' => 'Critical position created successfully.'
        ]);
    }

    public function paginate(Request $request){
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return CriticalPosition::when($request->position_id != null && isset($request->position_id) && $request->position_id != '', function($query) use($request){
            $query->whereHas('itemCode', function($query) use($request){
                $query->where('position_id', $request->position_id);
            });
        })->when($request->item_code != null && isset($request->item_code) && $request->item_code != '', function($query) use($request){
            $query->whereHas('itemCode', function($query) use($request){
                $query->where('item_code', 'LIKE', '%'.$request->item_code .'%');
            });
        })->when($request->vacancy_start != null && isset($request->vacancy_start) && $request->vacancy_start != '', function($query) use($request){
            $query->whereDate('vacancy_start', $request->vacancy_start);
        })->when($request->current_incumbent_name != null && isset($request->current_incumbent_name) && $request->current_incumbent_name != '', function($query) use($request) {
            $query->whereHas('currentIncumbent', function($query) use($request) {
                $query->where('name', 'LIKE', '%'.$request->current_incumbent_name .'%');
            });
        })->when($request->reason_for_replacement_id != null && isset($request->reason_for_replacement_id) && $request->reason_for_replacement_id != '', function($query) use($request) {
            $query->where('reason_for_replacement_id', $request->reason_for_replacement_id);
        })->where('status', 1)->with(['firstThreeSuccessors.successor', 'currentIncumbent.salary.publicSalary', 'currentIncumbent.salary.privateSalary'])->paginate($paginate);
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\CriticalPosition  $criticalPosition
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        $criticalPosition = CriticalPosition::where('id', $request->critical_position_id)->with(['successors'])->first();

        $tree['id'] = $criticalPosition->id;
        $tree['name'] = $criticalPosition->currentIncumbent->name;
        $tree['item_code'] = $criticalPosition->itemCode->item_code;
        $tree['title'] = $criticalPosition->itemCode->position->title;
        $tree['date'] = $criticalPosition->vacancy_start;
        $tree['reason_for_replacement'] = $criticalPosition->reasonForReplacement->meaning;
        $tree['status'] = $criticalPosition->status;
        $tree['children'] = [];

        foreach ($criticalPosition->successors as $successor) {
            $tree['children'][] = [
                'id' => $successor->id,
                'successor_id' => $successor->successor->id,
                'name' => $successor->successor->name,
                'item_code' => $successor->successor->employeeMetaInfo->itemCode->item_code,
                'title' => $successor->successor->employeeMetaInfo->position->title,
                'number_of_promotions' => $successor->number_of_promotion,
                'performance' => $successor->successor->latest_evaluation ? $successor->successor->latest_evaluation->overall_rate : null
            ];
        }

        return $tree;
    }

}
