<?php

namespace Suiterus\Adg\Controllers;

use App\Enums\Log\BiometricsSetupLogType;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Setting;
use Suiterus\Adg\Models\SM\Sector;

/**
 * Controller for the general settings of the system
 * The settings are controlled by the developers through the functions in this controller.
 * 
 */

class SettingController extends Controller
{   
    use HasCustomLogs;
    private $db;
    private $sector;

    public function __construct(){
        $this->db = DB::connection('adg_db');
        $this->sector = Sector::where('status', 'active')->first();    
    }

    /**
     * Fetch setting record and value
     */
    public function fetch(Request $request) {
     
        $validate = Validator::make($request->all(), [
            'key'       => 'required|exists:adg_db.settings,key',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        };

        try {

            $setting = Setting::where('key', $request->key)->first();

            return response()->json([
                'data'  => $setting
            ]);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => ['There was a problem in saving the setting'],
                'message'   => $e->getMessage()
            ], 500);
        }
        
    }

    /**
     * Fetch all settings with key
     */
    public function all() {

        try {

            return response()->json([
                'data'  => Setting::all()
            ]);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Fetch all settings by selected module
     */
    public function fetchByModule(Request $request) {

        $validate = Validator::make($request->all(), [
            'module'    => 'required'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ]);
        }

        try {

            return response()->json([
                'data'  => Setting::where('module', $request->module)->get()
            ]);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Change Specific Setting Value
     */
    public function set(Request $request) {

        $validate = Validator::make($request->all(), [
            'key'       => 'required|exists:adg_db.settings,key',
            'value'     => 'required'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        };

        $this->db->beginTransaction();
        try {

            $setting = Setting::where('key', $request->key)->first();
            $setting->value = $request->value;
            $setting->save();

            $this->db->commit();
            return response()->json([
                'text'  => 'Settings saved'
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in saving the setting'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Save multiple settings by getting IDs of multiple
     */
    public function setMultiple(Request $request) {

        $validate = Validator::make($request->all(), [
            'settings.*.key'       => 'required|exists:adg_db.settings,key',
            'settings.*.value'     => 'required'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        };

        $this->db->beginTransaction();
        try {

            foreach($request->settings as $setting) {
                $record = Setting::where('key', $setting['key'])->first();
                $record->value = $setting['value'];
                $record->save();
            }

            
            $this->logCustomMessage(
                'set_auto_sync_settings',
                null,
                Auth::user()->name . ' Set auto sync settings',
                null,
                BiometricsSetupLogType::AUTOSYNC,
                new Activity()
            );

            $this->db->commit();
            return response()->json([
                'text'  => 'Settings has been saved.'
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Set setting as active/inactive
     */
    public function setActiveStatus(Request $request) {

        $validate = Validator::make($request->all(), [
            'key'       => 'required|exists:adg_db.settings,key',
            'status'    => 'required|in:1,2'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        };

        $this->db->beginTransaction();
        try {

            $setting = Setting::where('key', $request->key)->first();
            $setting->status = $request->status;
            $setting->save();

            $this->db->commit();
            return response()->json([
                'text'  => 'Settings saved'
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in saving the setting'],
                'message'   => $e->getMessage()
            ], 500);
        }
    }
}
