<?php

namespace Suiterus\Adg\Controllers\Salary;

use Exception;
use DateInterval;
use Carbon\Carbon;
use App\Models\User;
use App\Enums\Status;
use App\Models\PDSHistory;
use Illuminate\Support\Arr;
use Illuminate\Http\Request;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use Suiterus\Adg\Models\SM\Remark;
use Suiterus\Adg\Models\SM\Sector;
use Suiterus\Hrjp\Models\Position;
use App\Enums\Log\UserSalaryLogType;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\SM\SalaryGrade;
use Suiterus\Adg\Models\SM\EmployeeType;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Payroll\Allowance;
use Suiterus\Adg\Models\Salary\UserSalary;
use Suiterus\Hrjp\Models\Position_history;
use Suiterus\Adg\Models\Payroll\SpecialPay;
use Suiterus\Adg\Models\PDS\PDSPerPosition;
use Suiterus\Adg\Models\Salary\PublicSalary;
use Suiterus\Adg\Models\EMI\EmployeeMetaInfo;
use Suiterus\Adg\Models\Salary\PrivateSalary;
use Suiterus\Adg\Models\Salary\EmployeeAllowance;
use Suiterus\Adg\Models\SM\SalaryStandardization;
use Suiterus\Adg\Models\Salary\EmployeeSpecialPay;
use Suiterus\Adg\Models\ServiceRecord\ServiceRecord;
use Suiterus\Adg\Models\EMI\EmployeeExtraFieldColumn;

class UserSalaryController extends Controller
{
    use HasCustomLogs;

    public function create_user_salary(Request $req)
    {
        // Required info clarification

        $valid = Validator::make(
            $req->all(),
            [
                'selectedEmp' => 'required',
                'basicSalary' => 'required',
                'sssgID' => 'required_if:sector,==,public',
                'accountNumber' => 'required',
                'dateEffectivity' => 'required',
                'remarks' => 'nullable|string',
                'allowances' => 'nullable|array',
                'allowances.*.effectivity_date' => 'required',
                'specialPays' => 'nullable|array',
            ],
            [
                'allowances.*.effectivity_date.required' => 'Allowance effectivity date is required.'
            ]
        );

        if ($valid->fails()) {
            $errors = json_decode($valid->errors(), true);

            foreach ($errors as $key => $value) {
                $errorAttribute = explode('.', $key);
                if (count($errorAttribute) > 1) {
                    if ($errorAttribute[count($errorAttribute) - 1] == 'effectivity_date') {
                        $errors[$key] = $req->allowances[$errorAttribute[1]]['title'] . ' effectivity date is required.';
                    }
                }
            }

            return response()->json([
                'errors'    =>  $errors
            ], 400);
        }
        DB::beginTransaction();
        try {
            $totalAllowance = 0;
            $data = null;

            foreach ($req->allowances as $allowance) {

                if ($allowance['amount'] || $allowance['amount'] == '0') {
                    EmployeeAllowance::updateOrCreate(
                        [
                            'user_id' => $req->selectedEmp,
                            'allowance_id' => $allowance['id'],
                            'effectivity_date' => $allowance['effectivity_date']
                        ],
                        [
                            'amount' => $allowance['amount'],
                            'effectivity_date' => $allowance['effectivity_date'],
                            'updated_by' => Auth::id(),
                        ]
                    );
                }

                $name = strtolower(str_replace(' ', '_', $allowance['title']));
                $data[$name] = $allowance['amount'];
                $totalAllowance += $allowance['amount'];
            }

            foreach ($req->specialPays as $specialPay) {
                if ($specialPay['amount'] > 0) {
                    EmployeeSpecialPay::Create([
                        'user_id'  =>  $req->selectedEmp,
                        'special_pay_id' => $specialPay['id'],
                        'amount' => $specialPay['amount'],
                        'created_by' => Auth::id(),
                    ]);
                }

                $name = strtolower(str_replace(' ', '_', $specialPay['title']));
                $data[$name] = $specialPay['amount'];
            }

            $data = (object) $data;

            $userSalary = UserSalary::create([
                'user_id' => $req->selectedEmp,
                'account_number' => $req->accountNumber,
                'appointment_date' => now(),
                'date_of_effectivity' => $req->dateEffectivity,
                'status'    =>  0,
                'official_business_pay' => property_exists($data, 'official_business_pay') ? $data->official_business_pay : null,
                'e_cola' => property_exists($data, 'ecola') ? $data->ecola : null,
                'special_pay' => property_exists($data, 'special_pay') ? $data->special_pay : null,
                'load_allowance' => property_exists($data, 'load_allowance') ? $data->load_allowance : null,
                'transportation_allowance' => property_exists($data, 'transportation_allowance') ? $data->transportation_allowance : null,
                'branch_allowance' => property_exists($data, 'branch_allowance') ? $data->branch_allowance : null,
                'gate_allowance' => property_exists($data, 'gate_allowance') ? $data->gate_allowance : null,
                'rice_allowance' => property_exists($data, 'rice_allowance') ? $data->rice_allowance : null,
                'substinence_allowance' => property_exists($data, 'substinence_allowance') ? $data->substinence_allowance : null,
                'laundry_allowance' => property_exists($data, 'laundry_allowance') ? $data->laundry_allowance : null,
                'remarks' => $req->remarks,
                'created_by' => Auth::id()
            ]);

            $this->logCustomMessage(
                'create_user_salary',
                $userSalary,
                Auth::user()->name . ' created a new salary for ' . $userSalary->user->name,
                $userSalary,
                UserSalaryLogType::CREATE,
                new Activity()
            );

            $allAllowances = $totalAllowance;

            if ($req->sector == 'public') {
                $publicSalary = PublicSalary::create([
                    'user_salary_id' => $userSalary->id,
                    'salary_grade_id' => $req->sssgID,
                ]);
                $salary = $publicSalary->salaryGrade->value;
                $salary = str_replace(',', '', $salary);
                $annualSalary = $salary * 12;
            } else if ($req->sector == 'private') {
                PrivateSalary::create([
                    'user_salary_id' => $userSalary->id,
                    'salary' => $req->basicSalary
                ]);
                $annualSalary = $req->basicSalary * 12;
            }

            $getPosition = EmployeeMetaInfo::where('user_id', $req->selectedEmp)->first();

            $position_history = Position_history::with(['user' => function ($query) {
                $query->select('id', 'name')->without('permissions', 'roles', 'storage', 'employeeMetaInfo');
            }, 'user.user_employeeMetaInfo' => function ($query) {
                $query->select('id', 'user_id')->without(['storage', 'roles', 'permissions', 'department', 'branch', 'division', 'position', 'employeeMetaInfo']);
            }])->create([
                'user_id'               => $req->selectedEmp,
                'position_id'           => $getPosition->position_id,
                'appointment_date'      => now(),
                'date_of_effectivity'   => $req->dateEffectivity,
                'employment_status'     =>  0, //* currently employed
                'position_status'       => $req->position_status ? $req->position_status : 1,
                'created_by'            => Auth::id()
            ]);

            $employeeExtraFieldColumn = EmployeeExtraFieldColumn::whereHas('employeeExtraField', function ($query) use ($req) {
                $query->where('user_id', $req->selectedEmp);
            })->get();

            foreach ($employeeExtraFieldColumn as $pds) {
                PDSPerPosition::create([
                    'position_history_id'   =>    $position_history->id,
                    'eefc_id'               =>    $pds->id,
                    'field_name'           =>     $pds->field_name,
                    'field_value'           =>    $pds->field_value,
                    'created_by'            =>    Auth::id()
                ]);
            }

            $oldServiceRecord = ServiceRecord::where('user_id', $req->user_id)->whereNull('end_date'); //

            $endDate = date('Y-m-d', strtotime(now()));
            $oldServiceRecord->update([
                'end_date'  => $endDate
            ]);

            $remark = Remark::whereId(1)->first();

            ServiceRecord::create([
                'user_id'           =>  $req->selectedEmp,
                'start_date'        =>  $getPosition->date_hired,
                'position_id'       =>  $getPosition->position_id,
                'salary'            =>  $annualSalary,
                'allowance'         =>  $allAllowances,
                'employee_type_id'  =>  $getPosition->employee_type,
                'division_id'       =>  $getPosition->division_id,
                'remark_id'         =>  $remark->id,  //* Original or Re-employment
                'created_by'        =>  Auth::id()
            ]);

            DB::commit();
            return response()->json([
                'text'  =>  'An employee salary was created successfully.',
                'data' => $req->all()
            ]);
        } catch (Exception $e) {

            DB::rollback();
            return response()->json([
                'errors'    =>  ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01'],
                'msg'   =>  $e->getMessage(),
                'line'  => $e->getLine()
            ], 500);
        }
    }

    public function batch_create_user_salary(Request $req)
    {
        $parsedErrors = [];
        foreach ($req->salaries as $empSalary) {
            $valid = Validator::make(
                $empSalary,
                [
                    'allowances' => 'nullable|array',
                    'allowances.*.effectivity_date' => 'required',
                ],
                [
                    'allowances.*.effectivity_date.required' => 'Allowance effectivity date is required.'
                ]
            );

            if ($valid->fails()) {
                $errors = json_decode($valid->errors(), true);

                foreach ($errors as $key => $value) {
                    $errorAttribute = explode('.', $key);
                    if (count($errorAttribute) > 1) {
                        if ($errorAttribute[count($errorAttribute) - 1] == 'effectivity_date') {
                            $message = User::find($empSalary['selectedEmp'])->name . ": " . $empSalary['allowances'][$errorAttribute[1]]['title'] . ' effectivity date is required.';
                            if (!in_array($message, $parsedErrors)) {
                                $parsedErrors[] = $errors[$key] = $message;
                            }
                        }
                    }
                }
            }
        }

        return response()->json([
            'errors'    =>  $parsedErrors
        ], 400);

        DB::beginTransaction();
        try {
            $activeSector = Sector::where('status', 'active')->first();
            $sector = $activeSector->name;

            foreach ($req->salaries as $empSalary) {

                $totalAllowance = 0;
                $data = [];

                foreach ($empSalary['allowances'] as $allowance) {

                    if ($allowance['amount'] || $allowance['amount'] == '0') {
                        EmployeeAllowance::updateOrCreate(
                            [
                                'user_id'  =>  $empSalary['selectedEmp'],
                                'allowance_id' => $allowance['id'],
                                'effectivity_date' => $allowance['effectivity_date']
                            ],
                            [
                                'amount' => $allowance['amount'],
                                'effectivity_date' => $allowance['effectivity_date'],
                                'updated_by' => Auth::id(),
                            ]
                        );
                    }

                    $name = strtolower(str_replace(' ', '_', $allowance['title']));
                    $data[$name] = $allowance['amount'];
                    $totalAllowance += $allowance['amount'];
                }

                foreach ($empSalary['specialPays'] as $specialPay) {
                    if ($specialPay['amount'] > 0) {
                        EmployeeSpecialPay::Create([
                            'user_id'  =>  $empSalary['selectedEmp'],
                            'special_pay_id' => $specialPay['id'],
                            'amount' => $specialPay['amount'],
                            'created_by' => Auth::id(),
                        ]);
                    }

                    $name = strtolower(str_replace(' ', '_', $specialPay['title']));
                    $data[$name] = $specialPay['amount'];
                }

                $userSalary = UserSalary::create([
                    'user_id' => $empSalary['selectedEmp'],
                    'account_number' => $empSalary['accountNumber'],
                    'appointment_date' => now(),
                    'date_of_effectivity' => $empSalary['dateEffectivity'],
                    'status'    =>  0,
                    'official_business_pay' => array_key_exists('official_business_pay', $data) ? $data['official_business_pay'] : null,
                    'e_cola' => array_key_exists('ecola', $data) ? $data['ecola'] : null,
                    'special_pay' => array_key_exists('special_pay', $data) ? $data['special_pay'] : null,
                    'load_allowance' => array_key_exists('load_allowance', $data) ? $data['load_allowance'] : null,
                    'transportation_allowance' => array_key_exists('transportation_allowance', $data) ? $data['transportation_allowance'] : null,
                    'branch_allowance' => array_key_exists('branch_allowance', $data) ? $data['branch_allowance'] : null,
                    'gate_allowance' => array_key_exists('gate_allowance', $data) ? $data['gate_allowance'] : null,
                    'rice_allowance' => array_key_exists('rice_allowance', $data) ? $data['rice_allowance'] : null,
                    'substinence_allowance' => array_key_exists('substinence_allowance', $data) ? $data['substinence_allowance'] : null,
                    'laundry_allowance' => array_key_exists('laundry_allowance', $data) ? $data['laundry_allowance'] : null,
                    'remarks' => $empSalary['remarks'],
                    'created_by' => Auth::id()
                ]);

                $this->logCustomMessage(
                    'create_bulk_user_salary',
                    $userSalary,
                    Auth::user()->name . ' created a bulk salary for employees',
                    $userSalary,
                    UserSalaryLogType::CREATE_BULK,
                    new Activity()
                );
                $allAllowances = $totalAllowance;

                if ($sector == 'public') {
                    $publicSalary = PublicSalary::create([
                        'user_salary_id' => $userSalary->id,
                        'salary_grade_id' => $empSalary['sssgID'],
                    ]);
                    $salary = $publicSalary->salaryGrade->value;
                    $salary = str_replace(',', '', $salary);
                    $annualSalary = $salary * 12;
                } else if ($sector == 'private') {
                    PrivateSalary::create([
                        'user_salary_id' => $userSalary->id,
                        'salary' => $empSalary['basicSalary']
                    ]);
                    $annualSalary = $empSalary['basicSalary'] * 12;
                }

                $getPosition = EmployeeMetaInfo::where('user_id', $empSalary['selectedEmp'])->first();

                $position_history = Position_history::with(['user' => function ($query) {
                    $query->select('id', 'name')->without('permissions', 'roles', 'storage', 'employeeMetaInfo');
                }, 'user.user_employeeMetaInfo' => function ($query) {
                    $query->select('id', 'user_id')->without(['storage', 'roles', 'permissions', 'department', 'branch', 'division', 'position', 'employeeMetaInfo']);
                }])->create([
                    'user_id'               => $empSalary['selectedEmp'],
                    'position_id'           => $getPosition->position_id,
                    'appointment_date'      => now(),
                    'date_of_effectivity'   => $empSalary['dateEffectivity'],
                    'employment_status'     => 0, //* currently employed
                    'position_status'       => 1, //* current position
                    'created_by'            => Auth::id()
                ]);

                $employeeExtraFieldColumn = EmployeeExtraFieldColumn::whereHas('employeeExtraField', function ($query) use ($empSalary) {
                    $query->where('user_id', $empSalary['selectedEmp']);
                })->get();

                foreach ($employeeExtraFieldColumn as $pds) {
                    PDSPerPosition::create([
                        'position_history_id'   =>    $position_history->id,
                        'eefc_id'               =>    $pds->id,
                        'field_name'           =>     $pds->field_name,
                        'field_value'           =>    $pds->field_value,
                        'created_by'            =>    Auth::id()
                    ]);
                }

                if ($allAllowances === null) {
                    $allAllowances = 0;
                } else {
                    $allAllowances = $totalAllowance;
                }


                $oldServiceRecord = ServiceRecord::where('user_id', $empSalary['selectedEmp'])->whereNull('end_date');;

                $endDate = date('Y-m-d', strtotime(now()));
                $oldServiceRecord->update([
                    'end_date'  => $endDate
                ]);

                $remark = Remark::whereId(1)->first();

                ServiceRecord::create([
                    'user_id'           =>  $empSalary['selectedEmp'],
                    'start_date'        =>  $getPosition->date_hired,
                    'position_id'       =>  $getPosition->position_id,
                    'salary'            =>  $annualSalary,
                    'allowance'         =>  $allAllowances,
                    'employee_type_id'  =>  $getPosition->employee_type,
                    'division_id'       =>  $getPosition->division_id,
                    'remark_id'         =>  $remark->id,  //* Original or Re-employment
                    'created_by'        =>  Auth::id()
                ]);
            }

            DB::commit();
            return response()->json([
                'text'  =>  'Batch of employee salary was created successfully.',
                'data' => $req->all()
            ]);
        } catch (Exception $e) {

            DB::rollback();
            return response()->json([
                'errors'    =>  ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01'],
                'msg'   =>  $e->getMessage(),
                'line'  => $e->getLine()
            ], 500);
        }
    }

    public function fetch_employee_without_salary()
    {
        return User::select('id', 'name', 'email')->where('status', 1)->has('employeeMetaInfo')->doesntHave('salary')->without([
            'currentRole',
            'roles',
            'permissions',
            'storage',
            'supervisor',
            'user_supervisor',
            'exitInterview',
            'userProfilePicture',
            'profileBasicInfo'
        ])->with([
            'employeeMetaInfo' => function ($query) {
                $query->without([
                    'corporation',
                    'branch',
                    'division',
                    'department',
                    'position',
                    'itemCode'
                ]);
            }
        ])->get();
    }

    public function fetch_employee_with_salary()
    {
        return response()->json([
            'data' => UserSalary::where('status', 0)->whereHas('user', function ($query) {
                $query->where('status', 1)->has('employeeMetaInfo');
            })->with([
                'user' => function ($query) {
                    $query->select('id', 'name', 'email')->without([
                        'currentRole',
                        'roles',
                        'permissions',
                        'storage',
                        'supervisor',
                        'user_supervisor',
                        'exitInterview',
                        'userProfilePicture',
                        'profileBasicInfo'
                    ]);
                },
                'user.employeeMetaInfo' => function ($query) {
                    $query->select('id', 'user_id', 'employee_id')->without([
                        'corporation',
                        'branch',
                        'division',
                        'department',
                        'employeeType',
                        'position',
                        'itemCode'
                    ]);
                }
            ])->get()
        ]);
    }

    public function search_employee_with_salary(Request $req)
    {
        $paginate = $req->page_count ? intval($req->page_count) : env('DEFAULT_PAGECOUNT');

        if ($req->sector != null && $req->sector != '' && isset($req->sector)) {
            return UserSalary::where('status', 0)->whereHas('user', function ($query) use ($req) {
                $query->where('status', 1);
                $this->whereHasUser($query, $req);
            })->with([$req->sector . 'Salary', 'user.allowances', 'user.special_pays', 'user.longevity_pay' => function ($query) {
                $query->where('status', Status::ACTIVE)->orderBy('date', 'desc')->first();
            }])->paginate($paginate);
        } else {
            return UserSalary::where('status', 0)->whereHas('user', function ($query) use ($req) {
                $query->where('status', 1);
                $this->whereHasUser($query, $req);
            })->paginate($paginate);
        }
    }

    public function whereHasUser($query, $req)
    {

        $query->when($req->employeeName != null && $req->employeeName != '', function ($query) use ($req) {
            $query->where('name', 'LIKE', '%' . $req->employeeName . '%');
        });

        $query->when($req->employeeID != null && $req->employeeID != '', function ($query) use ($req) {
            $query->whereHas('employeeMetaInfo', function ($query) use ($req) {

                if (count($req->employeeID) > 0) {
                    $query->whereIn('employee_id', $req->employeeID);
                } else {
                    $query->where('employee_id', $req->employeeID);
                }
            });
        });

        $query->when($req->position != null && $req->position != '', function ($query) use ($req) {
            $query->whereHas('employeeMetaInfo', function ($query) use ($req) {
                $query->where('position_id', $req->position);
            });
        });

        $query->when($req->salary_value != null && $req->salary_value != '', function ($query) use ($req) {
            $query->whereHas('salary', function ($query) use ($req) {
                $query->whereHas('publicSalary', function ($query) use ($req) {
                    $query->whereHas('salaryGrade', function ($query) use ($req) {
                        $query->where('value', $req->salary_value);
                    });
                });
            });
        });

        $query->when($req->salary_step != null && $req->salary_step != '', function ($query) use ($req) {
            $query->whereHas('salary', function ($query) use ($req) {
                $query->whereHas('publicSalary', function ($query) use ($req) {
                    $query->whereHas('salaryGrade', function ($query) use ($req) {
                        $query->where('step', $req->salary_step);
                    });
                });
            });
        });

        $query->when($req->basic_salary != null && $req->basic_salary != '', function ($query) use ($req) {
            $query->whereHas('salary', function ($query) use ($req) {
                $query->whereHas('privateSalary', function ($query) use ($req) {
                    $query->where('salary', $req->basic_salary);
                });
            });
        });
    }

    public function update_user_salary(Request $req)
    {

        // ** Validator
        $valid = Validator::make($req->all(), [
            'id' => 'integer|required',
            'user_id' => 'required',
            'basic_salary' => 'required',
            'sssgID' => 'required_if:sector,==,public',
            'allowances' => 'nullable|array',
            'specialPays' => 'nullable|array',
            'appointment_date' => 'nullable',
            'date_of_effectivity' => 'required',
            'remarks' => 'nullable|string',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }

        DB::beginTransaction();

        try {
            $totalAllowance = 0;
            foreach ($req->allowances as $allowance) {
                if (array_key_exists('amount', $allowance)) {
                    if ($allowance['amount'] > 0) {
                        EmployeeAllowance::updateOrCreate(
                            [
                                'user_id' => $req->user_id,
                                'allowance_id' => $allowance['id']
                            ],
                            [
                                'amount' => $allowance['amount'],
                                'updated_by' => Auth::id(),
                            ]
                        );
                    }
                    $name = strtolower(str_replace(' ', '_', $allowance['title']));
                    $data[$name] = $allowance['amount'];
                    $totalAllowance += $allowance['amount'];
                }
            }

            foreach ($req->specialPays as $specialPay) {
                if (array_key_exists('amount', $specialPay)) {
                    if ($specialPay['amount'] > 0) {
                        EmployeeSpecialPay::updateOrCreate(
                            [
                                'user_id' => $req->user_id,
                                'special_pay_id' => $specialPay['id']
                            ],
                            [
                                'amount' => $specialPay['amount'],
                                'updated_by' => Auth::id(),
                            ]
                        );
                    }
                    $name = strtolower(str_replace(' ', '_', $specialPay['title']));
                    $data[$name] = $specialPay['amount'];
                }
            }

            $oldRecord = UserSalary::whereId($req->id)->first();

            $data = (object) $data;
            UserSalary::where('id', $req->id)->update([

                'account_number' => $req->account_number,
                'appointment_date' => now(),
                'date_of_effectivity' => $req->date_of_effectivity,
                'official_business_pay' =>  property_exists($data, 'official_business_pay') ? $data->official_business_pay : null,
                'e_cola' => property_exists($data, 'ecola') ? $data->ecola : null,
                'special_pay' => property_exists($data, 'special_pay') ? $data->special_pay : null,
                'load_allowance' => property_exists($data, 'load_allowance') ? $data->load_allowance : null,
                'transportation_allowance' => property_exists($data, 'transportation_allowance') ? $data->transportation_allowance : null,
                'branch_allowance' => property_exists($data, 'branch_allowance') ? $data->branch_allowance : null,
                'gate_allowance' => property_exists($data, 'gate_allowance') ? $data->gate_allowance : null,
                'rice_allowance' => property_exists($data, 'rice_allowance') ? $data->rice_allowance : null,
                'substinence_allowance' => property_exists($data, 'substinence_allowance') ? $data->substinence_allowance : null,
                'laundry_allowance' => property_exists($data, 'laundry_allowance') ? $data->laundry_allowance : null,
                'appointment_date' => $req->appointment_date,
                'remarks' => $req->remarks,
                'updated_by' => Auth::id()

            ]);

            $newRecord = UserSalary::whereId($req->id)->first();

            $newRecord->old = collect($oldRecord);
            $newRecord->attributes = collect($newRecord);

            $this->logCustomMessage(
                'update_all_user_salary',
                $newRecord,
                Auth::user()->name . ' updated the salary of ' . $newRecord->user->name,
                $newRecord,
                UserSalaryLogType::UPDATE_ALL,
                new Activity()
            );

            $allAllowances = $totalAllowance;

            // **  Condition for 'sector'; for declaring the value of 'status' in position history
            $status = 1; // ** for 'promoted'

            if ($req->sector == 'public') {

                $currentSalary = PublicSalary::where('id', $req->public_salary_id)->first();
                $newSalary = SalaryGrade::where('id', $req->sssgID)->first();

                if ($currentSalary->salaryGrade->salary_grade > $newSalary->salary_grade) {

                    $status = 2; // ** for 'demoted'

                }

                PublicSalary::where('id', $req->public_salary_id)->update(
                    [
                        'salary_grade_id' => $req->sssgID
                    ]
                );

                $salary = $newSalary->value;
                $salary = str_replace(',', '', $salary);
                $annualSalary = $salary * 12;
            } else if ($req->sector == 'private') {

                $currentPosition = EmployeeMetaInfo::where('user_id', $req->user_id)->first();

                if ($currentPosition->position_id != $req->position_id) {

                    $status = $req->status; // ** for 'demoted'

                }

                PrivateSalary::where('id', $req->private_salary_id)->update([
                    'salary' => $req->basic_salary
                ]);
                $annualSalary = $req->basic_salary * 12;
            }

            // ** When the user update the fields in update salary, the position_id in employeeMetaInfo change at the same time
            EmployeeMetaInfo::where('user_id', $req->user_id)->update([
                'position_id' => $req->position_id,
                'item_code_id' => $req->item_code,
                'modified_by' => Auth::id()
            ]);

            // ** Condition for declaring the new value of 'position_status' from the previous record in position history
            $position_status = 1; // ** for 'current position'
            $currentAssumptionDuty = Position_history::where('user_id', $req->user_id)->orderBy('id', 'desc')->first();

            $date1 = Carbon::createFromFormat('Y-m-d', strval($req->date_of_effectivity)); //** new Asssumption to Duty
            $date2 = Carbon::createFromFormat('Y-m-d', strval($currentAssumptionDuty->date_of_effectivity)); //** current Asssumption to Duty

            if ($date1->gt($date2)) {

                $position_status = 2; // ** for 'previous position'
                Position_history::where('user_id', $req->user_id)->orderBy('id', 'desc')->first()->update([
                    'position_status'   =>  $position_status,
                    'updated_by' => Auth::id()
                ]);
            }

            // ** Create function for new record of position history when the user update the position and salary
            $position_history = Position_history::where(['user_id', $req->user_id])->create([

                'user_id'               => $req->user_id,
                'position_id'           => $req->position_id,
                'appointment_date'      => now(),
                'date_of_effectivity'   => $req->date_of_effectivity,
                'status'                => $status,
                'employment_status'     =>  0, //* currently employed
                'created_by'            => Auth::id()

            ]);

            $employeeExtraFieldColumn = EmployeeExtraFieldColumn::whereHas('employeeExtraField', function ($query) use ($req) {
                $query->where('user_id', $req->user_id);
            })->get();

            foreach ($employeeExtraFieldColumn as $pds) {
                PDSPerPosition::create([
                    'position_history_id'   =>    $position_history->id,
                    'eefc_id'               =>    $pds->id,
                    'field_name'            =>    $pds->field_name,
                    'field_value'           =>    $pds->field_value,
                    'created_by'            =>    Auth::id()
                ]);
            }

            if ($status == 1) {

                $oldServiceRecord = ServiceRecord::where('user_id', $req->user_id)->whereNull('end_date');
                $endDate = date('Y-m-d', (strtotime('-1 day', strtotime($req->date_of_effectivity))));
                $oldServiceRecord->update([
                    'end_date'  => $endDate
                ]);


                $employee = EmployeeMetaInfo::where('user_id', $req->user_id)->first();

                $remark = Remark::whereId(2)->first();
                ServiceRecord::create([
                    'user_id'           =>  $req->user_id,
                    'start_date'        =>  $req->date_of_effectivity,
                    'position_id'       =>  $req->position_id,
                    'salary'            =>  $annualSalary,
                    'allowance'         =>  $allAllowances,
                    'employee_type_id'  =>  $employee->employee_type,
                    'division_id'       =>  $employee->division_id,
                    'remark_id'         =>  $remark->id,  //* Promoted
                    'created_by'        =>  Auth::id(),

                ]);
            }

            DB::commit();
            return response()->json([
                'text'  => 'User salary updated successfully.',
                'data'  => $req->all()
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01'],
                'msg'   =>  $e->getMessage(),
                'line' => $e->getLine()
            ], 500);
        }
    }

    // ** Update function when the user only update the 'account_number' or/with 'remarks'
    public function update_request_user_salary(Request $req)
    {

        // ** Validator
        $valid = Validator::make(
            $req->all(),
            [
                'id' => 'integer|required',
                'user_id' => 'required',
                'account_number' => 'required',
                'basic_salary' => 'required',
                'allowances' => 'nullable|array',
                'allowances.*.effectivity_date' => 'required',
                'specialPays' => 'nullable|array',
                'remarks' => 'nullable|string'
            ],
            [
                'allowances.*.effectivity_date.required' => 'Allowance effectivity date is required.'
            ]
        );

        if ($valid->fails()) {
            $errors = json_decode($valid->errors(), true);

            foreach ($errors as $key => $value) {
                $errorAttribute = explode('.', $key);
                if (count($errorAttribute) > 1) {
                    if ($errorAttribute[count($errorAttribute) - 1] == 'effectivity_date') {
                        $errors[$key] = $req->allowances[$errorAttribute[1]]['title'] . ' effectivity date is required.';
                    }
                }
            }

            return response()->json([
                'errors'    =>  $errors
            ], 400);
        }

        DB::beginTransaction();

        try {
            $empAllowance = EmployeeAllowance::where('user_id', $req->user_id)->get();

            $oldRecord = [];
            foreach ($empAllowance as $records) {
                $oldRecord[] = $records['amount'];
            }

            $totalAllowance = 0;

            foreach ($req->allowances as $allowance) {
                if ($allowance['amount'] || $allowance['amount'] == '0') {
                    EmployeeAllowance::updateOrCreate(
                        [
                            'user_id' => $req->user_id,
                            'allowance_id' => $allowance['id'],
                            'effectivity_date' => $allowance['effectivity_date']
                        ],
                        [
                            'amount' => $allowance['amount'],
                            'effectivity_date' => $allowance['effectivity_date'],
                            'updated_by' => Auth::id(),
                        ]
                    );
                }
                $name = strtolower(str_replace(' ', '_', $allowance['title']));
                $data[$name] = $allowance['amount'];
                $totalAllowance += $allowance['amount'];
            }

            foreach ($req->specialPays as $specialPay) {
                if ($specialPay['amount'] > 0) {
                    EmployeeSpecialPay::updateOrCreate(
                        [
                            'user_id' => $req->user_id,
                            'special_pay_id' => $specialPay['id']
                        ],
                        [
                            'amount' => $specialPay['amount'],
                            'updated_by' => Auth::id(),
                        ]
                    );
                }
                $name = strtolower(str_replace(' ', '_', $specialPay['title']));
                $data[$name] = $specialPay['amount'];
            }

            $userSalary = UserSalary::whereId($req->id)->first();

            $data = (object) $data;
            UserSalary::where('id', $req->id)->update([
                'account_number' => $req->account_number,
                'official_business_pay' => property_exists($data, 'official_business_pay') ? $data->official_business_pay : null,
                'e_cola' => property_exists($data, 'ecola') ? $data->ecola : null,
                'special_pay' => property_exists($data, 'special_pay') ? $data->special_pay : null,
                'load_allowance' => property_exists($data, 'load_allowance') ? $data->load_allowance : null,
                'transportation_allowance' => property_exists($data, 'transportation_allowance') ? $data->transportation_allowance : null,
                'branch_allowance' => property_exists($data, 'branch_allowance') ? $data->branch_allowance : null,
                'gate_allowance' => property_exists($data, 'gate_allowance') ? $data->gate_allowance : null,
                'rice_allowance' => property_exists($data, 'rice_allowance') ? $data->rice_allowance : null,
                'substinence_allowance' => property_exists($data, 'substinence_allowance') ? $data->substinence_allowance : null,
                'laundry_allowance' => property_exists($data, 'laundry_allowance') ? $data->laundry_allowance : null,
                'date_of_effectivity'   =>  $req->date_of_effectivity,
                'remarks' => $req->remarks,
                'updated_by' => Auth::id()
            ]);

            $allAllowances = $totalAllowance;

            $newRecord = UserSalary::whereId($req->id)->first();

            $newRecord->old = collect($userSalary);
            $newRecord->attributes = collect($newRecord);

            $this->logCustomMessage(
                'update_user_salary',
                $newRecord,
                Auth::user()->name . ' updated the salary of ' . $newRecord->user->name,
                $newRecord,
                UserSalaryLogType::UPDATE,
                new Activity()
            );


            if ($req->sector == 'public') {
                $newSalary = SalaryGrade::where('id', $req->sssgID)->first();
                $salary = $newSalary->value;
                $salary = str_replace(',', '', $salary);
                $annualSalary = $salary * 12;

                $employee = EmployeeMetaInfo::where('user_id', $userSalary->user_id)->first();

                $publicSalary = PublicSalary::whereHas('salaryGrade')->where('user_salary_id', $userSalary->id)->first();

                if ($publicSalary->salary_grade_id != $req->sssgID && $oldRecord != Arr::pluck($req->allowances, 'amount')) {
                    $publicSalary->update([
                        'salary_grade_id' => $req->sssgID
                    ]);

                    $oldServiceRecord = ServiceRecord::where('user_id', $employee->user_id)->whereNull('end_date');
                    $endDate = date('Y-m-d', strtotime(now()));
                    $oldServiceRecord->update([
                        'end_date'  => $endDate
                    ]);


                    $remark20 = Remark::whereId(20)->first();
                    ServiceRecord::create([
                        'user_id'           =>  $req->user_id,
                        'start_date'        =>  $req->date_of_effectivity,
                        'position_id'       =>  $employee->position_id,
                        'salary'            =>  $annualSalary,
                        'allowance'         =>  $allAllowances,
                        'employee_type_id'  =>  $employee->employee_type,
                        'division_id'       =>  $employee->division_id,
                        'remark_id'         =>  $remark20->id,  //* Step Increment
                        'created_by'        =>  Auth::id()
                    ]);

                    $remark11 = Remark::whereId(11)->first();
                    ServiceRecord::create([
                        'user_id'           =>  $req->user_id,
                        'start_date'        =>  now(),
                        'position_id'       =>  $employee->position_id,
                        'salary'            =>  $annualSalary,
                        'allowance'         =>  $allAllowances,
                        'employee_type_id'  =>  $employee->employee_type,
                        'division_id'       =>  $employee->division_id,
                        'remark_id'         =>  $remark11->id,  //* Allowance Adjustment
                        'created_by'        =>  Auth::id()
                    ]);
                } else if ($publicSalary->salary_grade_id != $req->sssgID) {
                    $publicSalary->update([
                        'salary_grade_id' => $req->sssgID
                    ]);

                    $oldServiceRecord = ServiceRecord::where('user_id', $employee->user_id)->whereNull('end_date');
                    $endDate = date('Y-m-d', strtotime(now()));
                    $oldServiceRecord->update([
                        'end_date'  => $endDate
                    ]);

                    $remark = Remark::whereId(20)->first();
                    ServiceRecord::create([
                        'user_id'           =>  $req->user_id,
                        'start_date'        =>  $req->date_of_effectivity,
                        'position_id'       =>  $employee->position_id,
                        'salary'            =>  $annualSalary,
                        'allowance'         =>  $allAllowances,
                        'employee_type_id'  =>  $employee->employee_type,
                        'division_id'       =>  $employee->division_id,
                        'remark_id'         =>  $remark->id,  //* Step Increment
                        'created_by'        =>  Auth::id()
                    ]);
                } else if ($oldRecord != Arr::pluck($req->allowances, 'amount')) {
                    $oldServiceRecord = ServiceRecord::where('user_id', $employee->user_id)->whereNull('end_date');
                    $endDate = date('Y-m-d', strtotime(now()));
                    $oldServiceRecord->update([
                        'end_date'  => $endDate
                    ]);


                    $remark = Remark::whereId(11)->first();
                    ServiceRecord::create([
                        'user_id'           =>  $req->user_id,
                        'start_date'        =>  now(),
                        'position_id'       =>  $employee->position_id,
                        'salary'            =>  $annualSalary,
                        'allowance'         =>  $allAllowances,
                        'employee_type_id'  =>  $employee->employee_type,
                        'division_id'       =>  $employee->division_id,
                        'remark_id'         =>  $remark->id,  //* Allowance Adjustment
                        'created_by'        =>  Auth::id()
                    ]);
                }
            } else {
                $employee = EmployeeMetaInfo::where('user_id', $userSalary->user_id)->first();
                $oldServiceRecord = ServiceRecord::where('user_id', $employee->user_id)->whereNull('end_date');
                $privateSalary = PrivateSalary::whereHas('userSalary', function ($query) use ($employee) {
                    $query->where('user_id', $employee->user_id);
                })->first();
                $oldSalary = $privateSalary->salary * 12;

                $newSalary = PrivateSalary::whereHas('userSalary', function ($query) use ($req) {
                    $query->where('user_id', $req->user_id);
                })->first();

                $salary = $newSalary->salary;
                $annualSalary = $salary * 12;

                PrivateSalary::where('id', $req->private_salary_id)->update([
                    'salary' => $req->basic_salary
                ]);
                $newUpdateSalary = $req->basic_salary * 12;

                if ($oldRecord != Arr::pluck($req->allowances, 'amount') && $annualSalary != $newUpdateSalary) {
                    $endDate = date('Y-m-d', strtotime(now()));
                    $oldServiceRecord->update([
                        'end_date'  => $endDate
                    ]);


                    $remark11 = Remark::whereId(11)->first();
                    ServiceRecord::create([
                        'user_id'           =>  $req->user_id,
                        'start_date'        =>  now(),
                        'position_id'       =>  $employee->position_id,
                        'salary'            =>  $annualSalary,
                        'allowance'         =>  $allAllowances,
                        'employee_type_id'  =>  $employee->employee_type,
                        'division_id'       =>  $employee->division_id,
                        'remark_id'         =>  $remark11->id,  //* Allowance Adjustment
                        'created_by'        =>  Auth::id()
                    ]);

                    $remark10 = Remark::whereId(10)->first();
                    ServiceRecord::create([
                        'user_id'           =>  $req->user_id,
                        'start_date'        =>  $req->date_of_effectivity,
                        'position_id'       =>  $employee->position_id,
                        'salary'            =>  $newUpdateSalary,
                        'allowance'         =>  $allAllowances,
                        'employee_type_id'  =>  $employee->employee_type,
                        'division_id'       =>  $employee->division_id,
                        'remark_id'         =>  $remark10->id,  //* Salary Adjustment
                        'created_by'        =>  Auth::id()
                    ]);
                } else if ($oldRecord != Arr::pluck($req->allowances, 'amount')) {
                    $endDate = date('Y-m-d', strtotime(now()));
                    $oldServiceRecord->update([
                        'end_date'  => $endDate
                    ]);

                    $remark = Remark::whereId(11)->first();
                    ServiceRecord::create([
                        'user_id'           =>  $req->user_id,
                        'start_date'        =>  now(),
                        'position_id'       =>  $employee->position_id,
                        'salary'            =>  $annualSalary,
                        'allowance'         =>  $allAllowances,
                        'employee_type_id'  =>  $employee->employee_type,
                        'division_id'       =>  $employee->division_id,
                        'remark_id'         =>  $remark->id,  //* Allowance Adjustment
                        'created_by'        =>  Auth::id()
                    ]);
                } else if ($annualSalary != $newUpdateSalary) {

                    $endDate = date('Y-m-d', strtotime(now()));
                    $oldServiceRecord->update([
                        'end_date'  => $endDate
                    ]);


                    $remark = Remark::whereId(10)->first();
                    ServiceRecord::create([
                        'user_id'           =>  $req->user_id,
                        'start_date'        =>  $req->date_of_effectivity,
                        'position_id'       =>  $employee->position_id,
                        'salary'            =>  $newUpdateSalary,
                        'allowance'         =>  $allAllowances,
                        'employee_type_id'  =>  $employee->employee_type,
                        'division_id'       =>  $employee->division_id,
                        'remark_id'         =>  $remark->id,  //* Salary Adjustment
                        'created_by'        =>  Auth::id()
                    ]);
                }

                Position_history::where('user_id', $req->user_id)->orderBy('id', 'desc')->first()->update([
                    'date_of_effectivity'   =>  $req->date_of_effectivity,
                    'updated_by'            =>  Auth::id()
                ]);
            }

            EmployeeMetaInfo::where('user_id', $req->user_id)->update([
                'item_code_id' => $req->item_code,
                'modified_by' => Auth::id()
            ]);

            DB::commit();
            return response()->json([
                'text'  => 'User salary updated successfully.',
                'data'  => $req->all(),

            ]);
        } catch (Exception $e) {

            DB::rollback();
            return response()->json([
                'errors'    =>  ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01'],
                'msg'       =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function fetch_salary_step()
    {

        $salary = SalaryGrade::select('step', 'value')->groupBy('step')->get();

        return response()->json([
            'text' => 'fetch successful.',
            'data' => $salary
        ]);
    }

    public function fetch_salary_value(Request $request)
    {

        if ($request->emp_type_title == 'Contractual') {
            $salary = SalaryGrade::whereHas('salary_standardization', function ($query) {
                $query->whereHas('employeeType', function ($query) {
                    $query->where('title', 'Contractual');
                })->where('active_tranche', 1);
            })->select('step', 'value')->groupBy('value')->get();
        } else {
            $salary = SalaryGrade::whereHas('salary_standardization', function ($query) {
                $query->whereHas('employeeType', function ($query) {
                    $query->where('title', 'Regular');
                })->where('active_tranche', 1);
            })->select('step', 'value')->groupBy('value')->get();
        }


        return response()->json([
            'text' => 'fetch successful.',
            'data' => $salary
        ]);
    }

    public function fetch_private_salary()
    {

        $salary = PrivateSalary::select('salary')->groupBy('salary')->get();

        return response()->json([
            'text' => 'fetch successful.',
            'data' => $salary
        ]);
    }


    public function fetch_positions(Request $req)
    {

        $position = Position::whereDoesntHave('employeeMetaInfo', function ($q) use ($req) {
            $q->where('user_id', $req->user_id);
        })->get();

        return response()->json([
            'text' => 'fetch successful.',
            'data' => $position,
        ]);
    }

    public function salary_for_reemployment(Request $req)
    {
        // ** Validator
        $valid = Validator::make($req->all(), [
            'selectedEmp' => 'required',
            'basicSalary' => 'required',
            'sssgID' => 'required_if:sector,==,public',
            'accountNumber' => 'required',
            'dateEffectivity' => 'required',
            'remarks' => 'nullable|string',
            'allowances' => 'nullable|array',
            'specialPays' => 'nullable|array',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $totalAllowance = 0;
            foreach ($req->allowances as $allowance) {
                if ($allowance['amount'] > 0) {
                    EmployeeAllowance::updateOrCreate(
                        [
                            'user_id' => $req->selectedEmp,
                            'allowance_id' => $allowance['id']
                        ],
                        [
                            'amount' => $allowance['amount'],
                            'updated_by' => Auth::id(),
                        ]
                    );
                }
                $name = strtolower(str_replace(' ', '_', $allowance['title']));
                $data[$name] = $allowance['amount'];
                $totalAllowance += $allowance['amount'];
            }

            foreach ($req->specialPays as $specialPay) {
                if ($specialPay['amount'] > 0) {
                    EmployeeSpecialPay::updateOrCreate(
                        [
                            'user_id' => $req->selectedEmp,
                            'special_pay_id' => $specialPay['id']
                        ],
                        [
                            'amount' => $specialPay['amount'],
                            'updated_by' => Auth::id(),
                        ]
                    );
                }
                $name = strtolower(str_replace(' ', '_', $specialPay['title']));
                $data[$name] = $specialPay['amount'];
            }

            $oldRecord = UserSalary::whereId($req->id)->first();

            $data = (object) $data;
            UserSalary::where('id', $req->id)->update([

                'account_number'            => $req->accountNumber,
                'appointment_date'          => now(),
                'date_of_effectivity'       => $req->dateEffectivity,
                'status'                    =>  0,
                'official_business_pay'     =>  property_exists($data, 'official_business_pay') ? $data->official_business_pay : null,
                'e_cola'                    =>  property_exists($data, 'ecola') ? $data->ecola : null,
                'special_pay'               =>  property_exists($data, 'special_pay') ? $data->special_pay : null,
                'load_allowance'            =>  property_exists($data, 'load_allowance') ? $data->load_allowance : null,
                'transportation_allowance'  =>  property_exists($data, 'transportation_allowance') ? $data->transportation_allowance : null,
                'branch_allowance'          =>  property_exists($data, 'branch_allowance') ? $data->branch_allowance : null,
                'gate_allowance'            =>  property_exists($data, 'gate_allowance') ? $data->gate_allowance : null,
                'rice_allowance'            =>  property_exists($data, 'rice_allowance') ? $data->rice_allowance : null,
                'substinence_allowance'     =>  property_exists($data, 'substinence_allowance') ? $data->substinence_allowance : null,
                'laundy_allowance'          =>  property_exists($data, 'laundry_allwoance') ? $data->laundry_allowance : null,
                'remarks'                   =>  $req->remarks,
                'updated_by'                => Auth::id()

            ]);

            $allAllowances = $totalAllowance;

            if ($req->sector == 'public') {

                $newSalary = SalaryGrade::where('id', $req->sssgID)->first();

                $status = null;
                PublicSalary::where('id', $req->public_salary_id)->update(
                    [
                        'salary_grade_id' => $req->sssgID
                    ]
                );

                $salary = $newSalary->value;
                $salary = str_replace(',', '', $salary);
                $annualSalary = $salary * 12;
            }

            // ** Condition for declaring the new value of 'position_status' from the previous record in position history
            $position_status = 1; // ** for 'current position'
            $currentAssumptionDuty = Position_history::where('user_id', $req->selectedEmp)->orderBy('id', 'desc')->first();

            $date1 = Carbon::createFromFormat('Y-m-d', strval($req->dateEffectivity)); //** new Asssumption to Duty
            $date2 = Carbon::createFromFormat('Y-m-d', strval($currentAssumptionDuty->date_of_effectivity)); //** current Asssumption to Duty

            if ($date1->gt($date2)) {
                $position_status = 2; // ** for 'previous position'
                Position_history::where('user_id', $req->selectedEmp)->orderBy('id', 'desc')->first()->update([
                    'position_status'   =>  $position_status,
                    'updated_by' => Auth::id()
                ]);
            }

            $getPosition = EmployeeMetaInfo::where('user_id', $req->selectedEmp)->first();

            $position_history = Position_history::with(['user' => function ($query) {
                $query->select('id', 'name')->without('permissions', 'roles', 'storage', 'employeeMetaInfo');
            }, 'user.user_employeeMetaInfo' => function ($query) {
                $query->select('id', 'user_id')->without(['storage', 'roles', 'permissions', 'department', 'branch', 'division', 'position', 'employeeMetaInfo']);
            }])->create([
                'user_id'               => $req->selectedEmp,
                'position_id'           => $getPosition->position_id,
                'appointment_date'      => now(),
                'date_of_effectivity'   => $req->dateEffectivity,
                'status'                => $status,
                'position_status'       => $req->position_status ? $req->position_status : 1,
                'employment_status'     =>  1, //* re-employed
                'created_by'            => Auth::id()
            ]);

            $employeeExtraFieldColumn = EmployeeExtraFieldColumn::whereHas('employeeExtraField', function ($query) use ($req) {
                $query->where('user_id', $req->selectedEmp);
            })->get();

            foreach ($employeeExtraFieldColumn as $pds) {
                PDSPerPosition::create([
                    'position_history_id'   =>    $position_history->id,
                    'eefc_id'               =>    $pds->id,
                    'field_name'           =>     $pds->field_name,
                    'field_value'           =>    $pds->field_value,
                    'created_by'            =>    Auth::id()
                ]);
            }

            $oldServiceRecord = ServiceRecord::where('user_id', $getPosition->user_id)->whereNull('end_date');
            $remark = Remark::whereId(26)->first();

            ServiceRecord::create([
                'user_id'           =>  $req->selectedEmp,
                'start_date'        =>  $req->dateEffectivity,
                'position_id'       =>  $getPosition->position_id,
                'salary'            =>  $annualSalary,
                'allowance'         =>  $allAllowances,
                'employee_type_id'  =>  $getPosition->employee_type,
                'division_id'       =>  $getPosition->division_id,
                'remark_id'         =>  $remark->id,  //* Re-employment
                'created_by'        =>  Auth::id()
            ]);

            UserSalary::where('user_id', $req->selectedEmp)->update([
                'status'    =>  0
            ]);

            $newRecord = UserSalary::whereId($req->id)->first();

            $newRecord->old = collect($oldRecord);
            $newRecord->attributes = collect($newRecord);

            $this->logCustomMessage(
                're_employment_user_salary',
                $newRecord,
                Auth::user()->name . ' updated the user salary',
                $newRecord,
                UserSalaryLogType::RE_EMPLOYMENT,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => 'Successfully assigned salary for re-employment',
                'data'  => $req->all()
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    =>  ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01'],
                'msg'       =>  $e->getMessage(),
                'line'      =>  $e->getLine()
            ], 500);
        }
    }

    public function fetch_employee_for_reemployment()
    {
        return response()->json([
            'data' => UserSalary::where('status', 1)->get()
        ]);
    }

    public function fetch_employee_allowances_special_pays(Request $req)
    {
        $allowances = Allowance::where('status', Status::ACTIVE)->get();
        $special_pays = SpecialPay::where('status', Status::ACTIVE)->get();

        foreach ($allowances as $allowance) {
            $employee_allowance = EmployeeAllowance::where(
                [
                    ['user_id', $req->user_id],
                    ['allowance_id', $allowance->id]
                ]
            )->orderBy('effectivity_date', 'desc')->first();

            if ($employee_allowance) {
                $allowance->effectivity_date = $employee_allowance->effectivity_date;
                $allowance->amount = $employee_allowance->amount;
            }
        }

        foreach ($special_pays as $special_pay) {
            $employee_special_pay = EmployeeSpecialPay::where([
                ['user_id', $req->user_id],
                ['special_pay_id', $special_pay->id]
            ])->value('amount');
            $special_pay->amount = $employee_special_pay;
        }
        $data['allowances'] = $allowances;
        $data['special_pays'] = $special_pays;
        return $data;
    }
}
