<?php

namespace Suiterus\Adg\Controllers\Salary\AllowanceAndPays;

use App\Enums\Status;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\User;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Salary\UserSalary;
use Suiterus\Adg\Models\Salary\EmployeeAllowance;
use Suiterus\Adg\Models\Payroll\Allowance;

/**
 * Controller for the functions of Employee's assigned allowances
 */

class EmployeeAllowanceController extends Controller
{

    private $db;

    function __construct(){
        $this->db = DB::connection('adg_db');
    }

    /**
     * Fetch Allowances
     */
    public function index(Request $request) {

        $validate = Validator::make($request->all(), [
            'user_id'   => 'required|exists:mysql.users,id',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        try {

            $user = User::find($request->user_id);

            $data = $user->allowances()->where('status', Status::ACTIVE)->get();

            return response()->json([
                'data'    => $data
            ]);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Assign allowances
     */
    public function assign(Request $request) {

        $validate = Validator::make($request->all(), [
            'user_id'   => 'required|exists:mysql.users,id',
            'allowances.*.id' => 'required|exists:adg_db.allowances,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $user = User::find($request->user_id);

            // This function sets an array with a key of the allowance's id
            // and the value as an associative array with a key 'amount'
            // and a value of the actual allowance amount for the specified allowance
            // Reference at https://laravel.com/docs/8.x/eloquent-relationships#syncing-associations

            $allowances = [];
            foreach($request->allowances as $allowance) {
                $allowances[$allowance['id']] = ['amount' => $allowance['amount']];
            }

            $user->allowances()->sync($allowances);

            $this->db->commit();
            return response()->json([
                'text'  => 'Allowances assigned'
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }

    }
    public function syncAllowances() {

        $users =  User::whereHas('salary')->with('salary')->get();

        $this->db->beginTransaction();
        try {
            foreach($users as $user){

                if ( $user->salary->e_cola > 0 ) {
                    $allowance = Allowance::where('title', 'ECOLA')->first();
                    EmployeeAllowance::firstOrCreate([
                        'user_id'  =>  $user->id,
                        'allowance_id' => $allowance->id,
                        'amount' => $user->salary->e_cola,
                        'created_by' => $user->id,
                    ]);
                };

                if ( $user->salary->load_allowance > 0 ) {
                    $allowance = Allowance::where('title', 'Load Allowance')->first();
                    EmployeeAllowance::firstOrCreate([
                        'user_id'  =>  $user->id,
                        'allowance_id' => $allowance->id,
                        'amount' => $user->salary->load_allowance,
                        'created_by' => $user->id,
                    ]);
                };

                if ( $user->salary->transportation_allowance > 0 ) {
                    $allowance = Allowance::where('title', 'Transportation Allowance')->first();
                    EmployeeAllowance::firstOrCreate([
                        'user_id'  =>  $user->id,
                        'allowance_id' => $allowance->id,
                        'amount' => $user->salary->transportation_allowance,
                        'created_by' => $user->id,
                    ]);
                };

                if ( $user->salary->branch_allowance > 0 ) {
                    $allowance = Allowance::where('title', 'Branch Allowance')->first();
                    EmployeeAllowance::firstOrCreate([
                        'user_id'  =>  $user->id,
                        'allowance_id' => $allowance->id,
                        'amount' => $user->salary->branch_allowance,
                        'created_by' => $user->id,
                    ]);
                };

                if ( $user->salary->gate_allowance > 0 ) {
                    $allowance = Allowance::where('title', 'Gate Allowance')->first();
                    EmployeeAllowance::firstOrCreate([
                        'user_id'  =>  $user->id,
                        'allowance_id' => $allowance->id,
                        'amount' => $user->salary->gate_allowance,
                        'created_by' => $user->id,
                    ]);
                };

                if ( $user->salary->rice_allowance > 0 ) {
                    $allowance = Allowance::where('title', 'Rice Allowance')->first();
                    EmployeeAllowance::firstOrCreate([
                        'user_id'  =>  $user->id,
                        'allowance_id' => $allowance->id,
                        'amount' => $user->salary->rice_allowance,
                        'created_by' => $user->id,
                    ]);
                };

                if ( $user->salary->substinence_allowance > 0 ) {
                    $allowance = Allowance::where('title', 'Substinence Allowance')->first();
                    EmployeeAllowance::firstOrCreate([
                        'user_id'  =>  $user->id,
                        'allowance_id' => $allowance->id,
                        'amount' => $user->salary->substinence_allowance,
                        'created_by' => $user->id,
                    ]);
                };

                if ( $user->salary->laundry_allowance > 0 ) {
                    $allowance = Allowance::where('title', 'Laundry Allowance')->first();
                    EmployeeAllowance::firstOrCreate([
                        'user_id'  =>  $user->id,
                        'allowance_id' => $allowance->id,
                        'amount' => $user->salary->laundry_allowance,
                        'created_by' => $user->id,
                    ]);
                };
            }
            $this->db->commit();
            return response()->json([
                'text'  => 'Allowances successfully synced to the new table'
            ]);
        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    public function getEmployeeAllowances(Request $request){
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');

        $hasEffectivityDate = false;

        if ($request->effectivity_dates != "") {
            $hasEffectivityDate = $request->effectivity_dates['from'] == "" && $request->effectivity_dates['to'] == "" ? false : true;
        }

        return EmployeeAllowance::where('user_id', $request->user_id)->when($request->allowance_id != 'null' && isset($request->allowance_id) && $request->allowance_id != '', function($query) use($request) {
            $query->where('allowance_id', $request->allowance_id);
        })->when($hasEffectivityDate, function($query) use($request) {
            $query->whereBetween('effectivity_date', [$request->effectivity_dates['from'], $request->effectivity_dates['to']]);
        })->with(['allowance'])->paginate($paginate);
    }
}
