<?php

namespace Suiterus\Adg\Controllers\SPMS;

use Exception;
use App\Models\User;
use Illuminate\Http\Request;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;
use Suiterus\Adg\Models\SPMS\UserMfo;
use Suiterus\Adg\Models\SPMS\UserPcrForm;
use Suiterus\Adg\Models\Activity\Activity;

class UserPcrFormController extends Controller
{

    use HasCustomLogs;

    public function store(Request $request)
    {
        try {
            return DB::transaction(function () use ($request) {
                return DB::connection(env('ADG_DB_CONNECTION'))->transaction(function () use ($request) {

                    $employees = collect($request->employees);

                    foreach ($employees->chunk(500) as $employeeChunks) {
                        foreach ($employeeChunks as $employee) {
                            UserPcrForm::where('user_id', $employee['user_id'])->delete();
                            UserMfo::where('user_id', $employee['user_id'])->delete();
                            foreach ($employee['pcrs'] as $pcr) {
                                $userPcr = UserPcrForm::create([
                                    'user_id' => $employee['user_id'],
                                    'pcr_form_id' => $pcr,
                                    'created_by' => Auth::id()
                                ]);

                                $this->logCustomMessage("pcr_configuration", $userPcr, Auth::user()->name . " has configured PCR form for {$userPcr->user->name} - {$userPcr->pcrForm->name} ({$userPcr->pcrForm->abbreviation})", $userPcr, "User pcr configuration", new Activity());
                            }

                            foreach ($employee['mfos'] as $mfo) {
                                $userMfo = UserMfo::create([
                                    'user_id' => $employee['user_id'],
                                    'mfo_id' => $mfo,
                                    'created_by' => Auth::id()
                                ]);

                                $this->logCustomMessage("mfo_configuration", $userMfo, Auth::user()->name . " has configured MFO for {$userMfo->user->name} - {$userMfo->mfo->name}", $userMfo, "User mfo configuration", new Activity());
                            }
                        }
                    }

                    return response()->json([
                        'message' => 'Configuration successfully saved.'
                    ]);
                });
            });
        } catch (Exception $e) {
            return response()->json([
                'errors' => $e->getMessage(),
                'message' => 'Configuration failed to save.',
            ], 500);
        }
    }

    public function destroy($userId)
    {
        try {
            return DB::transaction(function () use ($userId) {
                return DB::connection(env('ADG_DB_CONNECTION'))->transaction(function () use ($userId) {
                    $userPcr = UserPcrForm::where('user_id', $userId)->first();

                    $this->logCustomMessage('user_pcr_deleted', null, Auth::user()->name . " has deleted {$userPcr->user->name} PCR Forms", null, "User pcr deleted", new Activity());

                    $userPcr->delete();

                    $userMfo = UserMfo::where('user_id', $userId)->first();

                    $this->logCustomMessage('user_mfo_deleted', null, Auth::user()->name . " has deleted {$userMfo->user->name} MFOs", null, "User mfo deleted", new Activity());

                    $userMfo->delete();

                    return response()->json([
                        'message' => 'Record successfully deleted.'
                    ]);
                });
            });
        } catch (Exception $e) {
            return response()->json([
                'errors' => $e->getMessage(),
                'message' => 'There was a problem in deleting the record.',
            ], 500);
        }
    }

    public function paginateByUser(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return User::has('pcrForms')->has('mfos')->without([
            'currentRole',
            'roles',
            'permissions',
            'storage',
            'employeeMetaInfo',
            'supervisor',
            'user_supervisor',
            'exitInterview',
            'userProfilePicture',
            'profileBasicInfo'
        ])->when($request->pcrs, function ($query) use ($request) {
            $query->whereHas('pcrForms', function ($query) use ($request) {
                $query->whereIn('pcr_form_id', $request->pcrs);
            });
        })->when($request->mfos, function ($query) use ($request) {
            $query->whereHas('mfos', function ($query) use ($request) {
                $query->whereIn('mfo_id', $request->mfos);
            });
        })->when($request->user_ids, function ($query) use ($request) {
            $query->whereIn('id', $request->user_ids);
        })->when($request->search, function ($query) use ($request) {
            $query->where('name', 'LIKE', '%' . $request->search . '%');
        })->with(['pcrForms', 'mfos', 'employeeMetaInfo' => function ($query) {
            $query->select('id', 'user_id', 'employee_id')->without([
                'corporation',
                'branch',
                'division',
                'department',
                'employeeType',
                'position',
                'itemCode'
            ]);
        }])->paginate($paginate);
    }

    public function getEmployeesWithoutPCR()
    {
        return User::doesntHave('pcrForms')->doesntHave('mfos')->whereNotIn('id', [1, 2])->without([
            'currentRole',
            'roles',
            'permissions',
            'storage',
            'supervisor',
            'user_supervisor',
            'exitInterview',
            'userProfilePicture',
            'profileBasicInfo'
        ])->with(['employeeMetaInfo' => function ($query) {
            $query->select('id', 'user_id', 'employee_id')->without([
                'corporation',
                'branch',
                'division',
                'department',
                'employeeType',
                'position',
                'itemCode'
            ]);
        }])->get();
    }

    public function getEmployeesWithPCR()
    {
        return User::has('pcrForms')->has('mfos')->whereNotIn('id', [1, 2])->without([
            'currentRole',
            'roles',
            'permissions',
            'storage',
            'supervisor',
            'user_supervisor',
            'exitInterview',
            'userProfilePicture',
            'profileBasicInfo'
        ])->with(['employeeMetaInfo' => function ($query) {
            $query->select('id', 'user_id', 'employee_id')->without([
                'corporation',
                'branch',
                'division',
                'department',
                'employeeType',
                'position',
                'itemCode'
            ]);
        }])->get();
    }

    public function getEmployeePCR(Request $request)
    {
        $pcrs = UserPcrForm::where('user_id', $request->user_id)->get();
        $mfos = UserMfo::where('user_id', $request->user_id)->get();

        return response()->json(['pcrs' => $pcrs, 'mfos' => $mfos]);
    }
}
