<?php

namespace Suiterus\Adg\Controllers\SPMS;

use App\Models\User;
use Illuminate\Http\Request;
use Suiterus\Adg\Models\SM\Unit;
use Suiterus\Adg\Models\SM\Office;
use Suiterus\Adg\Models\SM\Section;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\SM\Division;
use Suiterus\Adg\Models\SM\Department;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;
use Suiterus\Adg\Models\SPMS\PerformanceRatingReport;
use Suiterus\Adg\Services\OrgStructure\OrgStructureService;

class PerformanceRatingReportController extends Controller
{
    private $orgStructureService;

    public function __construct(OrgStructureService $orgStructureService)
    {
        $this->orgStructureService = $orgStructureService;
    }

    public function getEvaluationScoreByOffice(Request $request)
    {

        $office = $this->officeEntity((new Office()), $request->office_id, urldecode($request->office_name));

        $office->evaluation_score = $office->evaluation()->when($request->start_date && $request->end_date, function ($query) use ($request) {
            $query->whereDate('start_date', $request->start_date)->whereDate('end_date', $request->end_date);
        })->avg('overall_rate');
        $office->evaluation_score_adjective = $this->orgStructureService->getEvaluationScoreAdjective($office->evaluation_score);

        $performanceRatingReport = PerformanceRatingReport::create([
            'office_id' => $request->office_id,
            'department_id' => $request->department_id,
            'division_id' => $request->division_id,
            'section_id' => $request->section_id,
            'unit_id' => $request->unit_id,
            'rating_numeric' => $office->evaluation_score,
            'rating_adjective' => $office->evaluation_score_adjective,
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
            'created_by' => Auth::id()
        ]);

        $department = $this->officeEntity(new Department(), $request->department_id, urldecode($request->department_name));
        $division = $this->officeEntity(new Division(), $request->division_id, urldecode($request->division_name));
        $section = $this->officeEntity(new Section(), $request->section_id, urldecode($request->section_name));
        $unit = $this->officeEntity(new Unit(), $request->unit_id, urldecode($request->unit_name));

        $model = $model ?? $unit;
        $model = $model ?? $section;
        $model = $model ?? $division;
        $model = $model ?? $department;
        $model = $model ?? $office;

        $users = User::select('id', 'name')->whereHas('evaluation', function ($query) use ($model) {
            $query->when($model instanceof Office, function ($query) use ($model) {
                $query->where('office_id', $model->id);
            })->when($model instanceof Department, function ($query) use ($model) {
                $query->where('department_id', $model->id);
            })->when($model instanceof Division, function ($query) use ($model) {
                $query->where('division_id', $model->id);
            })->when($model instanceof Section, function ($query) use ($model) {
                $query->where('section_id', $model->id);
            })->when($model instanceof Unit, function ($query) use ($model) {
                $query->where('unit_id', $model->id);
            });
        })->without([
            'currentRole',
            'roles',
            'permissions',
            'storage',
            'employeeMetaInfo',
            'supervisor',
            'user_supervisor',
            'exitInterview',
            'userProfilePicture',
            'profileBasicInfo'
        ])->get();

        $employeeWithEvaluation = [];
        $totalScores = 0;
        $totalEmployees = 0;

        foreach ($users as $user) {
            $user->evaluation_score = $user->evaluation()->when($request->start_date && $request->end_date, function ($query) use ($request) {
                $query->whereDate('start_date', $request->start_date)->whereDate('end_date', $request->end_date);
            })->avg('overall_rate');
            $user->evaluation_score_adjective = $this->orgStructureService->getEvaluationScoreAdjective($user->evaluation_score);

            $totalScores += $user->evaluation_score;

            $employeeWithEvaluation[] = $user;
            $totalEmployees++;
        }

        return [
            'report' => $performanceRatingReport,
            'office' => $office,
            'department' => $department,
            'division' => $division,
            'section' => $section,
            'unit' => $unit,
            'employees' => [
                'users' => collect($employeeWithEvaluation),
                'overall_rate' => $totalEmployees > 0 ? round($totalScores / $totalEmployees, 2) : 0.0,
                'overall_adjective' => $totalEmployees > 0 ? $this->orgStructureService->getEvaluationScoreAdjective($totalScores / $totalEmployees) : 'N/A'
            ]
        ];
    }

    public function paginatePerformanceRatingReport(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return PerformanceRatingReport::when($request->id, function($query) use($request) {
            $query->where('id', $request->id);
        })->when($request->adjectival, function($query) use($request) {
            $query->where('rating_adjective', $request->adjectival);
        })->when($request->from && $request->to, function($query) use($request) {
            $query->whereBetween('start_date', [$request->from, $request->to]);
        })->with(['office'])->orderBy('created_at', 'desc')->paginate($paginate);
    }

    public function updateReportFilePath(Request $request)
    {
        $report = PerformanceRatingReport::find($request->report_id);

        if (!$report) {
            return response()->json([
                'message' => 'Report not found'
            ], 404);
        }

        $path = Storage::disk('local')->put("performance-rating-reports/{$report->id}", $request->file('file'));
        $report->file_path = $path;
        $report->save();
        return $report;
    }

    public function downloadPerformanceReport(Request $request)
    {
        $report = PerformanceRatingReport::find($request->report_id);
        return Storage::disk('local')->download($report->file_path);
    }

    private function officeEntity(Model $model, $id = null, $name = null)
    {
        if ($id || $name) {
            return $model::when($id, function ($query) use ($id) {
                return $query->where('id', $id);
            })->when($name, function ($query) use ($name) {
                return $query->where('name', $name);
            })->when($model instanceof Office, function ($query) {
                return $query->without(['department']);
            })->first();
        }
    }
}
