<?php

namespace Suiterus\Adg\Controllers\SPMS;

use App\Enums\SPMS\EvaluationStatus;
use Exception;
use App\Enums\Status;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Models\MetricEvaluationHistory;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\SPMS\MetricEvaluation;
use Suiterus\Adg\Models\SPMS\EmployeeEvaluation;
use Suiterus\Adg\Models\SPMS\EvaluationClassification;
use Suiterus\Adg\Models\SPMS\PerformanceEvaluation;
use Suiterus\Adg\Requests\SPMS\MetricEvaluation\DeleteRequest;
use Suiterus\Adg\Requests\SPMS\MetricEvaluation\UpdateRequest;
use Suiterus\Adg\Services\SPMS\MetricEvaluationService;

class MetricEvaluationController extends Controller
{
    public function create_mfo_entry(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'employee_evaluation_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.employee_evaluation,id',
            'mfo'  =>  'required|integer',
            'evaluation_classification_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.evaluation_classifications,id',
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        try {
            return DB::transaction(function () use ($request) {
                return DB::connection(env('ADG_DB_CONNECTION'))->transaction(function () use ($request) {
                    $employeeEvaluation = EmployeeEvaluation::where('id', $request->employee_evaluation_id)->first();

                    $performanceEvaluation = PerformanceEvaluation::updateOrCreate(
                        [
                            'evaluation_id'     => $employeeEvaluation->id,
                            'mfo'               => $request->mfo,
                        ],
                        [
                            'evaluation_id'     => $employeeEvaluation->id,
                            'mfo'               => $request->mfo,
                        ]
                    );

                    $average = MetricEvaluationService::compute_average_mfo_entry($request);

                    MetricEvaluation::create([
                        'performance_eval_id' => $performanceEvaluation->id,
                        'eval_classification_id' => $request->evaluation_classification_id,
                        'description' => $request->description,
                        'success_q' => $request->success_q,
                        'success_e' => $request->success_e,
                        'success_t' => $request->success_t,
                        'actual_q' => $request->actual_q,
                        'actual_e' => $request->actual_e,
                        'actual_t' => $request->actual_t,
                        'rate_q' => $request->rate_q,
                        'rate_e' => $request->rate_e,
                        'rate_t' => $request->rate_t,
                        'remarks_q' => $request->remarks_q,
                        'remarks_e' => $request->remarks_e,
                        'remarks_t' => $request->remarks_t,
                        'average' => $average,
                        'status' => $request->status
                    ]);


                    $performanceEvaluations = PerformanceEvaluation::where([
                        ['evaluation_id', $request->employee_evaluation_id], ['mfo', $request->mfo]
                    ])->get();

                    foreach ($performanceEvaluations as $performanceEvaluation) {
                        MetricEvaluationService::compute_overall_rating_per_mfo($request, $performanceEvaluation);
                    }

                    $overallRating = MetricEvaluationService::compute_overall_rating($employeeEvaluation);

                    $employeeEvaluation->overall_rate = $overallRating;
                    $employeeEvaluation->save();

                    return response()->json([
                        'text'      => 'The MFO entry has been successfully added.',
                    ]);
                });
            });
        } catch (Exception $e) {
            return response()->json([
                'message'    => ['There was a problem with creating the evaluation.'],
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    public function update_mfo_entry(UpdateRequest $request, $metric_evaluation_id)
    {
        try {
            DB::beginTransaction();

            $performanceEvaluation = PerformanceEvaluation::where([
                ['evaluation_id', $request->employee_evaluation_id], ['mfo', $request->mfo]
            ])->first();

            if (!$performanceEvaluation) {
                $getPerformanceEvaluation = PerformanceEvaluation::create([
                    'evaluation_id'     => $request->employee_evaluation_id,
                    'mfo'               => $request->mfo,
                ]);
            }

            $metricEval = MetricEvaluation::where('id', $metric_evaluation_id)->first();

            $old = $metricEval->histories()->create([
                'user_id' => Auth::id(),
                'old' => $metricEval
            ]);

            $average = MetricEvaluationService::compute_average_mfo_entry($request);

            $metricEval->update([
                'performance_eval_id' => $performanceEvaluation ? $performanceEvaluation->id : $getPerformanceEvaluation->id,
                'eval_classification_id' => $request->evaluation_classification_id,
                'description' => $request->description,
                'success_q' => $request->success_q,
                'success_e' => $request->success_e,
                'success_t' => $request->success_t,
                'actual_q' => $request->actual_q,
                'actual_e' => $request->actual_e,
                'actual_t' => $request->actual_t,
                'rate_q' => $request->rate_q,
                'rate_e' => $request->rate_e,
                'rate_t' => $request->rate_t,
                'remarks_q' => $request->remarks_q,
                'remarks_e' => $request->remarks_e,
                'remarks_t' => $request->remarks_t,
                'average' => $average,
                'status' => $request->status
            ]);

            $old->update([
                'new' => $metricEval
            ]);

            $employeeEvaluation = EmployeeEvaluation::where([
                ['id', $request->employee_evaluation_id], ['evaluator', Auth::id()]
            ])->first();

            if ($employeeEvaluation) {
                $metricEval->update([
                    'rate_q' => $request->rate_q,
                    'rate_e' => $request->rate_e,
                    'rate_t' => $request->rate_t,
                    'average' => $average,
                    'remarks' => $request->remarks
                ]);
            }

            $fetchPerformanceEvaluations = PerformanceEvaluation::where([
                ['evaluation_id', $request->employee_evaluation_id], ['mfo', $request->mfo]
            ])->get();

            foreach ($fetchPerformanceEvaluations as $performanceEvaluation) {
                MetricEvaluationService::compute_overall_rating_per_mfo($request, $performanceEvaluation);
            }

            $overallRating = MetricEvaluationService::compute_overall_rating($performanceEvaluation->employeeEvaluation);

            $performanceEvaluation->employeeEvaluation->overall_rate = $overallRating;
            $performanceEvaluation->employeeEvaluation->save();

            DB::commit();

            return response()->json([
                'text' => 'The MFO entry has been successfully updated.',
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'There was a problem with updating the MFO entry.',
                'error' => $e->getMessage()
            ]);
        }
    }

    public function update_mfo_status(Request $request, $metricEvaluationId)
    {

        return MetricEvaluation::where('id', $metricEvaluationId)->update([
            'status' => $request->status,
            'revision_remarks' => $request->revision_remarks
        ]);
    }

    public function count_by_mfo_status(Request $request)
    {
        return response()->json([
            'count' => MetricEvaluation::where([
                ['status', $request->status]
            ])->whereHas('performanceEvaluation.employeeEvaluation', function ($query) use ($request) {
                $query->where('id', $request->employee_evaluation_id);
            })->count(),
            'total' => MetricEvaluation::whereHas('performanceEvaluation.employeeEvaluation', function ($query) use ($request) {
                $query->where('id', $request->employee_evaluation_id);
            })->count()
        ]);
    }

    public function submit_mfo_for_status(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'evaluation_id' => 'required|exists' . env('ADG_DB_CONNECTION') . '.employee_evaluation,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $employeeEvaluation = EmployeeEvaluation::whereId($request->evaluation_id)->with('performanceEvaluations.metricEvaluations.getEvaluationClassification')->first();

            if ($employeeEvaluation->status == EvaluationStatus::IN_PROGRESS && $employeeEvaluation->evaluator == Auth::id()) {
                $overallRating = MetricEvaluationService::compute_overall_rating($employeeEvaluation);
                $employeeEvaluation->overall_rate = $overallRating;
                $employeeEvaluation->status = EvaluationStatus::SUBMITTED;
                $employeeEvaluation->save();
            } elseif ($employeeEvaluation->status == EvaluationStatus::SUBMITTED && $employeeEvaluation->approver == Auth::id()) {
                $employeeEvaluation->status = EvaluationStatus::APPROVED;
                $employeeEvaluation->save();
            } elseif ($employeeEvaluation->status == EvaluationStatus::APPROVED && $employeeEvaluation->employee_id == Auth::id()) {
                $employeeEvaluation->status = EvaluationStatus::ACKNOWLEDGED;
                $employeeEvaluation->save();
            }

            DB::commit();
            return response()->json([
                'text' => 'The evaluation has been updated.',
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => ['There was a problem with submitting the evaluation.'],
                'error' => $e->getMessage()
            ]);
        }
    }

    public function delete_mfo_entry(DeleteRequest $request)
    {
        try {
            MetricEvaluation::whereId($request->metric_evaluation_id)->delete();

            return response()->json([
                'text'  =>  'MFO entry has been removed.'
            ]);
        } catch (Exception $e) {
            return response()->json([
                'errors'    =>  ['There is a problem in removing MFO entry.'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }

    public function fetch_mfo_entries(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'employee_evaluation_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.employee_evaluation,id',
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }

        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return MetricEvaluation::whereHas('performanceEvaluation', function ($query) use ($request) {
            $query->where('evaluation_id', $request->employee_evaluation_id);
        })->with([
            'performanceEvaluation.employeeEvaluation.office',
            'performanceEvaluation.employeeEvaluation.department',
            'performanceEvaluation.employeeEvaluation.division',
            'performanceEvaluation.employeeEvaluation.section',
            'performanceEvaluation.employeeEvaluation.unit',
            'evaluationClassification'
        ])->paginate($paginate);
    }
    public function fetch_evaluation_classification()
    {
        return EvaluationClassification::where('status', Status::ACTIVE)->get();
    }
}
