<?php

namespace Suiterus\Adg\Controllers\SPMS;

use Exception;
use Carbon\Carbon;
use App\Models\User;
use Illuminate\Http\Request;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Enums\SPMS\EvaluationStatus;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\ActualDesignation;
use Suiterus\Adg\Models\SPMS\EmployeeEvaluation;
use Suiterus\Adg\Services\SPMS\MetricEvaluationService;
use Suiterus\Adg\Services\OrgStructure\OrgStructureService;
use Suiterus\Adg\Requests\SPMS\EmployeeEvaluation\ViewRequest;
use Suiterus\Adg\Requests\SPMS\EmployeeEvaluation\StoreRequest;

class EmployeeEvaluationController extends Controller
{
    use HasCustomLogs;

    private $orgStructureService;

    public function __construct(OrgStructureService $orgStructureService)
    {

        $this->orgStructureService = $orgStructureService;
    }

    public function store(StoreRequest $request)
    {
        try {
            $actualDesignation = ActualDesignation::where('user_id', Auth::id())->first();
            EmployeeEvaluation::create([
                'employee_id'       =>  Auth::id(),
                'evaluator'         =>  $request->evaluator,
                'approver'          =>  $request->approver,
                'start_date'        =>  $request->start_date,
                'end_date'          =>  $request->end_date,
                'review_form_type'  =>  $request->form_type,
                'office_id' => $actualDesignation->office_id,
                'department_id' => $actualDesignation->department_id,
                'division_id' => $actualDesignation->division_id,
                'section_id' => $actualDesignation->section_id,
                'unit_id' => $actualDesignation->unit_id,
                'status'            =>  EvaluationStatus::IN_PROGRESS
            ]);

            return response()->json([
                'text'      => 'The evaluation has been successfully created.',
            ]);
        }catch(Exception $e){
            return response()->json([
                'message'    => ['There was a problem with creating the evaluation.'],
                'error'   => $e->getMessage()
            ]);
        }
    }

    public function fetchSupervisor()
    {
        return User::whereId(Auth::id())->without(['roles', 'storage', 'permissions', 'employeeMetaInfo'])->first();
    }

    public function updateEvaluationStatus(Request $request, $evaluationId){
        $updateQuery = [
            'status' => $request->status,
        ];

        if ($request->comment_recommendation) {
            $updateQuery['comment_recommendation'] = $request->comment_recommendation;
        }

        return EmployeeEvaluation::where('id', $evaluationId)->update($updateQuery);
    }

    public function submit_evaluation(Request $request)
    {
        $valid = Validator::make($request->all(),[
            'employee_evaluation_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.employee_evaluation,id',
        ]);

        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }

        try {
            $employee_evaluation = EmployeeEvaluation::whereId($request->employee_evaluation_id)->first();

            $overall_rating = MetricEvaluationService::compute_overall_rating($employee_evaluation);

            $auth_id = Auth::id();

            if($auth_id == $employee_evaluation->employee_id) {
                $employee_evaluation->update([
                    'status' => EvaluationStatus::SUBMITTED,
                    'submitted_date' => Carbon::now()->format('Y-m-d H:i:s')
                ]);
            }
            elseif($auth_id == $employee_evaluation->evaluator) {
                $employee_evaluation->update([
                    'overall_rate' => $overall_rating,
                    'status' => EvaluationStatus::FOR_APPROVAL,
                    'comment_recommendation' => $request->comment_recommendation,
                    'review_date' => Carbon::now()->format('Y-m-d H:i:s')
                ]);
            }
            elseif($auth_id == $employee_evaluation->approver) {
                if ($request->has('declined') && $request->declined) {
                    $employee_evaluation->update([
                        'overall_rate' => null,
                        'status' => EvaluationStatus::DECLINED,
                        'reason' => $request->reason
                    ]);

                    return response()->json([
                        'text' => 'The evaluation has been successfully declined.',
                    ]);
                } else {
                    $employee_evaluation->update([
                        'status' => EvaluationStatus::APPROVED,
                        'reason' => $request->reason,
                        'comment_recommendation' => $request->comment_recommendation,
                        'approved_date' => Carbon::now()->format('Y-m-d H:i:s')
                    ]);

                    return response()->json([
                        'text' => 'The evaluation has been successfully approved.',
                    ]);
                }
            }
            else {
                return response()->json([
                    'error' => 'The authenticated user does not have permission for this action.'
                ], 403);
            }

            return response()->json([
                'text' => 'The evaluation has been successfully submitted.',
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'There was a problem with submitting the evaluation.',
                'error' => $e->getMessage()
            ]);
        }
    }

    public function list_evaluation_form(ViewRequest $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');

        $eval_form = EmployeeEvaluation::where('employee_id', Auth::user()->id)->when(count($request->coverage_date) > 0, function($query) use ($request){
            //if filter_dates has FROM and TO
            $query->when(count($request->coverage_date) == 2, function($query) use ($request){
                $query->whereDate('start_date', '>=', $request->coverage_date[0])
                ->whereDate('end_date', '<=', $request->coverage_date[1]);
            });
            //if filter_dates has only one date
            $query->when(count($request->coverage_date) == 1, function($query) use ($request){
                $query->whereDate('start_date', $request->coverage_date[0])
                ->orWhere('end_date', $request->coverage_date[0]);
            });
        })->when($request->status != null, function($query) use ($request){
            $query->where('status', $request->status);
        })
        ->where('review_form_type', $request->form_type)
        ->with(['employee', 'employee.office.office', 'evaluator','approver']);

        if(count($request->approvers) > 0) {
            $eval_form = $eval_form->whereIn('approver', $request->approvers);
        }

        if(count($request->evaluators) > 0) {
            $eval_form = $eval_form->whereIn('evaluator', $request->evaluators);
        }

        return $eval_form->paginate($paginate);
    }

    public function list_other_evaluation(ViewRequest $request)
    {
        $user = User::whereId(Auth::user()->id)->first();
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');

        $organizationArr = $this->orgStructureService->getEmployeeOrganizationChild($user);

        if (!$organizationArr) {
            return response()->json([
                'error' => 'The authenticated user does not have actual designation.'
            ], 403);
        }

        $evaluationFormOrganization = $user->actualDesignation->unit;
        $evaluationFormOrganization = $evaluationFormOrganization ?? $user->actualDesignation->section;
        $evaluationFormOrganization = $evaluationFormOrganization ?? $user->actualDesignation->division;
        $evaluationFormOrganization = $evaluationFormOrganization ?? $user->actualDesignation->department;
        $evaluationFormOrganization = $evaluationFormOrganization ?? $user->actualDesignation->office;

        $pcrs = $this->orgStructureService->getPcrConfig($evaluationFormOrganization);

        if (!$pcrs->first()) {
            return response()->json([
                'error' => 'The authenticated user does not have permission for this action.'
            ], 403);
        }

        $other_eval_form = EmployeeEvaluation::where(function ($query) {
            $query->where('evaluator', Auth::id())
                ->orWhere('approver', Auth::id());
        })
        ->with(['employee', 'employee.office.office', 'evaluator', 'approver']);

        if($request->employees) {
            $other_eval_form = $other_eval_form->whereIn('employee_id', $request->employees);
        }

        if($request->approvers) {
            $other_eval_form = $other_eval_form->whereIn('approver', $request->approvers);
        }

        if($request->evaluators) {
            $other_eval_form = $other_eval_form->whereIn('evaluator', $request->evaluators);
        }

        $other_eval_form->where(function($query) use($organizationArr, $request){
            $query->when(count($request->coverage_date) > 0, function($query) use ($request){
                //if filter_dates has FROM and TO
                $query->when(count($request->coverage_date) == 2, function($query) use ($request){
                    $query->whereDate('start_date', '>=', $request->coverage_date[0])
                    ->whereDate('end_date', '<=', $request->coverage_date[1]);
                });
                //if filter_dates has only one date
                $query->when(count($request->coverage_date) == 1, function($query) use ($request){
                    $query->whereDate('start_date', $request->coverage_date[0])
                    ->orWhere('end_date', $request->coverage_date[0]);
                });
            })->when($request->status != null, function($query) use ($request){
                $query->where('status', $request->status);
            })->when($request->employee_name != null && $request->employee_name != '', function($query) use ($request){
                $query->whereHas('employee', function($query) use ($request){
                    $query->where('name', 'LIKE', '%'.$request->employee_name.'%');
                });
            })
            ->where('review_form_type', $request->form_type)->when(isset($organizationArr['officeIds'][0]) && (bool) $request->is_searching == false, function($query) use($organizationArr){
                $query->orWhereIn('office_id', $organizationArr['officeIds'])->where('employee_id', '!=', Auth::id());
            })->when(isset($organizationArr['departmentIds'][0]) && (bool) $request->is_searching == false, function($query) use($organizationArr) {
                $query->orWhereIn('department_id', $organizationArr['departmentIds'])->where('employee_id', '!=', Auth::id());
            })->when(isset($organization['divisionIds'][0]) && (bool) $request->is_searching == false, function($query) use($organizationArr){
                $query->orWhereIn('division_id', $organizationArr['divisionIds'])->where('employee_id', '!=', Auth::id());
            })->when(isset($organization['sectionIds'][0]) && (bool) $request->is_searching == false, function($query) use($organizationArr) {
                $query->orWhereIn('section_id', $organizationArr['sectionIds'])->where('employee_id', '!=', Auth::id());
            })->when(isset($organizationArr['unitIds'][0]) && (bool) $request->is_searching == false, function($query) use($organizationArr) {
                $query->orWhereIn('unit_id', $organizationArr['unitIds'])->where('employee_id', '!=', Auth::id());
            });
        });

        return $other_eval_form->paginate($paginate);
    }

    public function view_evaluation_form(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'evaluation_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.employee_evaluation,id'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }

        $organizationArr = $this->orgStructureService->getEmployeeOrganizationChild(Auth::user());

        return EmployeeEvaluation::whereId($request->evaluation_id)->where(function($query) use($organizationArr){
            $query->where('employee_id', Auth::user()->id)
            ->orWhere('evaluator', Auth::user()->id)
            ->orWhere('approver', Auth::user()->id)->orWhereIn('office_id', $organizationArr['officeIds'])->orWhereIn('department_id', $organizationArr['departmentIds'])->orWhereIn('division_id', $organizationArr['divisionIds'])->orWhereIn('section_id', $organizationArr['sectionIds'])->orWhereIn('unit_id', $organizationArr['unitIds']);
        })
        ->with(['employee', 'employee.office.office', 'evaluator','approver','performanceEvaluations.metricEvaluations.evaluationClassification'])
        ->first();
    }

    public function stamp(Request $request) {
        $valid = Validator::make($request->all(), [
            'evaluation_id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.employee_evaluation,id'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors' => $valid->errors()
            ], 400);
        }

        try {

            $employee_evaluation = EmployeeEvaluation::where('id', $request->evaluation_id)->first();

            $auth_id = Auth::id();

            if($auth_id == $employee_evaluation->employee_id) {
                $employee_evaluation->update([
                    'submitted_date' => Carbon::now(),
                    'employee_ip_address' => $request->ip_address,
                ]);
                $description = Auth::user()->name . ' stamped the ' . $employee_evaluation->pcrFormType->name . ' Form';
            }
            elseif($auth_id == $employee_evaluation->evaluator) {
                $employee_evaluation->update([
                    'review_date' => Carbon::now(),
                    'evaluator_ip_address' => $request->ip_address,
                ]);
                $description = Auth::user()->name . ' stamped the ' . $employee_evaluation->pcrFormType->name . ' Form of ' . $employee_evaluation->employee->name;
            }
            elseif($auth_id == $employee_evaluation->approver) {
                $employee_evaluation->update([
                    'approved_date' => Carbon::now(),
                    'approver_ip_address' => $request->ip_address,
                    'status' => 6
                ]);
                $description = Auth::user()->name . ' stamped the ' . $employee_evaluation->pcrFormType->name . ' Form of ' . $employee_evaluation->employee->name;
            }
            else {
                return response()->json([
                    'error' => 'The authenticated user does not have permission for this action.'
                ], 403);
            }

            $old_employee_evaluation = clone $employee_evaluation;

            $employee_evaluation->attributes = collect($employee_evaluation);
            $employee_evaluation->old = collect($old_employee_evaluation);

            $this->logCustomMessage('stamp_pcr_form', $employee_evaluation, $description, $employee_evaluation, 'Stamp a PCR Form', new Activity());

            DB::commit();

            return response()->json([
                'text' => 'The evaluation has been successfully stamped.',
            ]);

        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'There was a problem with stamping the evaluation.',
                'error' => $e->getMessage()
            ]);
        }
    }
}
