<?php

namespace Suiterus\Adg\Controllers\SPMS;

use Exception;
use App\Enums\SPMS\MFOType;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\Enums\Log\BalanceScoreCardLogType;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\SPMS\BalanceScoreCard;
use Suiterus\Adg\Models\SPMS\MetricEvaluation;

class BalanceScoreCardController extends Controller
{

    use HasCustomLogs;

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();
        try {
            $metricEvaluation = MetricEvaluation::find($request->metric_evaluation_id);

            if (!$metricEvaluation) {
                return response()->json([
                    'error' => 'Metric evaluation not found'
                ], 404);
            }

            $balanceScoreCard = BalanceScoreCard::where('first_half_metric_evaluation_id', $metricEvaluation->id)->orWhere('second_half_metric_evaluation_id', $metricEvaluation->id)->first();

            if ($balanceScoreCard) {
                return response()->json([
                    'error' => 'You have already created a balance score card for this metric evaluation'
                ], 400);
            }

            $performanceEvaluation = $metricEvaluation->performanceEvaluation;

            if ($performanceEvaluation->mfo != MFOType::STRATEGIC) {
                return response()->json([
                    'error' => 'Balance score card is only available for strategic metric evaluations'
                ], 400);
            }

            $employeeEvaluation = $performanceEvaluation->employeeEvaluation;
            $isValidForm = $employeeEvaluation->whereHas('pcrFormType', function ($query) {
                $query->where('abbreviation', 'DePCR');
            })->first();

            if (!$isValidForm) {
                return response()->json([
                    'error' => 'Invalid PCR form, only DePCR is allowed'
                ], 400);
            }

            $year = Carbon::parse($employeeEvaluation->start_date)->format('Y');
            $employeeId = $employeeEvaluation->employee_id;
            $classificationId = $metricEvaluation->eval_classification_id;
            $pcrFormTypeId = $employeeEvaluation->review_form_type;

            $metricEvaluationIds = MetricEvaluation::select('id')->whereHas('performanceEvaluation.employeeEvaluation', function ($query) use ($employeeId, $year, $pcrFormTypeId) {
                $query->where([
                    ['employee_id', $employeeId],
                    ['review_form_type', $pcrFormTypeId],
                ])->whereYear('start_date', $year);
            })->where('eval_classification_id', $classificationId)->get()->pluck('id');

            $balanceScoreCard = BalanceScoreCard::whereIn('first_half_metric_evaluation_id', $metricEvaluationIds)->orWhereIn('second_half_metric_evaluation_id', $metricEvaluationIds)->first();

            if ($balanceScoreCard) {
                $oldBalanceScoreCard = clone $balanceScoreCard;
                $balanceScoreCard->update([
                    'first_half_metric_evaluation_id' => $balanceScoreCard->first_half_metric_evaluation_id ?? $request->metric_evaluation_id,
                    'second_half_metric_evaluation_id' => $balanceScoreCard->second_half_metric_evaluation_id ?? $request->metric_evaluation_id,
                    'objectives' => $balanceScoreCard->objectives ?? $request->objectives,
                    'plans' => $balanceScoreCard->plans ?? $request->plans,
                    'success_indicator' => $balanceScoreCard->success_indicator ?? $request->success_indicator,
                    'q1_action_plan' => $balanceScoreCard->q1_action_plan ?? $request->q1_action_plan,
                    'q1_status' => $balanceScoreCard->q1_status ?? $request->q1_status,
                    'q1_legend' => $balanceScoreCard->q1_legend ?? $request->q1_legend,
                    'q2_action_plan' => $balanceScoreCard->q2_action_plan ?? $request->q2_action_plan,
                    'q2_status' => $balanceScoreCard->q2_status ?? $request->q2_status,
                    'q2_legend' => $balanceScoreCard->q2_legend ?? $request->q2_legend,
                    'first_half_legend' => $balanceScoreCard->first_half_legend ?? $request->first_half_legend,
                    'q3_action_plan' => $balanceScoreCard->q3_action_plan ?? $request->q3_action_plan,
                    'q3_status' => $balanceScoreCard->q3_status ?? $request->q3_status,
                    'q3_legend' => $balanceScoreCard->q3_legend ?? $request->q3_legend,
                    'q4_action_plan' => $balanceScoreCard->q4_action_plan ?? $request->q4_action_plan,
                    'q4_status' => $balanceScoreCard->q4_status ?? $request->q4_status,
                    'q4_legend' => $balanceScoreCard->q4_legend ?? $request->q4_legend,
                    'annual_legend' => $balanceScoreCard->annual_legend ?? $request->annual_legend,
                    'updated_by' => Auth::id(),
                ]);

                $balanceScoreCard->old = collect($oldBalanceScoreCard);
                $balanceScoreCard->attributes = collect($balanceScoreCard);

                $this->logCustomMessage('create_balance_score_card', $balanceScoreCard, Auth::user()->name . ' updated a balance score card', $balanceScoreCard, BalanceScoreCardLogType::UPDATE, new Activity());

                DB::connection(env('ADG_DB_CONNECTION'))->commit();

                return response()->json([
                    'message' => 'Balance score card updated successfully'
                ], 200);
            }

            $balanceScoreCard = BalanceScoreCard::create([
                'first_half_metric_evaluation_id' => $request->metric_evaluation_id,
                'objectives' => $request->objectives,
                'plans' => $request->plans,
                'success_indicator' => $request->success_indicator,
                'q1_action_plan' => $request->q1_action_plan,
                'q1_status' => $request->q1_status,
                'q1_legend' => $request->q1_legend,
                'q2_action_plan' => $request->q2_action_plan,
                'q2_status' => $request->q2_status,
                'q2_legend' => $request->q2_legend,
                'first_half_legend' => $request->first_half_legend,
                'q3_action_plan' => $request->q3_action_plan,
                'q3_status' => $request->q3_status,
                'q3_legend' => $request->q3_legend,
                'q4_action_plan' => $request->q4_action_plan,
                'q4_status' => $request->q4_status,
                'q4_legend' => $request->q4_legend,
                'annual_legend' => $request->annual_legend,
                'created_by' => Auth::id()
            ]);

            $this->logCustomMessage('create_balance_score_card', $balanceScoreCard, Auth::user()->name . ' created a balance score card', $balanceScoreCard, BalanceScoreCardLogType::CREATE, new Activity());

            DB::connection(env('ADG_DB_CONNECTION'))->commit();

            return response()->json([
                'message' => 'Balance score card created successfully',
            ]);
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollBack();
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => 'Something went wrong while processing your request.'
            ]);
        }
    }

    public function update(Request $request, $scoreCardId)
    {

        DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();
        try {
            $balanceScoreCard = BalanceScoreCard::where('id', $scoreCardId)->first();

            if (!$balanceScoreCard) {
                return response()->json([
                    'message' => 'Balance score card not found'
                ], 404);
            }

            $oldBalanceScoreCard = clone $balanceScoreCard;

            $balanceScoreCard->update([
                'objectives' =>  $request->objectives ?? $balanceScoreCard->objectives,
                'plans' =>  $request->plans ?? $balanceScoreCard->plans,
                'success_indicator' =>  $request->success_indicator ?? $balanceScoreCard->success_indicator,
                'q1_action_plan' =>  $request->q1_action_plan ?? $balanceScoreCard->q1_action_plan,
                'q1_status' =>  $request->q1_status ?? $balanceScoreCard->q1_status,
                'q1_legend' =>  $request->q1_legend ?? $balanceScoreCard->q1_legend,
                'q2_action_plan' =>  $request->q2_action_plan ?? $balanceScoreCard->q2_action_plan,
                'q2_status' =>  $request->q2_status ?? $balanceScoreCard->q2_status,
                'q2_legend' =>  $request->q2_legend ?? $balanceScoreCard->q2_legend,
                'first_half_legend' =>  $request->first_half_legend ?? $balanceScoreCard->first_half_legend,
                'q3_action_plan' =>  $request->q3_action_plan ?? $balanceScoreCard->q3_action_plan,
                'q3_status' =>  $request->q3_status ?? $balanceScoreCard->q3_status,
                'q3_legend' =>  $request->q3_legend ?? $balanceScoreCard->q3_legend,
                'q4_action_plan' =>  $request->q4_action_plan ?? $balanceScoreCard->q4_action_plan,
                'q4_status' =>  $request->q4_status ?? $balanceScoreCard->q4_status,
                'q4_legend' =>  $request->q4_legend ?? $balanceScoreCard->q4_legend,
                'annual_legend' =>  $request->annual_legend ?? $balanceScoreCard->annual_legend,
                'updated_by' => Auth::id(),
            ]);

            $balanceScoreCard->old = collect($oldBalanceScoreCard);
            $balanceScoreCard->attributes = collect($balanceScoreCard);

            $this->logCustomMessage('update_balance_score_card', $balanceScoreCard, Auth::user()->name . ' updated a balance score card', $balanceScoreCard, BalanceScoreCardLogType::UPDATE, new Activity());

            DB::connection(env('ADG_DB_CONNECTION'))->commit();

            return response()->json([
                'message' => 'Balance score card updated successfully'
            ], 200);
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollBack();
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => 'Something went wrong while processing your request.'
            ]);
        }
    }

    public function getByMfo(Request $request)
    {
        $metricEvaluation = MetricEvaluation::find($request->metric_evaluation_id);

        if (!$metricEvaluation) {
            return response()->json([
                'message' => 'Metric evaluation not found'
            ], 404);
        }

        $employeeEvaluation = $metricEvaluation->performanceEvaluation->employeeEvaluation;

        //getting dependencies
        $year = Carbon::parse($employeeEvaluation->start_date)->format('Y');
        $employeeId = $employeeEvaluation->employee_id;
        $classificationId = $metricEvaluation->eval_classification_id;
        $pcrFormTypeId = $employeeEvaluation->review_form_type;

        $metricEvaluationIds = MetricEvaluation::select('id')->whereHas('performanceEvaluation.employeeEvaluation', function ($query) use ($employeeId, $year, $pcrFormTypeId) {
            $query->where([
                ['employee_id', $employeeId],
                ['review_form_type', $pcrFormTypeId],
            ])->whereYear('start_date', $year);
        })->where('eval_classification_id', $classificationId)->get()->pluck('id');

        return BalanceScoreCard::whereIn('first_half_metric_evaluation_id', $metricEvaluationIds)->orWhereIn('second_half_metric_evaluation_id', $metricEvaluationIds)->first();
    }
}
