<?php

namespace Suiterus\Adg\Controllers\SM;


use Exception;
use App\Models\User;
use Illuminate\Http\Request;
use App\Enums\Log\UnitLogType;
use Suiterus\Adg\Models\SM\Unit;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\ActualDesignation;
use Suiterus\Adg\Models\EMI\EmployeeMetaInfo;
use Suiterus\Adg\Services\OrgStructure\Unit\UnitService;
use Suiterus\Adg\Services\OrgStructure\OrgStructureService;

class UnitsController extends Controller
{
    use HasCustomLogs;

    private $orgStructureService;
    private $unitService;

    public function __construct(UnitService $unitService, OrgStructureService $orgStructureService)
    {
        $this->unitService = $unitService;
        $this->orgStructureService = $orgStructureService;
    }

    public function create_unit(Request $req)
    {

        $valid = Validator::make($req->all(), [
            'name'    =>  'required|string|unique:adg_db.units,name,NULL,id,deleted_at,NULL',
            'status' => 'integer|required'
        ], [
            'name.unique'   =>  $req->name . ' is already taken.',
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        try {
            return DB::transaction(function () use ($req) {
                return DB::connection(env('ADG_DB_CONNECTION'))->transaction(function () use ($req) {
                    $unit = Unit::create([
                        'name' => $req->name,
                        'description' => $req->description,
                        'office_id' => $req->office_id,
                        'department_id' => $req->department_id,
                        'division_id' => $req->division_id,
                        'section_id' => $req->section_id,
                        'head_employee_id' => $req->head_employee_id,
                        'status' => $req->status,
                        'created_by' => Auth::id()
                    ]);

                    if(isset($req->pcfr_configuration)){
                        $this->orgStructureService->configPcr($unit, $req->pcfr_configuration);
                    }

                    if (isset($req->minimum_overtime)) {
                        $this->orgStructureService->configMinimumOvertime($unit, $req->minimum_overtime);
                    }        

                    if ($req->employees) {
                        foreach ($req->employees as $userId) {
                            $this->unitService->assignEmployeeUnit($userId, $unit->id);
                        }
                    }

                    $this->logCustomMessage(
                        'create_unit',
                        $unit,
                        Auth::user()->name . ' created a new Unit: ' . $req->name,
                        $unit,
                        UnitLogType::CREATE,
                        new Activity()
                    );

                    return response()->json([
                        'text'  =>  $req->name . ' created successfully!'
                    ]);
                });
            });
        } catch (Exception $e) {
            return response()->json([
                'errors'    =>  ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01'],
                'msg'   =>  $e->getMessage()
            ], 500);
        }
    }

    public function init_list_unit(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return response()->json([
            'data'  =>  Unit::with('section', 'pcrs', 'office', 'department', 'division', 'overtime')->orderBy('id', 'desc')->paginate($paginate)
        ]);
    }

    public function search_unit_name(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return response()->json([
            'data'  =>  Unit::where('name', 'LIKE', '%' . $request->keyword . '%')->with('section', 'pcrs', 'office', 'department', 'division', 'overtime')->paginate($paginate)
        ]);
    }

    public function show_employees($unitId)
    {
        return User::whereHas('actualDesignation', function ($query) use ($unitId) {
            $query->where('unit_id', $unitId);
        })->without([
            'currentRole',
            'roles',
            'permissions',
            'storage',
            'employeeMetaInfo',
            'supervisor',
            'user_supervisor',
            'exitInterview',
            'userProfilePicture',
            'profileBasicInfo'
        ])->get();
    }

    public function edit_unit(Request $req)
    {
        $connect = env('ADG_DB_CONNECTION');
        $valid = Validator::make($req->all(), [
            'id'        =>  "required|exists:{$connect}.units,id",
            'name'      =>  'required|string|min:4|unique:adg_db.units,name,' . $req->id . ',id,deleted_at,NULL',
            'status'    => 'integer|required'
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }

        try {
            return DB::transaction(function () use ($req) {
                return DB::connection(env('ADG_DB_CONNECTION'))->transaction(function () use ($req) {
                    $perms = Unit::find($req->id);

                    $perms->name = $req->name;
                    $perms->description = $req->description;
                    $perms->office_id = $req->office_id;
                    $perms->department_id = $req->department_id;
                    $perms->division_id = $req->division_id;
                    $perms->section_id = $req->section_id;
                    $perms->head_employee_id = $req->head_employee_id;
                    $perms->status = $req->status;
                    $perms->updated_by = Auth::id();

                    if(isset($req->pcfr_configuration)){
                        $this->orgStructureService->configPcr($perms, $req->pcfr_configuration);
                    }

                    if (isset($req->minimum_overtime)) {
                        $this->orgStructureService->configMinimumOvertime($perms, $req->minimum_overtime);
                    }  

                    ActualDesignation::where('unit_id', $perms->id)->update([
                        'office_id' => null,
                        'department_id' => null,
                        'division_id' => null,
                        'section_id' => null,
                        'unit_id' => null,
                    ]);

                    $updated = Unit::find($req->id);
                    $updated->old = collect($perms);
                    $updated->attributes = collect($updated);

                    $this->logCustomMessage(
                        'update_unit',
                        $updated,
                        Auth::user()->name . ' updated the Unit: ' . $req->name,
                        $updated,
                        UnitLogType::UPDATE,
                        new Activity()
                    );

                    $perms->save();

                    if (isset($req->employees)) {
                        foreach ($req->employees as $userId) {
                            $this->unitService->assignEmployeeUnit($userId, $req->id);
                        }
                    }

                    return response()->json([
                        'text'  =>  $req->name . ' has been updated.'
                    ]);
                });
            });
        } catch (Exception $e) {

            return response()->json([
                'errors'    =>  ['There is a problem in updating a Unit.'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }

    public function delete_unit(Request $req)
    {
        DB::beginTransaction();
        try {
            $perm = Unit::findOrFail($req->id);

            $this->logCustomMessage(
                'delete_unit',
                $perm,
                Auth::user()->name . ' deleted the Unit: ' . $perm->name,
                $perm,
                UnitLogType::DELETE,
                new Activity()
            );

            $perm->delete();
            DB::commit();
            return response()->json([
                'text'  =>  'Unit has been deleted.'
            ]);
        } catch (ME $ee) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Unit doesnt exists.'],
            ], 400);
        }
    }
}
