<?php

namespace Suiterus\Adg\Controllers\SM;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Suiterus\Adg\Models\SM\TrainingType;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;

class TrainingTypeController extends Controller
{
    public function create(Request $request) {

        $validate = Validator::make($request->all(), [
            'name'         => 'required|min:2|unique:adg_db.training_type,name,NULL,id,deleted_at,NULL',
            'status'        => 'required|integer|in:1,2'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $training_type = TrainingType::create([
                'name'         => $request->name,
                'description'   => $request->description,
                'status'        => $request->status,
                'created_by'    => $request->user()->id,
            ]);

            DB::commit();
            return response()->json([
                'text'  =>  $training_type->name . ' has been created.'
            ]);

        } catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => [ 'There was a problem in creating the training type record.' ],
                'message'   => $e->getMessage()
            ], 500); 
        }

    }

    public function edit(Request $request) {
        
        $validate = Validator::make($request->all(), [
            'name'         => 'required|min:2|unique:adg_db.training_type,name,'. $request->id.',id,deleted_at,NULL',
            'status'        => 'required|integer|in:1,2'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $training_type = TrainingType::findOrFail($request->id);
            $training_type->update([
                'name'     => $request->name,
                'description'     => $request->description,
                'status'     => $request->status,
                'updated_by'     => $request->user()->id,
            ]);
            $training_type->save();
            DB::commit();
            return response()->json([
                'text'  => $request->name . ' has been updated.'
            ]);

        } catch(ModelNotFoundException $me) {
            DB::rollBack();
            return response()->json([
                'errors'    => ["Training Type doesn't exist"],
                'message'   => $me->getMessage()
            ], 500);
        }
        catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in editing the training type'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    public function delete(Request $request) {

        DB::beginTransaction();
        try {

            $training_type = TrainingType::findOrFail($request->id);
            $name = $training_type['name'];
            $training_type->delete();
            DB::commit();
            return response()->json([
                'text'  => $name. ' has been deleted.'
            ]);
            
        } catch(ModelNotFoundException $me) {
            DB::rollBack();
            return response()->json([
                'errors'    => ["Training type doesn't exist"],
                'message'   => $me->getMessage()
            ], 500);
        } catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in deleting the record'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    public function init_list(Request $request) {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return response()->json([
            'data'  => TrainingType::orderBy('id', 'desc')->paginate($paginate)
        ]);
    }

    public function list_all() {
        return TrainingType::all();
    }

    public function search(Request $request) {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return response()->json([
            'data'  => TrainingType::where('name', 'LIKE', '%'.$request->keyword.'%')->paginate($paginate)
        ]);
    }
}
