<?php

namespace Suiterus\Adg\Controllers\SM;

use DB;
use Exception;
use Validator;
use Illuminate\Http\Request;
use App\Traits\Logs\HasCustomLogs;
use Suiterus\Adg\Models\SM\Branch;
use App\Enums\Log\SuspensionLogType;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\SM\Suspension;
use Suiterus\Adg\Models\Activity\Activity;

class SuspensionController extends Controller
{
    use HasCustomLogs;
    
    public function create(Request $request) {

        $valid = Validator::make($request->all(),[
            'title'    => 'required|string|unique:adg_db.suspensions,title,NULL,id,deleted_at,NULL',
            'status'   => 'integer|required'
            
        ],[
            'title.unique'   =>  $request->title.' is already taken.',
        ]);

        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }

        DB::beginTransaction();
        try {

            $suspension = Suspension::create([
                'title'             =>  $request->title,
                'description'       =>  $request->description,
                'time_effectivity'  =>  $request->time_effectivity,
                'branch_id'         =>  $request->branch_id,
                'date'              =>  $request->date,
                'status'            =>  $request->status, 
                'created_by'        =>  Auth::id()
            ]);

            $this->logCustomMessage(
                'create_suspension',
                $suspension,
                Auth::user()->name . ' created a Suspension: ' . $request->title,
                $suspension,
                SuspensionLogType::CREATE,
                new Activity()
            );
            
            DB::commit();
            return response()->json([
                'text'  =>  $request->title.' created successfully!'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01' ],
                'msg'       =>  $e->getMessage(),
                'line'      => $e->getLine()
            ],500);
        }
    }

    public function init_list(Request $request){
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');

        return response()->json([
            'data'  =>  Suspension::orderBy('id', 'desc')->paginate($paginate)
        ]);
    }

    public function search(Request $request){
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');

        return response()->json([
            'data'  =>  Suspension::where('title', 'LIKE', '%'.$request->keyword.'%')->paginate($paginate)
        ]);
    }

    public function edit(Request $request){
        $valid = Validator::make($request->all(),[
            'id'      =>  'required|integer|exists:adg_db.suspensions,id',
            'title'   =>  'required|string|min:4|unique:adg_db.suspensions,title,'.$request->id.',id,deleted_at,NULL',
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try{
            try{
                $perms = Suspension::findOrFail($request->id);
                $perms->title = $request->title;
                $perms->description = $request->description;
                $perms->time_effectivity = $request->time_effectivity;
                $perms->branch_id = $request->branch_id;
                $perms->date = $request->date;
                $perms->status = $request->status;
                
                $updated = Suspension::find($request->id);
                $updated->old = collect($perms);
                $updated->attributes = collect($updated);

                $this->logCustomMessage(
                    'update_suspension',
                    $updated,
                    Auth::user()->name . ' updated the Suspension: ' . $request->title,
                    $updated,
                    SuspensionLogType::UPDATE,
                    new Activity()
                );

                $perms->save();
                DB::commit();
                return response()->json([
                    'text'  =>  $request->title.' has been updated.'
                ]);
            }catch(ME $ee){
                DB::rollback();
                return response()->json([
                    'errors'    =>  [ 'Suspension doesnt exists.' ],
                ],400);
            }
        }catch(Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'There is a problem in updating a Suspension.' ],
                'msg'       =>  $e->getMessage()
            ],500);
        }
    }

    public function delete(Request $request){
        DB::beginTransaction();
        try{
            $perm = Suspension::findOrFail($request->id);

            $this->logCustomMessage(
                'delete_suspension',
                $perm,
                Auth::user()->name . ' deleted the Suspension: ' . $perm->title,
                $perm,
                SuspensionLogType::DELETE,
                new Activity()
            );

            $perm->delete();
            DB::commit();
            return response()->json([
                'text'  =>  'Suspension has been deleted.'
            ]);
        }catch(ME $ee){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Suspension doesnt exists.' ],
            ],400);
        }
    }
}