<?php

namespace Suiterus\Adg\Controllers\SM;

use Exception;
use Carbon\Carbon;
use App\Enums\Status;
use Illuminate\Http\Request;
use App\Enums\Log\PayrollLogType;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Payroll\SpecialPay;
use Suiterus\Adg\Models\Salary\EmployeeSpecialPay;
use Suiterus\Adg\Services\Payroll\SpecialPayService;
use Suiterus\Adg\Exports\SpecialPay\SpecialPayExport;
use Suiterus\Adg\Imports\SpecialPayImport;

/**
 * Setup Controller for special pays used in payroll
 */

class SpecialPayController extends Controller
{
    use HasCustomLogs;

    private $specialPayService;

    private $db;

    public function __construct(SpecialPayService $specialPayService)
    {
        $this->specialPayService = $specialPayService;
        $this->db = DB::connection('adg_db');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $paginate = isset($request->paginate) && $request->paginate !== null ? $request->paginate : env('DEFAULT_PAGECOUNT');
        try {

            $data = SpecialPay::when(isset($request->search) && $request->search !== null, function($query) use ($request) {
                $keyword = '%' . $request->search . '%';
                $query->where('title', 'LIKE', $keyword);
            })->with('specialPayFrequency', 'employees', 'employees.user')->paginate($paginate);

            return response()->json([
                'data' => $data
            ]);

        } catch(Exception $e) {
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Fetch all special pays list for display at forms
     */
    public function all() {
        return response()->json([
            'data'  => SpecialPay::where('status', Status::ACTIVE)->with('specialPayFrequency', 'employees', 'employees.user')->get()
        ]); 
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'title'     => 'required',
            'status'    => 'required|in:1,2'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $specialPay = SpecialPay::create([
                'title' => $request->title,
                'description' => $request->description,
                'special_pay_frequency_id' => $request->special_pay_frequency_id,
                'amount' => $request->amount, 
                'amount_per_frequency' => $request->amount_per_frequency,
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
                'period' => $request->period,
                'created_by' => Auth::id(),
                'status'      => $request->status  
            ]);

            $addEmployee = $this->specialPayService->addEmployee($request->employee_ids, $specialPay->id, $request->amount, $request->title);
            if ($addEmployee != null) {
                return $addEmployee;
            }

            $this->logCustomMessage(
                'create_special_pay',
                $specialPay,
                Auth::user()->name . ' created a special pay: ' . $specialPay->title,
                $specialPay,
                PayrollLogType::CREATE_SPECIAL_PAY,
                new Activity()
            );

            $this->db->commit();
            return response()->json([
                'text'  => __('responses.success.create')
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        return response()->json([
            'data' => SpecialPay::find($request->id)->with('specialPayFrequency', 'employees', 'employees.user')->first()
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.' . 'special_pays,id',
            'title'     => 'required',
            'status'    => 'required|in:1,2'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $special_pay = SpecialPay::find($request->id);
            $special_pay->update([
                'title' => $request->title,
                'description' => $request->description,
                'special_pay_frequency_id' => $request->special_pay_frequency_id,
                'amount' => $request->amount, 
                'amount_per_frequency' => $request->amount_per_frequency,
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
                'period' => $request->period,
                'status'      => $request->status  
            ]);
            $special_pay->save();

            EmployeeSpecialPay::where('special_pay_id', $special_pay->id)->delete();

            $addEmployee = $this->specialPayService->addEmployee($request->employee_ids, $special_pay->id, $request->amount, $request->title);
            if ($addEmployee != null) {
                return $addEmployee;
            }

            $updatedSpecialPay = SpecialPay::find($request->id);
            $updatedSpecialPay->old = collect($special_pay);
            $updatedSpecialPay->attributes = collect($updatedSpecialPay);
                
            $this->logCustomMessage(
                'update_special_pay',
                $updatedSpecialPay,
                Auth::user()->name . ' updated the special pay: ' . $updatedSpecialPay->title,
                $updatedSpecialPay,
                PayrollLogType::UPDATE_SPECIAL_PAY,
                new Activity()
            );

            $this->db->commit();

            return response()->json([
                'text'  => __('responses.success.update')
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function delete(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.' . 'special_pays,id',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $special_pay = SpecialPay::find($request->id);

            EmployeeSpecialPay::where('special_pay_id', $special_pay->id)->delete();

            $this->logCustomMessage(
                'delete_special_pay',
                $special_pay,
                Auth::user()->name . ' deleted the special pay: ' . $special_pay->title,
                $special_pay,
                PayrollLogType::DELETE_SPECIAL_PAY,
                new Activity()
            );

            $special_pay->delete();

            $this->db->commit();
            return response()->json([
                'text' => __('responses.success.delete')
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    public function update_employee_special_pay(Request $request) 
    {
        $validate = Validator::make($request->all(), [
            'id' => 'required|exists:' . env('ADG_DB_CONNECTION') . '.' . 'employee_special_pays,id',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();

        try {

            $employeeSpecialPay = EmployeeSpecialPay::find($request->id);
            $employeeSpecialPay->update([
                'amount' => $request->amount,
                'updated_by' => Auth::id(),
                'updated_now' => Carbon::now()
            ]);
            $employeeSpecialPay->save();

            $updatedEmployeeSpecialPay = EmployeeSpecialPay::find($request->id);
            $updatedEmployeeSpecialPay->old = collect($employeeSpecialPay);
            $updatedEmployeeSpecialPay->attributes = collect($employeeSpecialPay);
                
            $this->logCustomMessage(
                'update_special_pay',
                $updatedEmployeeSpecialPay,
                Auth::user()->name . ' updated the special pay: ' . $updatedEmployeeSpecialPay->special_pay->title . ' of ' . $updatedEmployeeSpecialPay->user->name,
                $updatedEmployeeSpecialPay,
                PayrollLogType::UPDATE_SPECIAL_PAY,
                new Activity()
            );

            $this->db->commit();

            return response()->json([
                'text'  => __('responses.success.update')
            ]);

        } catch(Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage()
            ], 500);
        }
    }

    public function upload_template(Request $request) 
    {
        try {
            $file = $request->file('special_pay');
      
            $import = new SpecialPayImport();
            $import->import($file);
    
            foreach ($import->failures() as $failure) {
                $failure->row(); 
                $failure->attribute();
                $failure->errors();
                $failure->values();
            }

            return response()->json([
                'message' => 'Successfully uploaded the template.',
            ]);
            
        } catch (Exception $e) {
            return response()->json([
                'errors' => $import->failures(),
            ]);
        }
    }

    public function download_template(Request $request) 
    {
        $this->logCustomMessage(
            'download_special_pay_template',
            null,
            Auth::user()->name . ' downloaded the template for special pay',
            null,
            PayrollLogType::DOWNLOAD_SPECIAL_PAY_TEMPLATE,
            new Activity()
        );

        $timestamp = Carbon::now()->format('D M d Y H_i_s e');
        $filename = "SPECIAL_PAY_TEMPLATE_" . str_replace(['/', '\\', ':', ' '], '_', $timestamp . '.xlsx');

        return Excel::download(new SpecialPayExport($request->row), $filename);
    }
}
