<?php

namespace Suiterus\Adg\Controllers\SM;

use App\Enums\Log\RosterSetupLogType;
use App\Http\Controllers\Controller;
use App\Traits\Logs\HasCustomLogs;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\SM\Shift;

class ShiftController extends Controller
{
    use HasCustomLogs;
    public function create_shift(Request $request) {

        $valid = Validator::make($request->all(),[
            'code'  =>  'required|string|unique:adg_db.shifts,code',
            'name'    =>  'required|string|unique:adg_db.shifts,name,NULL,id,deleted_at,NULL',
            'description'   =>  'string|nullable',
            'shift_in'   =>  'nullable',
            'shift_out'   =>  'nullable',
            'break_in'   =>  'nullable',
            'break_out'   =>  'nullable',
            'work_hours'    =>  'numeric|nullable',
            'status'    =>  'integer|required'
        ],[
            'name.unique'   =>  $request->name.' is already taken.',
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try {
            
            $shift = Shift::create([
                'code'   =>  $request->code,
                'name'  =>   $request->name,
                'description'   =>  $request->description,
                'shift_in'   =>  $request->shift_in,
                'shift_out'   =>  $request->shift_out,
                'break_in'   =>  $request->break_in,
                'break_out'   =>  $request->break_out,
                'work_hours'    =>  $request->work_hours,
                'status'    => $request->status ? $request->status : 1
            ]);
            
            $this->logCustomMessage(
                'create_roster_shift',
                $shift,
                Auth::user()->name . ' Create roster shift',
                $shift,
                RosterSetupLogType::CREATE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  =>  $request->name.' created successfully.'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01' ],
                'msg'   =>  $e->getMessage()
            ],500);
        }
    }

    public function init_list_shift(){
        return response()->json([
            'data'  =>  Shift::orderBy('id', 'desc')->paginate(5)
        ]);
    }

    public function search_shift_name(Request $request){
        return response()->json([
            'data'  =>  Shift::where('name', 'LIKE', '%'.$request->keyword.'%')->paginate(5)
        ]);
    }

    public function update_shift(Request $request){
        $valid = Validator::make($request->all(),[
            'name'      =>  'required|string|unique:adg_db.shifts,name,'.$request->id.',id,deleted_at,NULL',
            'status'    => 'required|integer'
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try{
            try{
                $shift = Shift::findOrFail($request->id);
                $old_record = $shift;
                $shift->update([
                    'code'   =>  $request->code,
                    'name'  =>   $request->name,
                    'description'   =>  $request->description,
                    'shift_in'   =>  $request->shift_in,
                    'shift_out'   =>  $request->shift_out,
                    'break_in'   =>  $request->break_in,
                    'break_out'   =>  $request->break_out,
                    'work_hours'    =>  $request->work_hours,
                    'status'    => $request->status
                ]);
                
                $shift->save();

                $this->logCustomMessage(
                    'update_roster_shift',
                    $old_record,
                    Auth::user()->name . ' Update roster shift',
                    $shift,
                    RosterSetupLogType::UPDATE,
                    new Activity()
                );
    
                DB::commit();
                return response()->json([
                    'text'  =>  $request->name.' has been updated.'
                ]);
            }catch(ME $ee){
                DB::rollback();
                return response()->json([
                    'errors'    =>  [ 'Shift doesn`t exists.' ],
                ],400);
            }
        }catch(Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'There is a problem in updating a Shift.' ],
                'msg'       =>  $e->getMessage()
            ],500);
        }
    }

    public function delete_shift(Request $request){
        DB::beginTransaction();
        try{
            $perm = Shift::findOrFail($request->id);
            $perm->delete();

            $this->logCustomMessage(
                'delete_roster_shift',
                $perm,
                Auth::user()->name . ' Delete roster shift',
                $perm,
                RosterSetupLogType::DELETE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  =>  'Shift has been deleted.'
            ]);
        }catch(ME $ee){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Shift doesn`t exists.' ],
            ],400);
        }
    }

    public function fetch_shifts() {
        return response()->json([
            'data'  => Shift::all()
        ]);
    }
}
