<?php

namespace Suiterus\Adg\Controllers\SM;

use Exception;
use App\Models\User;
use Illuminate\Http\Request;
use Suiterus\Adg\Models\SM\Unit;
use App\Enums\Log\SectionLogType;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use Suiterus\Adg\Models\SM\Section;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\ActualDesignation;
use Suiterus\Adg\Services\OrgStructure\OrgStructureService;
use Suiterus\Adg\Services\OrgStructure\Section\SectionService;

class SectionsController extends Controller
{

    use HasCustomLogs;

    private $orgStructureService;
    private $sectionService;

    public function __construct(SectionService $sectionService, OrgStructureService $orgStructureService)
    {
        $this->sectionService = $sectionService;
        $this->orgStructureService = $orgStructureService;
    }

    public function create_section(Request $req)
    {

        $valid = Validator::make($req->all(), [
            'name'    =>  'required|string|unique:adg_db.sections,name,NULL,id,deleted_at,NULL',
            'status' => 'integer|required'
        ], [
            'name.unique'   =>  $req->name . ' is already taken.',
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {
            $section = Section::create([
                'name' => $req->name,
                'description' => $req->description,
                'division_id' => $req->division_id,
                'head_employee_id' => $req->head_employee_id,
                'status' => $req->status,
                'created_by' => Auth::id()
            ]);

            $this->orgStructureService->configPcr($section, $req->pcfr_configuration);

            if (isset($req->minimum_overtime)) {
                $this->orgStructureService->configMinimumOvertime($section, $req->minimum_overtime);
            }

            foreach ($req->employees as $userId) {
                $this->sectionService->assignEmployeeSection($userId, $section->id);
            }

            $this->update_unit($req->unit_ids, $section->id);

            $this->logCustomMessage(
                'create_section',
                $section,
                Auth::user()->name . ' created a new Section: ' . $req->name,
                $section,
                SectionLogType::CREATE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  =>  $req->name . ' created successfully!'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01'],
                'msg'   =>  $e->getMessage()
            ], 500);
        }
    }

    public function init_list_section(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return response()->json([
            'data'  =>  Section::with('units', 'division', 'pcrs', 'overtime')->orderBy('id', 'desc')->paginate($paginate)
        ]);
    }

    public function search_section_name(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        return response()->json([
            'data'  =>  Section::where('name', 'LIKE', '%' . $request->keyword . '%')->with('units', 'division', 'pcrs', 'overtime')->paginate($paginate)
        ]);
    }

    public function show_employees($sectionId)
    {
        return User::whereHas('actualDesignation', function ($query) use ($sectionId) {
            $query->where('section_id', $sectionId)->whereNull(['unit_id']);
        })->without([
            'currentRole',
            'roles',
            'permissions',
            'storage',
            'employeeMetaInfo',
            'supervisor',
            'user_supervisor',
            'exitInterview',
            'userProfilePicture',
            'profileBasicInfo'
        ])->get();
    }

    public function edit_section(Request $req)
    {
        $valid = Validator::make($req->all(), [
            'name'      =>  'required|string|min:4|unique:adg_db.sections,name,' . $req->id . ',id,deleted_at,NULL',
            'status'    => 'integer|required'
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {

            $perms = Section::findOrFail($req->id);
            $perms->name = $req->name;
            $perms->description = $req->description;
            $perms->division_id = $req->division_id;
            $perms->head_employee_id = $req->head_employee_id;
            $perms->status = $req->status;
            $perms->updated_by = Auth::id();

            $this->orgStructureService->configPcr($perms, $req->pcfr_configuration);

            if (isset($req->minimum_overtime)) {
                $this->orgStructureService->configMinimumOvertime($perms, $req->minimum_overtime);
            }

            Unit::where('section_id', $perms->id)->update(['section_id' => null]);

            if (isset($req->unit_ids)) {
                $this->sectionService->assignUnits($req->unit_ids, $perms);
            }

            ActualDesignation::where('section_id', $perms->id)->whereNull('unit_id')->update([
                'office_id' => null,
                'department_id' => null,
                'division_id' => null,
                'section_id' => null,
                'unit_id' => null,
            ]);

            $updated = Section::find($req->id);
            $updated->old = collect($perms);
            $updated->attributes = collect($updated);

            $this->logCustomMessage(
                'update_section',
                $updated,
                Auth::user()->name . ' updated the Section: ' . $req->name,
                $updated,
                SectionLogType::UPDATE,
                new Activity()
            );

            $perms->save();

            foreach ($req->employees as $userId) {
                $this->sectionService->assignEmployeeSection($userId, $perms->id);
            }

            DB::commit();
            return response()->json([
                'text'  =>  $req->name . ' has been updated.'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in updating a Section.'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }

    public function delete_section(Request $req)
    {
        DB::beginTransaction();
        try {
            $perm = Section::findOrFail($req->id);

            $has_unit = Unit::where('section_id', $perm->id)->first();
            if ($has_unit) {
                return response()->json([
                    'errors' => ['Cannot delete section with units.'],
                ], 400);
            }

            $this->logCustomMessage(
                'delete_section',
                $perm,
                Auth::user()->name . ' deleted the Section: ' . $perm->name,
                $perm,
                SectionLogType::DELETE,
                new Activity()
            );

            $perm->delete();
            DB::commit();
            return response()->json([
                'text'  =>  'Section has been deleted.'
            ]);
        } catch (ME $ee) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Section doesnt exists.'],
            ], 400);
        }
    }

    private function validate_unit($unit_ids)
    {
        if ($unit_ids != null) {
            foreach ($unit_ids as $unit_id) {
                $unit = Unit::whereId($unit_id)->first();
                if ($unit->section_id != null) {
                    return true;
                }
            }
        }
        return false;
    }

    private function update_unit($unit_ids, $section_id)
    {
        if ($unit_ids != null) {
            foreach ($unit_ids as $unit_id) {
                $unit = Unit::whereId($unit_id)->first();
                $unit->update([
                    'section_id' => $section_id,
                ]);
            }
        }
    }

    public function show_units(Request $request, $section_id)
    {
        $units = Unit::where('section_id', $section_id)->get()->append('organization_type');

        $units = $this->orgStructureService->evaluationScoreMapper($units, $request->start_date, $request->end_date);
        return $units;
    }
}
