<?php

namespace Suiterus\Adg\Controllers\SM;

use App\Enums\Log\ScheduleSetupLogType;
use App\Traits\Logs\HasCustomLogs;
use Suiterus\Adg\Models\SM\ScheduleType;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Auth;
use DB;
use Validator;
use Exception;
use Suiterus\Adg\Models\Activity\Activity;

/**
 * Controller to manage schedule types - CRUD functionalities
 */

class ScheduleTypeController extends Controller
{
    use HasCustomLogs;
    /**
     * Create new schedule type
     */
    public function create(Request $request) {

        $validate = Validator::make($request->all(), [
            'title'         => 'required|string|min:1|unique:adg_db.schedule_types,title,NULL,id,deleted_at,NULL',
            'status'        => 'required|integer',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {
            $schedule_type = ScheduleType::create([
                'title'         => $request->title,
                'status'        => $request->status,
                'description'   => $request->description,
                'created_by'    => $request->user()->id,
                'updated_by'    => $request->user()->id,
            ]);
           
            $this->logCustomMessage(
                'create_schedule_type',
                $schedule_type,
                Auth::user()->name . ' Create schedule type',
                $schedule_type,
                ScheduleSetupLogType::CREATE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => $schedule_type->title . ' has been created.'
            ]);

        } catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem with creating the schedule type'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Edit selected schedule type
     */
    public function edit(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'        => 'required|integer',
            'title'     => 'required|string|min:1|unique:adg_db.schedule_types,title,'. $request->id.',id,deleted_at,NULL',
            'status'    => 'required|integer',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $schedule_type = ScheduleType::findOrFail($request->id);
            $old_record = $schedule_type;
            $schedule_type->update([
                'title'         => $request->title,
                'status'        => $request->status,
                'description'   => $request->description,
                'updated_by'    => $request->user()->id,
            ]);
            $schedule_type->save();

            $this->logCustomMessage(
                'update_schedule_type',
                $old_record,
                Auth::user()->name . ' Update schedule type',
                $schedule_type,
                ScheduleSetupLogType::UPDATE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => $schedule_type->title .' has been updated.'
            ]);

        } catch (ModelNotFoundException $me) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['Schedule Type does not exist.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem with updating the schedule type.'],
                'message'   => $e->getMessage()
            ], 500);
        }


    } 

    /**
     * Delete schedule type
     */
    public function delete(Request $request) {

        DB::beginTransaction();
        try {

            $schedule_type = ScheduleType::findOrFail($request->id);
            $title = $schedule_type->title;
            if($schedule_type->schedule_titles->count() > 0) {
                return response()->json([
                    'errors'  => ['Cannot delete ' . $title . ' as it still has schedules with this type.']
                ], 400);
            }
            $schedule_type->delete();

            $this->logCustomMessage(
                'delete_schedule_type',
                $schedule_type,
                Auth::user()->name . ' Delete schedule type',
                $schedule_type,
                ScheduleSetupLogType::DELETE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => $title . ' has been deleted'
            ]);

        } catch (ModelNotFoundException $me) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['Schedule Type does not exist.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem with deleting the schedule type'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }

    } 

    /**
     * Init list of all schedule types paginated
     */
    public function init_list() {
        return response()->json([
            'data'  => ScheduleType::orderBy('id', 'desc')->paginate(5)
        ]);
    }

    /**
     * Search a schedule type
     */
    public function search(Request $request) {
        return response()->json([
            'data'  => ScheduleType::where('title', 'LIKE', '%'.$request->keyword.'%')->paginate(5)
        ]);
    }

    /**
     * Fetch all schedule types
     */
    public function fetch_all(){

        return response()->json([
            'data'  => ScheduleType::orderBy('title', 'desc')->get()
        ]);

    }

}
