<?php

namespace Suiterus\Adg\Controllers\SM;

use App\Enums\Log\ScheduleSetupLogType;
use App\Traits\Logs\HasCustomLogs;
use Suiterus\Adg\Models\SM\ScheduleTitle;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use DB;
use Validator;
use Exception;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\SM\ScheduleType;

/**
 * Controller to manage all schedule titles used - CRUD functionalities
 * 
 * Note:
 *  Essentially, the schedule_titles is the template of the schedule
 *  And the schedule_templates table is the table where the days are stored for the template
 *  To avoid confusion, use schedule_titles simply as the title of the template,
 *  and the schedule_templates table as the days of the template. 
 */

class ScheduleTitleController extends Controller
{
    use HasCustomLogs;
    /**
     * Create new schedule title
     */
    public function create(Request $request) {

        $validate = Validator::make($request->all(), [
            'type'          => 'required|integer',
            'title'         => 'required|string|min:2|unique:adg_db.schedule_titles,title,NULL,id,deleted_at,NULL',
            'status'        => 'required|integer',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $schedule_title = ScheduleTitle::create([
                'type'          => $request->type,
                'title'         => $request->title,
                'shift_code'    => $request->shift_code,
                'description'   => $request->description,
                'hasSchedule'   => 2,
                'basis'      => 1,
                'status'        => $request->status,
                'created_by'    => $request->user()->id,
                'updated_by'    => $request->user()->id,
            ]);

            $this->logCustomMessage(
                'create_schedule_title',
                $schedule_title,
                Auth::user()->name . ' Create schedule title',
                $schedule_title,
                ScheduleSetupLogType::CREATE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => $schedule_title->title . ' has been created.'
            ]);

        } catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem with creating the schedule title'],
                'message'   => $e->getMessage(),
            ], 500);
        }

    }

    /**
     * Edit selected schedule title
     */
    public function edit(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'            => 'required|integer',
            'type'          => 'required|integer',
            'title'         => 'required|string|min:2|unique:adg_db.schedule_titles,title,'. $request->id.',id,deleted_at,NULL',
            'status'        => 'required|integer',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $schedule_title = ScheduleTitle::findOrFail($request->id);
            $old_record = $schedule_title;
            $schedule_title->update([
                'type'          => $request->type,
                'title'         => $request->title,
                'shift_code'    => $request->shift_code,
                'description'   => $request->description,
                'status'        => $request->status,
                'updated_by'    => $request->user()->id,
            ]);
            $schedule_title->save();

            $this->logCustomMessage(
                'update_schedule_title',
                $old_record,
                Auth::user()->name . ' Update schedule title',
                $schedule_title,
                ScheduleSetupLogType::UPDATE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => $schedule_title->title . ' has been updated.'
            ]);

        } catch (ModelNotFoundException $me) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['Schedule Temlate does not exist.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in updating the schedule template'],
                'message'   => $e->getMessage()
            ], 500);
        }


    } 

    /**
     * Delete schedule title
     */
    public function delete(Request $request) {

        DB::beginTransaction();
        try {

            $schedule_title = ScheduleTitle::findOrFail($request->id);
            $title = $schedule_title->title;
            $schedule_title->delete();

            $this->logCustomMessage(
                'delete_schedule_title',
                $schedule_title,
                Auth::user()->name . ' Delete schedule title',
                $schedule_title,
                ScheduleSetupLogType::DELETE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => $title . ' has been deleted.'
            ]);

        } catch (ModelNotFoundException $me) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['Schedule title does not exist.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem with deleting the schedule title.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    } 

    /**
     * Fetch all schedule titles as raw data - not paginated
     */
    public function fetch_schedule_titles() {
        return response()->json([
            'data'  => ScheduleTitle::orderBy('title', 'desc')->get()
        ]);
    }

    /**
     * Fetch all schedule titles with its schedule template - paginated
     */
    public function init_list() {
        return response()->json([
            'data'  => ScheduleTitle::orderBy('id', 'desc')->whereHas('schedule_type')->with(['schedule_template','schedule_type' => function ($query) {
                $query->select('id','title');
            }])->paginate(5)
        ]);
    }

    /**
     * Search and return schedule titles
     */
    public function search(Request $request) {
        return response()->json([
            'data'  => ScheduleTitle::where('title', 'LIKE', '%'.$request->keyword.'%')->whereHas('schedule_type')->with(['schedule_template', 'schedule_type'])->paginate(5)
        ]);
    }
}
