<?php

namespace Suiterus\Adg\Controllers\SM;

use App\Enums\Log\ScheduleSetupLogType;
use App\Traits\Logs\HasCustomLogs;
use Suiterus\Adg\Models\SM\ScheduleTitle;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Validator;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\SM\ScheduleTemplate;

/** 
 * Schedule Template Controller with timetables for admin side - control of schedules 
 * 
 * This controller syncs the days of the schedule template, and is assigned to the schedule_titles table
 * Please see the ScheduleTitleController.php for the explanation of the template, title, and days  
 */

class ScheduleTemplateController extends Controller
{
    use HasCustomLogs;
    private $db;

    public function __construct() {
        $this->db = DB::connection('adg_db');
    }

    /**
     * Fetch all the schedule template days 
     */
    public function fetch_schedule_template(Request $request) {
        try {
            $schedule_title = ScheduleTitle::findOrFail($request->title_id);
            return response()->json([
                'data'  => $schedule_title->schedule_template()->get()
            ]);
        } catch (ModelNotFoundException $me) {
            return response()->json([
                'errors'    => ['The Schedule Template does not exist.'],
                'message'   => $me->getMessage()
            ]);
        }
    }
    
    /**
     * Create schedule template days for a schedule
     */
    public function create(Request $request) {
    
        $validate = Validator::make($request->all(), [
            'title_id'  => 'required|exists:adg_db.schedule_titles,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $db = DB::connection('adg_db');
        $db->beginTransaction();
        try {

            $schedule = ScheduleTitle::findOrFail($request->title_id);
            $template = $schedule->schedule_template()->create([
                'title_id'   => $request->title_id,
                'day'           => $request->day,
                'status'        => $request->status,
                'created_by'    => $request->user()->id,
                'updated_by'    => $request->user()->id,
            ]);

            // Synchronize the timetables with the schedule template day
            $template->timetables()->sync($request->timetables);
            $schedule->hasSchedule = 1;
            $schedule->save();

            $this->logCustomMessage(
                'create_schedule_template',
                $schedule,
                Auth::user()->name . ' Create schedule template',
                $schedule,
                ScheduleSetupLogType::CREATE,
                new Activity()
            );

            $db->commit();
            return response()->json([
                'text'  => __('responses.success.create')
            ]);

        } catch(Exception $e) {
            $db->rollBack();
            return response()->json([
                'errors'    => [ __('responses.exception') ],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Create schedule template for all the days in a selected schedule - synchronizing with timetables
     */
    public function create_full(Request $request) {
    
        $validate = Validator::make($request->all(), [
            'title_id'  => 'required|exists:adg_db.schedule_titles,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $db = DB::connection('adg_db');
        $db->beginTransaction();
        try {

            $schedule_title = ScheduleTitle::findOrFail($request->title_id);
            foreach($request->schedules as $schedule) {
                $record = [
                    'day'           => $schedule['day'],
                    'status'        => $schedule['status'],
                    'created_by'    => $request->user()->id,
                    'updated_by'    => $request->user()->id,
                    'created_at'    => now(),
                    'updated_at'    => now(),
                ];
                $template = $schedule_title->schedule_template()->create($record);
                $template->timetables()->sync($schedule['timetables']);
            }
            $schedule_title->hasSchedule = 1;
            $schedule_title->save();

            $this->logCustomMessage(
                'create_full_schedule_template',
                $schedule_title,
                Auth::user()->name . ' Create full schedule template',
                $schedule_title,
                ScheduleSetupLogType::CREATE,
                new Activity()
            );

            $db->commit();
            return response()->json([
                'text'  => __('responses.success.create')
            ]);

        } catch(Exception $e) {
            $db->rollBack();
            return response()->json([
                'errors'    => [ __('responses.exception') ],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Edit the schedule template days of the schedule with syncing timetables
     */
    public function edit_full(Request $request) {
    
        $validate = Validator::make($request->all(), [
            'title_id'  => 'required|exists:adg_db.schedule_titles,id'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $db = DB::connection('adg_db');
        $db->beginTransaction();
        try {

            $schedule_title = ScheduleTitle::findOrFail($request->title_id);
            $old_record = $schedule_title;
            // detach timetables for all the templates before deleting the template - 2 level syncing
            foreach($schedule_title->schedule_template as $template) {
                $template->timetables()->detach();
            }
            $schedule_title->schedule_template()->delete();
            foreach($request->schedules as $schedule) {
                $record = [
                    'day'           => $schedule['day'],
                    'status'        => $schedule['status'],
                    'created_by'    => $request->user()->id,
                    'updated_by'    => $request->user()->id,
                    'created_at'    => now(),
                    'updated_at'    => now(),
                ];

                // create a record for a day, and sync the timetable with the day
                $template = $schedule_title->schedule_template()->create($record);
                $template->timetables()->sync($schedule['timetables']);
            }
            $schedule_title->hasSchedule = 1;
            $schedule_title->save();

            $this->logCustomMessage(
                'Update_full_schedule_title',
                $old_record,
                Auth::user()->name . ' Update full schedule template',
                $schedule_title,
                ScheduleSetupLogType::UPDATE,
                new Activity()
            );

            $db->commit();
            return response()->json([
                'text'  => __('responses.success.create')
            ]);

        } catch(Exception $e) {
            $db->rollBack();
            return response()->json([
                'errors'    => [ __('responses.exception') ],
                'message'   => $e->getMessage(),
            ], 500);
        }

    }

    /**
     * Set t he basis for the schedule template
     */
    public function set_basis(Request $request) {

        $validate = Validator::make($request->all(), [
            'title_id'  => 'required|exists:adg_db.schedule_titles,id',
            'basis'     => 'required|in:1,2'
        ]);
        
        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $template = ScheduleTitle::findOrFail($request->title_id);
            $old_record = $template;
            $template->basis = $request->basis;
            $template->save();

            $this->logCustomMessage(
                'set_basis_schedule_template',
                $old_record,
                Auth::user()->name . ' Set Basis for schedule template',
                $template,
                ScheduleSetupLogType::SET_BASIS,
                new Activity()
            );

            $this->db->commit();            
            $text = $request->basiis == 2 ? $template->title ." has been set to hour-based." : "Flexible schedule has been set to schedule-based."; 

            return response()->json([
                'text'  => $text
            ]);

        } catch(ModelNotFoundException $me){
            DB::rollBack();
            $this->db->rollBack();
            return response()->json([
                'errors'    => ['The selected template could not be found.'],
                'message'   => $me->getMessage() 
            ]);
        } catch(ModelNotFoundException $e){
            $this->db->rollBack();
            return response()->json([
                'errors'    => ['There was a problem in setting the flexible type.'],
                'message'   => $e->getMessage() 
            ]);
        }

    }

    /**
     * Delete a particular schedule template DAY - not an entire schedule
     */
    public function delete(Request $request) {

        DB::beginTransaction();
        try {

            $schedule_template = ScheduleTemplate::findOrFail($request->id);
            $title = $schedule_template->schedule_title()->title;
            $schedule_template->delete();

            $this->logCustomMessage(
                'delete_schedule_template',
                $schedule_template,
                Auth::user()->name . ' Delete schedule template',
                $schedule_template,
                ScheduleSetupLogType::DELETE,
                new Activity()
            );
    
            DB::commit();
            return response()->json([
                'text'  => 'A schedule for ' . $title . ' has been deleted.'
            ]);

        } catch (ModelNotFoundException $me) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['Schedule does not exist.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem with deleting the schedule.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    } 

}
