<?php

namespace Suiterus\Adg\Controllers\SM;

use App\Models\User;
use Suiterus\Adg\Models\SM\SalaryStandardization;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Suiterus\Adg\Models\ServiceRecord\ServiceRecord;
use Suiterus\Adg\Models\SM\Remark;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\EMI\EmployeeMetaInfo;
use Suiterus\Adg\Models\Salary\UserSalary;
use Suiterus\Adg\Models\Salary\EmployeeAllowance;
use Suiterus\Adg\Models\SM\SalaryGrade;
use Suiterus\Adg\Models\Salary\PublicSalary;
use App\Enums\EmployeeTypeId;
use App\Enums\Status;
use Exception;
use Suiterus\Adg\Models\Activity\Activity;
use App\Traits\Logs\HasCustomLogs;
use App\Enums\Log\SalaryLogType;

class SalaryStandardizationController extends Controller
{
    use HasCustomLogs;
    public function create(Request $request) {

        $validate = Validator::make($request->all(), [
            'title'             => 'required|string|min:2|unique:adg_db.salary_standardizations,title,NULL,id,deleted_at,NULL',
            'effective_year'    => 'required',
            'status'            => 'required|integer',
        ], [
            'title.unique'      => $request->title . ' is already taken.'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $ss = SalaryStandardization::create([
                'title'             => $request->title,
                'effective_year'    => $request->effective_year,
                'status'            => $request->status,
                'employee_type_id'  => $request->employee_type_id,
                'premium_payment'   => $request->premium_payment ?? null,
                'created_by'        => $request->user()->id,
                'updated_by'        => $request->user()->id
            ]);

            if ($ss->employee_type_id == EmployeeTypeId::CONTRACTUAL) {
                for ($i = 1; $i <=33; $i++) {
                    $regular_sg = SalaryStandardization::where('employee_type_id', EmployeeTypeId::REGULAR)->where('active_tranche', Status::ACTIVE)->with(['salary_grades' => function($query) use($i) {
                        $query->where('salary_grade', $i)->where('step', 1);
                    }])->get();
                    
                    $rec = [];
                    $steps = [];
                    $grades = [];
                    foreach($regular_sg as $step) {
                        $record = [
                            'id'    => $step->id,
                            'step'  => $step->step,
                            'value' => $step->value
                        ];
                        array_push($steps, $record);
                    }
                
                    $rec['steps'] = $steps;
                    array_push($grades, $rec); 

                    $data = array();
                    $ss = SalaryStandardization::find($ss->id);
                    foreach($regular_sg as $salary_grade) {
                        foreach($salary_grade['salary_grades'] as $step) {
                            $regular_value = floatval(str_replace(',', '', $step['value']));
                            $premium_payment = $ss->premium_payment / 100 * $regular_value;
                            $total_salary = $premium_payment + $regular_value;
                            array_push($data, [
                                'ss_id'         => $ss->id,
                                'salary_grade'  => $step['salary_grade'],
                                'step'          => $step['step'],
                                'value'         => $total_salary,
                                'status'        => 1,
                                'created_by'    => $request->user()->id,
                                'updated_by'    => $request->user()->id,
                                'created_at'    => now(),
                                'updated_at'    => now(),
                            ]);
                        }
                    }
                    $ss->salary_grades()->insert($data);
                }
            }

            $this->logCustomMessage(
                'create_salary_tranche',
                $ss,
                Auth::user()->name . ' created a new tranche: ' . $ss->title,
                $ss,
                SalaryLogType::CREATE_TRANCHE,
                new Activity()
            );
 
            DB::commit();
            return response()->json([
                'text'  => $ss->title . ' has been created.',
            ]);
            
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in creating the record.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }

    }

    public function edit(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'                => 'required|integer',
            'title'             => 'required|string|min:2|unique:adg_db.salary_standardizations,title,'. $request->id.',id,deleted_at,NULL',
            'effective_year'    => 'required',
            'status'            => 'required|integer',
        ],[
            'title.unique'      => $request->title . ' is already taken.'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $ss = SalaryStandardization::findOrFail($request->id);
            $ss->update([
                'title'             => $request->title,
                'effective_year'    => $request->effective_year,
                'status'            => $request->status,
                'employee_type_id'  => $request->employee_type_id,
                'premium_payment'   => $request->premium_payment ?? null,
                'updated_by'        => $request->user()->id
            ]);
            $ss->save();

            if ($ss->employee_type_id == EmployeeTypeId::CONTRACTUAL) {
                for ($i = 1; $i <=33; $i++) {
                    $regular_sg = SalaryStandardization::where('employee_type_id', EmployeeTypeId::REGULAR)->where('active_tranche', Status::ACTIVE)->with(['salary_grades' => function($query) use($i) {
                        $query->where('salary_grade', $i)->where('step', 1);
                    }])->get();
                    
                    $rec = [];
                    $steps = [];
                    $grades = [];
                    foreach($regular_sg as $step) {
                        $record = [
                            'id'    => $step->id,
                            'step'  => $step->step,
                            'value' => $step->value
                        ];
                        array_push($steps, $record);
                    }
                
                    $rec['steps'] = $steps;
                    array_push($grades, $rec); 

                    $data = array();
                    $ss = SalaryStandardization::find($ss->id);
                    foreach($regular_sg as $salary_grade) {
                        foreach($salary_grade['salary_grades'] as $step) {
                            $regular_value = floatval(str_replace(',', '', $step['value']));
                            $premium_payment = $ss->premium_payment / 100 * $regular_value;
                            $total_salary = $premium_payment + $regular_value;
                            
                            SalaryGrade::where([['ss_id', $ss->id], ['salary_grade', $step['salary_grade']]])->update([
                                'value' => $total_salary
                            ]);
                        }
                    }
                }
            }
            $updatedTranche = SalaryStandardization::find($ss->id);

            $updatedTranche->old = collect($ss);
            $updatedTranche->attributes = collect($updatedTranche);
                    
            $this->logCustomMessage(
                'update_tranche',
                $updatedTranche,
                Auth::user()->name . ' updated the tranche: ' . $updatedTranche->title,
                $updatedTranche,
                SalaryLogType::UPDATE_TRANCHE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => $ss->title . ' has been updated.'
            ]);
            
        } catch(ModelNotFoundException $me) {
            DB::rollBack();
            return response()->json([
                'errors '   => ['The record does not exist.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in updating the record.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }

    }

    public function delete(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'    => 'required|integer'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $ss = SalaryStandardization::findOrFail($request->id);
            $title = $ss->title;
      
            $this->logCustomMessage(
                'delete_tranche',
                $ss,
                Auth::user()->name . ' deleted the tranche: ' . $title,
                $ss,
                SalaryLogType::DELETE_TRANCHE,
                new Activity()
            );

            $ss->delete();
            DB::commit();
            return response()->json([
                'text'  =>  $title . ' has been deleted.'
            ]);
            
        } catch(ModelNotFoundException $me) {
            DB::rollBack();
            return response()->json([
                'errors '   => ['The record does not exist.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in creating the record.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    public function set_active_tranche(Request $request) {
        $validate = Validator::make($request->all(), [
            'id'    => 'required|numeric',
            'emp_type_id'    => 'required|numeric'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            if($request->emp_type_id == EmployeeTypeId::REGULAR) {
                $current_tranche = SalaryStandardization::whereHas('employeeType', function($query){
                    $query->where('id', EmployeeTypeId::REGULAR);
                })->where('active_tranche', 1)->first();
            }else{
                $current_tranche = SalaryStandardization::whereHas('employeeType', function($query){
                    $query->where('id', EmployeeTypeId::JOB_ORDER)->orWhere('id', EmployeeTypeId::CONTRACTUAL);
                })->where('active_tranche', 1)->first();
            }
            
            $selected_tranche = SalaryStandardization::findOrFail($request->id);
            $selected_tranche->active_tranche = 1;

            if($current_tranche !== null) {
                $current_tranche->active_tranche = 0;
                $current_tranche->save();
            }
            $selected_tranche->save();

            $updatedTranche = SalaryStandardization::findOrFail($request->id);

            $updatedTranche->old = collect($selected_tranche);
            $updatedTranche->attributes = collect($updatedTranche);
                    
            $this->logCustomMessage(
                'active_tranche',
                $updatedTranche,
                Auth::user()->name . ' set as active tranche: ' . $updatedTranche->title,
                $updatedTranche,
                SalaryLogType::ACTIVE_TRANCHE,
                new Activity()
            );

            $user = User::where('status', '1')->orderBy('id', 'asc')->get()->pluck('id');
            
            $remark = Remark::whereId(10)->first();


            foreach($user as $data){

                $employeeInfo = EmployeeMetaInfo::where('user_id', $data)->first();

                $totalAllowance = 0;
            
                $allowance = EmployeeAllowance::where('user_id', $data)->get();
    
                foreach($allowance as $allowances) {
                    $totalAllowance += $allowances['amount'];
                }
    
                if($allowance === null){
                    $allAllowances = 0;
                }
                else{
                    $allAllowances = $totalAllowance;
                }

                $newSalary = SalaryGrade::whereHas('publicSalary', function($query) use($data){
                    $query->whereHas('userSalary', function($query) use($data){
                        $query->where([
                            ['user_id', $data]
                        ]);
                    }); //update to current tranche
                })->first();

                if($newSalary === null){

                    $annualSalary = 0;
                }
                else{
                    $salaryGrades = SalaryGrade::where('ss_id', $selected_tranche->id)->where('salary_grade', $newSalary->salary_grade)
                    ->where('step', $newSalary->step)->first();


                    $publicSalary = PublicSalary::whereHas('userSalary', function ($query) use($data){
                        $query->where('user_id', $data);
                    })->first();
    
                    if($publicSalary)
                    {
                        $publicSalary->update([
                            'salary_grade_id'  =>  $salaryGrades->id
                        ]);
                    }

                    $salary = $salaryGrades->value;
                    $salary = str_replace( ',', '', $salary);
                    $annualSalary = $salary * 12;
                }
                
                if($employeeInfo){                    
                    $oldServiceRecord = ServiceRecord::where('user_id', $data)->whereNull('end_date');
                    $endDate = date('Y-m-d', (strtotime(now())));
                    if($oldServiceRecord){
                        $oldServiceRecord->update([
                            'end_date'  => $endDate
                        ]);
                    }
                    
                    ServiceRecord::create([
                        'user_id'           =>  $data,
                        'start_date'        =>  now(),
                        'position_id'       =>  $employeeInfo->position_id,
                        'salary'            =>  $annualSalary,
                        'allowance'         =>  $allAllowances,
                        'employee_type_id'  =>  $employeeInfo->employee_type,
                        'division_id'       =>  $employeeInfo->division_id,
                        'remark_id'         =>  $remark->id,  //* Change of Status
                        'created_by'        =>  Auth::id(),
                    ]);
                }    
            }

            DB::commit();
            return response()->json([
                'text'  => $selected_tranche->title . ' has been set as the active tranche'
            ]);
            
        } catch(ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['The selected tranche could not be found.'],
                'message'   => $e->getMessage()
            ], 500);    
        } catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in selecting the tranche.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()

            ], 500);    
        }
    }

    public function fetch_inactive_tranches(Request $request) {
        $paginate = $request->paginate ? intval($request->paginate) : env('DEFAULT_PAGECOUNT');

        if($request->emp_type_id == EmployeeTypeId::CONTRACTUAL) {
            $inactive_tranches = SalaryStandardization::whereHas('employeeType', function($query){
                $query->where('id', EmployeeTypeId::CONTRACTUAL)->orWhere('id', EmployeeTypeId::JOB_ORDER);
            })->where('active_tranche', 0)->where('title', 'LIKE', '%'.$request->keyword.'%')->paginate($paginate);
        }else{
            $inactive_tranches = SalaryStandardization::whereHas('employeeType', function($query){
                $query->where('id', EmployeeTypeId::REGULAR);
            })->where('active_tranche', 0)->where('title', 'LIKE', '%'.$request->keyword.'%')->paginate($paginate);
        }

        return response()->json([
            'data' => $inactive_tranches
        ]); 
    }

    public function fetch_current_active(Request $request) {
        
        if($request->emp_type_id == EmployeeTypeId::CONTRACTUAL) {
            $active_tranche = SalaryStandardization::whereHas('employeeType', function($query){
                $query->where('id', EmployeeTypeId::CONTRACTUAL);
            })->where('active_tranche', 1)->first();
        }else{
            $active_tranche = SalaryStandardization::whereHas('employeeType', function($query){
                $query->where('id', EmployeeTypeId::REGULAR);
            })->where('active_tranche', 1)->first();
        }

        return response()->json([
            'data'  => $active_tranche
        ]);
    }

    public function fetch() {
        return response()->json([
            'data'  => SalaryStandardization::orderBy('title', 'desc')->get()
        ]); 
    }

    public function init_list() {
        return response()->json([
            'data'  => SalaryStandardization::orderBy('id', 'desc')->paginate(5)
        ]);
    }

    public function search(Request $request) {
        return response()->json([
            'data'  => SalaryStandardization::where('title', 'LIKE', '%'.$request->keyword.'%')->paginate(5)
        ]);
    }
}
