<?php

namespace Suiterus\Adg\Controllers\SM;
use Suiterus\Adg\Models\SM\SalaryStandardization;
use Suiterus\Adg\Models\SM\SalaryGrade;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Exception;
use Suiterus\Adg\Models\SM\EmployeeType;
use App\Enums\EmployeeTypeId;
use Suiterus\Adg\Models\Activity\Activity;
use App\Traits\Logs\HasCustomLogs;
use App\Enums\Log\SalaryLogType;
use Illuminate\Support\Facades\Auth;
class SalaryGradeController extends Controller
{
    use HasCustomLogs;

    public function create(Request $request) {

        $validate = Validator::make($request->all(), [
            'ss_id'             => 'required|integer',
            'salary_grade'      => 'required|integer',
            'step'              => 'required|integer',
            'value'             => 'required|numeric',
            'status'            => 'required|integer'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $ss = SalaryStandardization::findOrFail($request->ss_id);
            $salary_grade = $ss->salary_grades()->create([
                'salary_grade'      => $request->salary_grade,
                'step'              => $request->step,
                'value'             => $request->value,
                'status'            => $request->status,
                'created_by'        => $request->user()->id,
                'updated_by'        => $request->user()->id
            ]);

            $this->logCustomMessage(
                'add_salary_grade',
                $salary_grade,
                Auth::user()->name . ' added a value in salary grade: ' . $salary_grade->salary_grade.'.'.$salary_grade->step,
                $salary_grade,
                SalaryLogType::ADD_GRADE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => 'A salary grade for ' . $ss->title . ' has been created.',
            ]);

        } catch (ModelNotFoundException $me) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['Salary Standardization record does not exist.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errirs'    => ['There was a problem in creating the salary grade,'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    public function edit(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'                => 'required|integer',
            'ss_id'             => 'required|integer',
            'salary_grade'      => 'required|integer',
            'step'              => 'required|integer',
            'value'             => 'required|numeric',
            'status'            => 'required|integer'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $salary_grade = SalaryGrade::findOrFail($request->id);
            $salary_grade->update([
                'salary_grade'      => $request->salary_grade,
                'step'              => $request->step,
                'value'             => $request->value,
                'status'            => $request->status,
                'updated_by'        => $request->user()->id
            ]);
            $salary_grade->save();

            $updated_salary_grade = SalaryGrade::findOrFail($request->id);
            $updated_salary_grade->old = collect($updated_salary_grade);
            $updated_salary_grade->attributes = collect($updated_salary_grade);
                    
            $this->logCustomMessage(
                'update_salary_grade',
                $updated_salary_grade,
                Auth::user()->name . ' updated the value of salary grade: ' . $salary_grade->salary_grade.'.'.$salary_grade->step,
                $updated_salary_grade,
                SalaryLogType::UPDATE_GRADE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => 'Salary Grade for ' . $salary_grade->salary_standardization->title . ' has been updated.'
            ]);

        } catch(ModelNotFoundException $me) {
            DB::rollBack();
            return response()->json([
                'errors '   => ['The record does not exist.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in creating the record.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    public function batch_create(Request $request) {

        $validate = Validator::make($request->all(), [
            'ss_id'                            => 'required|integer',
            'salary_grades.*.salary_grade'     => 'required|numeric',
            'salary_grades.*.steps.*.step'     => 'required|integer',
            'salary_grades.*.steps.*.value'    => 'required|numeric'
        ],[
            'salary_grades.*.required'    => ':attribute is a required field.'
        ], [
            'salary_grades.*.salary_grade'     => 'salary grade',
            'salary_grades.*.steps.*.step'     => 'step',
            'salary_grades.*.steps.*.value'    => 'value'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $data = array();
            $ss = SalaryStandardization::findOrFail($request->ss_id);
            foreach($request->salary_grades as $salary_grade) {
                foreach($salary_grade['steps'] as $step) {
                    array_push($data, [
                        'ss_id'         => $request->ss_id,
                        'salary_grade'  => $salary_grade['salary_grade'],
                        'step'          => $step['step'],
                        'value'         => $step['value'],
                        'status'        => 1,
                        'created_by'    => $request->user()->id,
                        'updated_by'    => $request->user()->id,
                        'created_at'    => now(),
                        'updated_at'    => now(),
                    ]);
                }

            }
            $salary_grade = $ss->salary_grades()->insert($data);

            $this->logCustomMessage(
                'batch_add_salary_grade',
                $ss,
                Auth::user()->name . ' added the values in salary grade/s of ' . $ss->title,
                $ss,
                SalaryLogType::ADD_GRADE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => 'Salary Grades for ' . $ss->title . ' has been created.'
            ]);

        } catch (ModelNotFoundException $me) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['The salary standardization record does not exist.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in creating the records.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }

    }

    public function batch_update(Request $request) {

        $validate = Validator::make($request->all(), [
            'ss_id'                         => 'required|integer',
            'salary_grades.*.salary_grade'  => 'required|numeric',
            'salary_grades.*.steps.*.step'  => 'required|integer',
            'salary_grades.*.steps.*.value' => 'required|numeric'
        ],[
            'salary_grades.*.required'    => ':attribute is a required field.'
        ], [
            'salary_grades.*.salary_grade'  => 'salary grade',
            'salary_grades.*.steps.*.step'  => 'step',
            'salary_grades.*.steps.*.value' => 'value'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $ss = SalaryStandardization::find($request->ss_id);
            foreach($request->salary_grades as $salary_grade) {

                foreach($salary_grade['steps'] as $step) {
                    $ss->salary_grades()->where('id', $step['id'])->update([
                        'value'         => $step['value'],
                        'updated_at'    => now()
                    ]);                    
                }
            }
            $ss->save();

            $updated_salary_grade = SalaryGrade::findOrFail($ss->id);
            $updated_salary_grade->old = collect($updated_salary_grade);
            $updated_salary_grade->attributes = collect($updated_salary_grade);
                    
            $this->logCustomMessage(
                'batch_update_salary_grade',
                $updated_salary_grade,
                Auth::user()->name . ' updated the values in salary grade/s of ' . $ss->title,
                $updated_salary_grade,
                SalaryLogType::UPDATE_GRADE,
                new Activity()
            );

            $current_active = 1; //current active tranche
            
            if($ss->employee_type_id == EmployeeTypeId::REGULAR && $ss->active_tranche == $current_active) {

                $reg_sg = SalaryGrade::where('step', 1)->whereHas('salary_standardization', function($query) {
                    $query->where('employee_type_id', EmployeeTypeId::REGULAR)->where('active_tranche', $current_active);
                })->get();
                 
                $cos_sg = SalaryGrade::where('step', 1)->whereHas('salary_standardization', function($query) {
                    $query->where('employee_type_id', EmployeeTypeId::CONTRACTUAL)->where('active_tranche', $current_active);
                })->get();

                foreach($reg_sg as $salary_grade){
                    $cos_update = $cos_sg->where('salary_grade', $salary_grade['salary_grade'])->where('step', 1)->first();

                    $regular_value = floatval(str_replace(',', '', $salary_grade['value']));
                    $premium_payment = $cos_update->salary_standardization->premium_payment / 100 * $regular_value;
                    $total_salary = $premium_payment + $regular_value;
                    
                    $cos_update->value = $total_salary;
                    $cos_update->save();
                }      
            }

            DB::commit();
            return response()->json([
                'text'  => 'Salary Grades has been updated.'
            ]);

        } catch (ModelNotFoundException $me) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['The salary standardization record does not exist.'],
                'message'   => $me->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in creating the records.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }

    }

    public function delete(Request $request) {

        $validate = Validator::make($request->all(), [
            'id'                => 'required|integer',
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try {

            $salary_grade = SalaryGrade::findOrFail($request->id);
            $title = $salary_grade->salary_standardization->title;

            $this->logCustomMessage(
                'delete_salary_grade',
                $salary_grade,
                Auth::user()->name . ' deleted the salary grade: ' . $title,
                $salary_grade,
                SalaryLogType::DELETE_GRADE,
                new Activity()
            );
            $salary_grade->delete();
            DB::commit();
            return response()->json([
                'text'  =>  'The salary grade for ' . $title . ' has been deleted.'
            ]);

        } catch(ModelNotFoundException $me) {
            DB::rollBack();
            return response()->json([
                'errors '   => ['The record does not exist.'],
                'message'   => $me->getMessage()
            ], 500);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in creating the record.'],
                'message'   => $e->getMessage()
            ], 500);
        }

    }

    public function fetch_by_standardization(Request $request) {

        try {

            $ss = SalaryStandardization::findOrFail($request->ss_id);

            $grades = [];
            if(count($ss->salary_grades) > 0) {
                for ($i = 1; $i <=33; $i++) {
                    $sg = $ss->salary_grades()->where('salary_grade', $i)->get();
                    $rec = [];
                    $steps = [];
                    $rec['salary_grade'] = $i;
                    foreach($sg as $step) {
                        $record = [
                            'id'    => $step->id,
                            'step'  => $step->step,
                            'value' => $step->value
                        ];
                        array_push($steps, $record);
                    }
                    $rec['steps'] = $steps;
                    array_push($grades, $rec); 
                }
            } 

            return response()->json([
                'data'  => $grades
            ]);

        } catch(ModelNotFoundException $me) {
            return response()->json([
                'text'      => ['The standardization could not be found'],
                'message'   => $me->getMessage()
            ], 500);
        } catch(Exception $e) {
            return response()->json([
                'text'      => ['There was a problem in getting the records'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500); 
        }

    }

    public function init_list_regular() {
        return response()->json([
            'data'  => SalaryStandardization::whereHas('employeeType', function($query) {
                $query->where('id', EmployeeTypeId::REGULAR);
            })->orderBy('id', 'desc')->paginate(5)
        ]);
    }

    public function init_list_contractual() {
        return response()->json([
            'data'  => SalaryStandardization::whereHas('employeeType', function($query) {
                $query->where('id', EmployeeTypeId::JOB_ORDER)->orWhere('id', EmployeeTypeId::CONTRACTUAL);
            })->orderBy('id', 'desc')->paginate(5)
        ]);
    }

    public function search_regular(Request $request) {
        return response()->json([
            'data'  => SalaryStandardization::whereHas('employeeType', function($query) {
                $query->where('id', EmployeeTypeId::REGULAR);
            })->where('title', 'LIKE', '%'.$request->keyword.'%')->paginate(5)
        ]);
    }

    public function search_contractual(Request $request) {
        return response()->json([
            'data'  => SalaryStandardization::whereHas('employeeType', function($query) {
                $query->where('id', EmployeeTypeId::JOB_ORDER)->orWhere('id', EmployeeTypeId::CONTRACTUAL);
            })->where('title', 'LIKE', '%'.$request->keyword.'%')->paginate(5)
        ]);
    }

}
