<?php

namespace Suiterus\Adg\Controllers\SM;

use Exception;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use App\Enums\Log\RosterLogType;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use Suiterus\Hrjp\Models\Position;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\Enums\Log\RosterSetupLogType;
use Suiterus\Adg\Models\SM\RosterGroup;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\SM\DesignationHistory;
use Suiterus\Adg\Services\Roster\RosterService;
use Suiterus\Adg\Models\Salary\EmployeeAllowance;
use Suiterus\Adg\Models\ServiceRecord\ServiceRecord;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use Suiterus\Adg\Models\SM\Remark;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterTransferDate;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterEmployeeGroup;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterTransferRemarks;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterEmployeesPerGroup;

class RosterGroupController extends Controller
{
    use HasCustomLogs;
    public function create_group(Request $request)
    {

        $valid = Validator::make($request->all(), [
            'name'    =>  'required|string|unique:adg_db.roster_groups,name,NULL,id,deleted_at,NULL',
            'description'   =>  'string|nullable',
            'status' => 'integer|required'
        ], [
            'name.unique'   =>  $request->name . ' is already taken.',
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {

            $data =  RosterGroup::create([
                'name'  =>   $request->name,
                'description'   =>  $request->description,
                'status'    => $request->status ? $request->status : 1
            ]);

            $this->logCustomMessage(
                'create_roster_group',
                $data,
                Auth::user()->name . ' Create roster group',
                $data,
                RosterSetupLogType::CREATE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  =>  $request->name . ' created successfully.'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01'],
                'msg'   =>  $e->getMessage()
            ], 500);
        }
    }

    public function init_list_group()
    {
        return response()->json([
            'data'  =>  RosterGroup::orderBy('id', 'desc')->paginate(5)
        ]);
    }

    public function search_group_name(Request $request)
    {
        return response()->json([
            'data'  =>  RosterGroup::where('name', 'LIKE', '%' . $request->keyword . '%')->paginate(5)
        ]);
    }

    public function update_group(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'name'      =>  'required|string|unique:adg_db.roster_groups,name,' . $request->id . ',id,deleted_at,NULL',
            'status'    => 'integer|required'
        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {
            try {
                $group = RosterGroup::findOrFail($request->id);
                $old_record = $group;
                $group->update([
                    'name'  =>   $request->name,
                    'description'   =>  $request->description,
                    'status'    => $request->status
                ]);

                $group->save();

                $this->logCustomMessage(
                    'update_roster_group',
                    $old_record,
                    Auth::user()->name . ' Update roster group',
                    $group,
                    RosterSetupLogType::UPDATE,
                    new Activity()
                );

                DB::commit();
                return response()->json([
                    'text'  =>  $request->name . ' has been updated.'
                ]);
            } catch (ME $ee) {
                DB::rollback();
                return response()->json([
                    'errors'    =>  ['Roster Group doesn`t exists.'],
                ], 400);
            }
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in updating a Roster Group.'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }

    public function delete_group(Request $request)
    {
        DB::beginTransaction();
        try {
            $perm = RosterGroup::findOrFail($request->id);
            $perm->delete();

            $this->logCustomMessage(
                'delete_roster_group',
                $perm,
                Auth::user()->name . ' Delete roster group',
                $perm,
                RosterSetupLogType::DELETE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  =>  'Roster Group has been deleted.'
            ]);
        } catch (ME $ee) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Roster Group doesn`t exists.'],
            ], 400);
        }
    }


    //* Functions for grouping positions
    public function assign_group(Request $request)
    {

        $validate = Validator::make($request->all(), [
            'position_id'  =>    'required|exists:hrjp_db.positions,id',
            'group_id'     =>    'required|exists:adg_db.roster_groups,id'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();

        try {

            foreach ($request->position_id as $position) {

                $data = Position::whereId($position)->first();

                $data->update([
                    'group_id'  =>  $request->group_id,
                    'updated_by'    =>  Auth::id()
                ]);

                $data->save();
            }

            $this->logCustomMessage(
                'assign_position_group',
                null,
                Auth::user()->name . ' Assign position group',
                null,
                RosterSetupLogType::ASSIGN,
                new Activity()
            );

            DB::commit();

            return response()->json([
                'success'   => true,
                'text'      => 'Record have been successfully updated.'
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem with assigning a group.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ]);
        }
    }

    public function roster_groups()
    {
        return response()->json([
            'data'  => RosterGroup::all()
        ]);
    }

    public function fetch_group_positions(Request $request)
    {
        return response()->json([
            'data'  =>  Position::whereNotNull('group_id')->when($request->title != null && isset($request->title) && $request->title != '', function ($query) use ($request) {
                $query->whereHas('rosterGroup', function ($query) use ($request) {
                    $query->where('title', 'LIKE', '%' . $request->title . '%');
                });
            })->with('rosterGroup')->paginate(5)
        ]);
    }

    public function edit_grouped_position(Request $request)
    {
        $valid = Validator::make($request->all(), [
            'id'      =>  'required|integer|unique:hrjp_db.positions,id,' . $request->id . ',id,deleted_at,NULL',

        ]);
        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }
        DB::beginTransaction();
        try {
            try {
                $data = Position::findOrFail($request->id);
                $old_record = $data;
                $data->group_id = $request->group_id;
                $data->updated_by = Auth::id();

                $data->save();

                $this->logCustomMessage(
                    'update_grouped_position',
                    $old_record,
                    Auth::user()->name . ' Update grouped position',
                    $data,
                    RosterSetupLogType::UPDATE,
                    new Activity()
                );

                DB::commit();
                return response()->json([
                    'text'  =>  'Group has been updated.'
                ]);
            } catch (ME $ee) {
                DB::rollback();
                return response()->json([
                    'errors'    =>  ['ID doesnt exists.'],
                ], 400);
            }
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['There is a problem in updating a record.'],
                'msg'       =>  $e->getMessage()
            ], 500);
        }
    }

    public function delete_grouped_position(Request $request)
    {
        DB::beginTransaction();
        try {
            $data = Position::findOrFail($request->id);
            $data->update([
                'group_id' => null
            ]);

            $this->logCustomMessage(
                'delete_grouped_position',
                $data,
                Auth::user()->name . ' Delete grouped position',
                $data,
                RosterSetupLogType::DELETE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  =>  'Record has been deleted.'
            ]);
        } catch (ME $ee) {
            DB::rollback();
            return response()->json([
                'errors'    =>  ['Record doesnt exists.'],
            ], 400);
        }
    }

    public function fetch_positions()
    {
        return response()->json([
            'data'  =>  Position::whereNull('group_id')->get()
        ]);
    }

    public function assign_employee(Request $request, RosterService $rosterService)
    {
        DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();
        try {
            $rosterEmployeeGroup = RosterEmployeeGroup::find($request->id);
            $rosterService->setRoster($rosterEmployeeGroup->roster_id);
            foreach ($request->employees as $employee) {
                $user = User::find($employee);
                $rosterService->addEmployeeToGroup($employee, $request->id);

                $this->logCustomMessage(
                    'assign_employee_roster_group',
                    $rosterEmployeeGroup,
                    Auth::user()->name . " assign {$user->name} to roster group {$rosterEmployeeGroup->rosterGroup->name}",
                    $rosterEmployeeGroup,
                    RosterLogType::ASSIGN_EMPLOYEE_GROUP,
                    new Activity()
                );
            }

            DB::connection(env('ADG_DB_CONNECTION'))->commit();
            return response()->json([
                'text'  =>  'Employees have been assigned.'
            ]);
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollBack();
            return response()->json([
                'errors'    => ['There was a problem with assigning a group.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }

    public function store(Request $request, RosterService $rosterService)
    {
        DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();
        try {

            $rosterGroup = RosterGroup::where('name', $request->name)->first();

            $group = RosterGroup::updateOrCreate(
                [
                    'name'  =>   $request->name
                ],
                [
                    'name'  =>   $request->name,
                    'status'    => $request->status ? $request->status : 1
                ]
            );

            $rosterEmployeeGroups = $rosterService->setRoster($request->roster_id)->createGroups([
                [
                    'group_id' => $group->id,
                    'description' => null
                ]
            ])->getGroups();

            foreach ($rosterEmployeeGroups as $group) {
                foreach ($request->employees as $employee) {
                    $rosterService->addEmployeeToGroup($employee, $group->id);
                    $user = User::find($employee);
                    $this->logCustomMessage(
                        'assign_employee_roster_group',
                        $group,
                        Auth::user()->name . " assign {$user->name} to roster {$group->roster->title} group {$group->rosterGroup->name}",
                        $group,
                        RosterLogType::ASSIGN_EMPLOYEE_GROUP,
                        new Activity()
                    );
                }
            }

            if (!$rosterGroup) {
                $this->logCustomMessage(
                    'create_roster_group',
                    $group,
                    Auth::user()->name . " create roster {$rosterService->getRoster()->title} group {$request->name}",
                    $group,
                    'Create roster group',
                    new Activity()
                );
            }

            DB::connection(env('ADG_DB_CONNECTION'))->commit();
            return response()->json([
                'message'   => 'Success',
            ]);
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollBack();
            return response()->json([
                'errors'    => ['There was a problem with assigning a group.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }

    public function transfer_employee(Request $request, RosterService $rosterService)
    {

        DB::beginTransaction();
        try {
            $oldEmployeeGroup = RosterEmployeesPerGroup::find($request->old_employee_group_id);

            $metaInfo = $oldEmployeeGroup->user->employeeMetaInfo;

            $rosterEmployeeGroup = RosterEmployeeGroup::where([
                ['roster_id', $request->new['roster_id']],
                ['group_id', $request->new['group_id']],
            ])->first();

            $newRosterGroup = $rosterService->setRoster($request->new['roster_id'])->addEmployeeToGroup($oldEmployeeGroup->user_id, $rosterEmployeeGroup->id)->getEmployeeGroup();

            foreach ($request->partitions as $partition) {
                $remarks = RosterTransferRemarks::create([
                    'old_employee_group_id' => $request->old_employee_group_id,
                    'new_employee_group_id' => $newRosterGroup->id,
                    'remarks' => $partition['remarks'],
                    'created_by' => Auth::id()
                ]);

                foreach ($partition['days'] as $day) {
                    RosterTransferDate::create([
                        'roster_transfer_remarks_id' => $remarks->id,
                        'date' => $day['date'],
                        'created_by' => Auth::id()
                    ]);
                }

                $oldEmployeeShifts = $oldEmployeeGroup->dayEmployeeShifts()->whereHas('rosterDay', function ($query) use ($partition) {
                    $query->where([
                        ['roster_partition_id', $partition['id']],
                    ]);
                })->get();

                $rosterService->removeEmployeeShifts($oldEmployeeShifts->pluck('id'));
            }

            $rosterDepartment = $rosterService->getRoster()->department_id;
            $rosterDivision = $rosterService->getRoster()->division_id;

            if ($metaInfo->department_id != $rosterDepartment || $metaInfo->division_id != $rosterDivision) {
                DesignationHistory::where('user_id', $oldEmployeeGroup->user_id)->update([
                    'status' => 2
                ]);

                $designationDate = Carbon::now();

                DesignationHistory::create([
                    'user_id' => $oldEmployeeGroup->user_id,
                    'division_id' =>  $rosterDivision,
                    'department_id' => $rosterDepartment,
                    'date_of_designation' => $designationDate,
                    'status' => 1,
                    'created_by' => Auth::id(),
                ]);

                $metaInfo->update([
                    'department_id' => $rosterDepartment,
                    'division_id' => $rosterDivision,
                ]);
            }

            $newRosterGroup->attributes = collect($newRosterGroup);
            $newRosterGroup->old = collect($oldEmployeeGroup);

            $this->logCustomMessage(
                'transfer_employee_roster_group',
                $newRosterGroup,
                Auth::user()->name . " transfer {$oldEmployeeGroup->user->name} to roster {$newRosterGroup->employeeGroup->roster->title} group {$newRosterGroup->employeeGroup->rosterGroup->name}",
                $newRosterGroup,
                RosterLogType::TRANSFER_EMPLOYEE_GROUP,
                new Activity()
            );

            DB::commit();
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem with transferring an employee'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }
}
