<?php

namespace Suiterus\Adg\Controllers\SM;

use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Suiterus\Hrjp\Models\Position;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use Suiterus\Adg\Models\Activity\Activity;
use App\Traits\Logs\HasCustomLogs;
use App\Enums\Log\PositionLogType;
use Illuminate\Support\Facades\Auth;

class PositionController extends Controller
{
    use HasCustomLogs;


    public function index(){
        return Position::all();
    }

    public function create_position(Request $req) {

        $valid = Validator::make($req->all(),[
            'title'    =>  'required|string|unique:hrjp_db.positions,title,NULL,id,deleted_at,NULL'

        ],[
            'title.unique'   =>  $req->title.' is already taken.',
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try {
            $position = Position::create([
                'title' => $req->title,
                'created_by' => $req->user()->id
            ]);

            $this->logCustomMessage(
                'create_position',
                $position,
                Auth::user()->name . ' created a position: ' . $position->title,
                $position,
                PositionLogType::CREATE,
                new Activity()
            );

            DB::commit();
            $user = $req->user();
            return response()->json([
                'text'  =>  $req->title.' created successfully!'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01' ],
                'msg'   =>  $e->getMessage()
            ],500);
        }
    }

    public function init_list_position(Request $req){
        $paginate = $req->page_count ? intval($req->page_count) : env('DEFAULT_PAGECOUNT');
        return response()->json([
            'data'  =>  Position::orderBy('id', 'DESC')->paginate($paginate)
        ]);
    }

    public function list_all_soft_deleted(){
        return Position::onlyTrashed()->get();
    }

    public function search_soft_delete(Request $req){
        return response()->json([
            'data'  =>  Position::onlyTrashed()->where('id', '=', $req->id)->get()
        ]);
    }

    public function search_soft_keyword(Request $req){
        $paginate = $req->page_count ? intval($req->page_count) : env('DEFAULT_PAGECOUNT');
        return response()->json([
            'data'  =>  Position::onlyTrashed()->where('title', 'LIKE', '%'.$req->keyword.'%')->paginate($paginate)
        ]);
    }

    public function restore_soft_delete(Request $req){
        return response()->json([
            Position::onlyTrashed()->where('id', '=', $req->id)->restore()
        ]);
    }

    public function search_position_name(Request $req){
        $paginate = $req->page_count ? intval($req->page_count) : env('DEFAULT_PAGECOUNT');
        return response()->json([
            'data'  =>  Position::where('title', 'LIKE', '%'.$req->keyword.'%')->paginate($paginate)
        ]);
    }

    public function edit_position(Request $req){
        $valid = Validator::make($req->all(),[
            'user_id'      =>  'required|string|min:4|unique:hrjp_db.positions,user_id,'.$req->id.',id,deleted_at,NULL',

        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try{
            try{
                $perms = Position::findOrFail($req->id);
                $perms->title = $req->title;
                $perms->created_by = $req->user()->id;
                $perms->updated_by = $req->user()->id;

                $perms->save();
                DB::commit();
                return response()->json([
                    'text'  =>  $req->title.' has been updated.'
                ]);
            }catch(ME $ee){
                DB::rollback();
                return response()->json([
                    'errors'    =>  [ 'Position doesnt exists.' ],
                ],400);
            }
        }catch(Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'There is a problem in updating a Position.' ],
                'msg'       =>  $e->getMessage()
            ],500);
        }
    }

    public function delete_position(Request $req){
        DB::beginTransaction();
        try{
            $perm = Position::findOrFail($req->id);
            $perm->delete();
            DB::commit();
            return response()->json([
                'text'  =>  'Position has been deleted.'
            ]);
        }catch(ME $ee){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Position doesnt exists.' ],
            ],400);
        }
    }

}
