<?php

namespace Suiterus\Adg\Controllers\SM;

use Exception;
use App\Models\User;
use Illuminate\Http\Request;
use App\Enums\Log\OfficeLogType;
use Suiterus\Adg\Models\SM\Unit;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\DB;
use Suiterus\Adg\Models\SM\Office;
use Illuminate\Support\Facades\Log;
use Suiterus\Adg\Models\SM\Section;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\SM\Division;
use Suiterus\Adg\Models\SM\Department;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\ActualDesignation;
use Suiterus\Adg\Models\SM\OfficeHierarchy;
use Suiterus\Adg\Models\EMI\EmployeeMetaInfo;
use Suiterus\Adg\Models\SM\OfficePerEmployee;
use Suiterus\Adg\Services\OrgStructure\OrgStructureService;
use Suiterus\Adg\Requests\Offices\StoreUpdateOfficesRequest;
use Suiterus\Adg\Services\OrgStructure\Office\OfficeService;

class OfficesController extends Controller
{
    use HasCustomLogs;

    private $officeService;
    private $orgStructureService;

    public function __construct(OfficeService $officeService, OrgStructureService $orgStructureService)
    {
        $this->officeService = $officeService;
        $this->orgStructureService = $orgStructureService;
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        return Office::get();
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  StoreUpdateOfficesRequest  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StoreUpdateOfficesRequest $request)
    {

        try {
            return DB::transaction(function () use ($request) {
                return DB::connection(env('ADG_DB_CONNECTION'))->transaction(function () use ($request) {
                    $office = Office::create([
                        'name' => $request->name,
                        'description' => $request->description,
                        'head_employee_id' => $request->head_employee_id,
                        'status' => $request->status,
                        'created_by' => Auth::id(),
                    ]);

                    if (isset($request->pcfr_configuration)) {
                        $this->orgStructureService->configPcr($office, $request->pcfr_configuration);
                    }

                    if (isset($request->minimum_overtime)) {
                        $this->orgStructureService->configMinimumOvertime($office, $request->minimum_overtime);
                    }

                    $this->officeService->assignOffice(
                        $request->parent_office_id,
                        $request->child_office_ids,
                        $request->department_ids,
                        $request->user_ids,
                        $office
                    );

                    if (isset($request->units)) {
                        $this->officeService->assignUnits($request->units, $office);
                    }

                    if (isset($request->division_ids)) {
                        $this->officeService->assignDivisions($request->division_ids, $office);
                    }

                    $this->logCustomMessage(
                        'create_office',
                        $office,
                        Auth::user()->name . ' created a new Office: ' . $request->name,
                        $office,
                        OfficeLogType::CREATE,
                        new Activity()
                    );

                    return response()->json([
                        'text' => 'New office has been created.'
                    ]);
                });
            });
        } catch (Exception $e) {
            return response()->json([
                'errors'    => ['The request could not be process.'],
                'message'   => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Office   $office
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request, $officeId)
    {
        $validate = Validator::make(['id' => $officeId], [
            'id'  =>  'required|exists:' . env('ADG_DB_CONNECTION') . '.offices,id',
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors'    =>  $validate->errors()
            ], 400);
        }

        $office = Office::where('id', $officeId)->without('childOffices')->first();

        $office->department = $office->department->map(function ($department) use($request) {

            $department->evaluation_score = $department->evaluation()->when($request->start_date && $request->end_date, function ($query) use ($request) {
                $query->whereDate('start_date', $request->start_date)->whereDate('end_date', $request->end_date);
            })->avg('overall_rate');
            $department->evaluation_score_adjective = $this->orgStructureService->getEvaluationScoreAdjective($department->evaluation_score);

            $department->divisions = $this->orgStructureService->evaluationScoreMapper($department->divisions, $request->start_date, $request->end_date);
            $department->units = $this->orgStructureService->evaluationScoreMapper($department->units, $request->start_date, $request->end_date);

            return $department;
        });

        $office->units = $this->orgStructureService->evaluationScoreMapper($office->units, $request->start_date, $request->end_date);
        $office->divisions = $this->orgStructureService->evaluationScoreMapper($office->divisions, $request->start_date, $request->end_date);

        $childOffices = $office->childOffices()->without(['childOffices', 'department'])->get()->map(function ($office) {
            return $office->childOffice()->with(['units', 'divisions'])->first();
        });

        $childOffices = $childOffices->map(function ($office) use ($request) {
            $office->evaluation_score = $office->evaluation()->when($request->start_date && $request->end_date, function ($query) use ($request) {
                $query->whereDate('start_date', $request->start_date)->whereDate('end_date', $request->end_date);
            })->avg('overall_rate');
            $office->evaluation_score_adjective = $this->orgStructureService->getEvaluationScoreAdjective($office->evaluation_score);

            $office->department = $office->department->map(function ($department) use ($request) {

                $department->evaluation_score = $department->evaluation()->when($request->start_date && $request->end_date, function ($query) use ($request) {
                    $query->whereDate('start_date', $request->start_date)->whereDate('end_date', $request->end_date);
                })->avg('overall_rate');
                $department->evaluation_score_adjective = $this->orgStructureService->getEvaluationScoreAdjective($department->evaluation_score);
                return $department;
            });

            return $office;
        });

        $office->child_offices = $childOffices;

        return $office;
    }

    public function showEmployees($officeId)
    {
        return User::whereHas('actualDesignation', function ($query) use ($officeId) {
            $query->where('office_id', $officeId)->whereNull(['department_id']);
        })->without([
            'currentRole',
            'roles',
            'permissions',
            'storage',
            'employeeMetaInfo',
            'supervisor',
            'user_supervisor',
            'exitInterview',
            'userProfilePicture',
            'profileBasicInfo'
        ])->get();
    }

    public function search(Request $request)
    {
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');

        $offices =  Office::where('name', 'LIKE', '%' . $request->keyword . '%')->with(['pcrs', 'units', 'divisions', 'overtime'])->paginate($paginate);

        foreach ($offices as $office) {
            $parentOffice = OfficeHierarchy::where('parent_office_id', $office->id)
                ->without(['parentOffice'])->with(['childOffice'])
                ->get();

            if ($parentOffice) {
                $office->setAttribute('childOffices', $parentOffice);
            }

            $office->setAttribute('parentOffice', OfficeHierarchy::where('child_office_id', $office->id)
                ->without(['childOffice'])->with(['parentOffice'])
                ->get());
        }

        return $offices;
    }

    /**
     * The function paginate takes a request object and returns a paginated list of offices filtered by
     * a keyword, with each office having its child offices and parent office attributes set.
     *
     * @param Request request is an instance of the `Illuminate\Http\Request`
     * class. It represents the HTTP request made to the server and contains information such as the
     * request method, headers, query parameters, and request body.
     *
     * @return a paginated collection of offices.
     */
    public function paginate(Request $request)
    {

        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');
        $offices = Office::where('name', 'LIKE', '%' . $request->keyword . '%')->with(['pcrs', 'units', 'divisions', 'overtime'])->paginate($paginate);

        foreach ($offices as $office) {
            $parentOffice = OfficeHierarchy::where('parent_office_id', $office->id)
                ->without(['parentOffice'])->with(['childOffice'])
                ->get();

            if ($parentOffice) {
                $office->setAttribute('childOffices', $parentOffice);
            }

            $office->setAttribute('parentOffice', OfficeHierarchy::where('child_office_id', $office->id)
                ->without(['childOffice'])->with(['parentOffice'])
                ->get());
        }

        return $offices;
    }

    /**
     * The function updates an office record in the database along with its related departments, office
     * hierarchy, and office per employee records.
     *
     * @param StoreUpdateOfficesRequest request The  parameter is an instance of the
     * StoreUpdateOfficesRequest class, which is used to validate and retrieve the data sent in the
     * request.
     * @param officeId The `officeId` parameter is the ID of the office that needs to be updated. It is
     * used to identify the specific office record in the database that needs to be updated.
     *
     * @return a JSON response. If the update is successful, it returns a JSON response with the
     * message "Office has been updated." If there is an error, it returns a JSON response with the
     * error message and a status code of 500.
     */
    public function update(StoreUpdateOfficesRequest $request, $officeId)
    {
        DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();

        try {

            $office = Office::findOrFail($officeId);
            $office->update([
                'name' => $request->name,
                'description' => $request->description,
                'head_employee_id' => $request->head_employee_id,
                'status' => $request->status,
                'updated_by' => Auth::id(),
            ]);

            if (isset($request->pcfr_configuration)) {
                $this->orgStructureService->configPcr($office, $request->pcfr_configuration);
            }

            if (isset($request->minimum_overtime)) {
                $this->orgStructureService->configMinimumOvertime($office, $request->minimum_overtime);
            }

            $updated = Office::find($officeId);
            $updated->old = collect($office);
            $updated->attributes = collect($updated);

            $this->logCustomMessage(
                'update_office',
                $updated,
                Auth::user()->name . ' updated the Office: ' . $request->name,
                $updated,
                OfficeLogType::UPDATE,
                new Activity()
            );

            $this->officeService->deleteOfficeRelations($office);

            $this->officeService->assignOffice(
                $request->parent_office_id,
                $request->child_office_ids,
                $request->department_ids,
                $request->user_ids,
                $office
            );

            $office->units()->update([
                'office_id' => null,
            ]);

            if (isset($request->units)) {
                $this->officeService->assignUnits($request->units, $office);
            }

            if (isset($request->division_ids)) {
                $this->officeService->assignDivisions($request->division_ids, $office);
            }

            DB::connection(env('ADG_DB_CONNECTION'))->commit();

            return response()->json([
                'text' => 'Office has been updated.'
            ]);
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollBack();
            return response()->json([
                'errors'    => ['The request could not be process.'],
                'message'   => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Office  $office
     * @return \Illuminate\Http\Response
     */
    public function delete(Request $request)
    {
        DB::connection(env('ADG_DB_CONNECTION'))->beginTransaction();

        $valid = Validator::make($request->all(), [
            'id' => 'required|integer'
        ]);

        if ($valid->fails()) {
            return response()->json([
                'errors'    =>  $valid->errors()
            ], 400);
        }

        try {

            $office = Office::findOrFail($request->id);

            $hasChildOffices = OfficeHierarchy::where('parent_office_id', $office->id)->first();
            if ($hasChildOffices) {
                return response()->json([
                    'errors' => ['Cannot delete office with sub-offices.'],
                ], 400);
            }

            $this->officeService->deleteOfficeRelations($office);

            $this->logCustomMessage(
                'delete_office',
                $office,
                Auth::user()->name . ' deleted the Office: ' . $office->name,
                $office,
                OfficeLogType::DELETE,
                new Activity()
            );

            $office->delete();
            DB::connection(env('ADG_DB_CONNECTION'))->commit();

            return response()->json([
                'text'  =>  'Office has been deleted.',
            ]);
        } catch (Exception $e) {
            DB::connection(env('ADG_DB_CONNECTION'))->rollback();
            return response()->json([
                'errors'    =>  ['The request could not be process.'],
                'msg'   =>  $e->getMessage()
            ], 500);
        }
    }

    public function showDepartments(Request $request, $officeId)
    {
        $departments = Department::where('office_id', $officeId)->without(['divisions'])->with(['units'])->get();

        $departments = $departments->map(function ($department) use ($request) {

            $department->evaluation_score = $department->evaluation()->when($request->start_date && $request->end_date, function ($query) use ($request) {
                $query->whereDate('start_date', $request->start_date)->whereDate('end_date', $request->end_date);
            })->avg('overall_rate');
            $department->evaluation_score_adjective = $this->orgStructureService->getEvaluationScoreAdjective($department->evaluation_score);

            $department->divisions = $department->divisions()->get()->map(function ($division) use ($request) {
                $division->evaluation_score = $division->evaluation()->when($request->start_date && $request->end_date, function ($query) use ($request) {
                    $query->whereDate('start_date', $request->start_date)->whereDate('end_date', $request->end_date);
                })->avg('overall_rate');
                $division->evaluation_score_adjective = $this->orgStructureService->getEvaluationScoreAdjective($division->evaluation_score);
                return $division;
            });
            return $department;
        });

        return $departments;
    }

    public function showUnits(Request $request, $officeId)
    {
        $units = Unit::where('office_id', $officeId)->get()->append('organization_type');

        $units = $this->orgStructureService->evaluationScoreMapper($units, $request->start_date, $request->end_date);
        return $units;
    }

    public function showDivisionsAndUnits($officeId){

        $divisions = Division::where('office_id', $officeId)->without(['sections'])->get()->append('organization_type');

        $units = Unit::where('office_id', $officeId)->orWherein('division_id', $divisions->pluck('id')->toArray())->get()->append('organization_type');

        return [...$divisions, ...$units];
    }

    public function showDivisions($officeId){
        return Division::where('office_id', $officeId)->without(['sections'])->get()->append('organization_type');
    }
}
