<?php

namespace Suiterus\Adg\Controllers\SM;

use Exception;
use App\Enums\Status;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Traits\Logs\HasCustomLogs;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;
use App\Enums\Log\NightDifferentialPayLogType;
use Suiterus\Adg\Models\Payroll\NightDifferentialPay;

class NightDifferentialPayController extends Controller
{
    use HasCustomLogs;

    public function index(){
        return response()->json([
            'data'  => NightDifferentialPay::orderBy('status', 'asc')->paginate(5)
        ]);
    }

    public function store(Request $request) {
        $validate = Validator::make($request->all(), [
            'percentage'    =>  'required',
            'start_time'    =>  'required',
            'end_time'    =>  'required',
            'effectivity_date'  =>  'required'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors' => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try{
            $night_diff_pay = NightDifferentialPay::create([
                'percentage' => $request->percentage,
                'range_start_time' => $request->start_time,
                'range_end_time' => $request->end_time,
                'effectivity_date' => $request->effectivity_date,
                'status'    =>  Status::INACTIVE
            ]);

            $this->logCustomMessage(
                'create_night_differential_pay_percentage',
                $night_diff_pay,
                Auth::user()->name . ' created a new night differential pay percentage: ' . $night_diff_pay->percentage . '%',
                $night_diff_pay,
                NightDifferentialPayLogType::CREATE_SETUP,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => __('responses.success.create')
            ]);
        } catch(Exception $e){
            DB::rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }

    public function update(Request $request) {
        $validate = Validator::make($request->all(), [
            'id'    => 'required|integer',
            'percentage'    =>  'required',
            'start_time'    =>  'required',
            'end_time'    =>  'required',
            'effectivity_date'  =>  'required'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors' => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try{
            
            $night_diff_pay = NightDifferentialPay::find($request->id);
            $oldData = clone $night_diff_pay;
            $night_diff_pay->whereId($request->id)->update([
                'percentage' => $request->percentage,
                'range_start_time' => $request->start_time,
                'range_end_time' => $request->end_time,
                'effectivity_date' => $request->effectivity_date
            ]);

            $night_diff_pay->old = collect($oldData);
            $night_diff_pay->attributes = collect($night_diff_pay);

            $this->logCustomMessage(
                'update_night_differential_pay_percentage',
                $night_diff_pay,
                Auth::user()->name . ' updated the night differential pay percentage: ' . $night_diff_pay->percentage . '%',
                $night_diff_pay,
                NightDifferentialPayLogType::UPDATE_SETUP,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => __('responses.success.update')
            ]);
        } catch(Exception $e){
            DB::rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }

    public function get_active() {
        return NightDifferentialPay::where('status', Status::ACTIVE)->first();
    }

    public function set_as_active(Request $request) {
        $validate = Validator::make($request->all(), [
            'id'    => 'required|numeric'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();

        try{
            $data = NightDifferentialPay::findOrFail($request->id);
            $oldData = clone $data;
            $active = NightDifferentialPay::where('status', Status::ACTIVE)->first();

            $data->update(['status' => Status::ACTIVE]);
            if($active){
                $active->update(['status' => Status::INACTIVE]);
            }

            $data->old = collect($oldData);
            $data->attributes = collect($data);

            $this->logCustomMessage(
                'active_night_differential_pay_percentage',
                $data,
                Auth::user()->name . ' set as active night differential pay percentage: ' . $data->percentage . '%',
                $data,
                NightDifferentialPayLogType::ACTIVE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => 'Successfully set as the active night differential pay percentage'
            ]);
        } catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in selecting the night differential pay percentage.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);  
        }
    }
}
