<?php

namespace Suiterus\Adg\Controllers\SM;

use App\Enums\Log\LongevityPayLogType;
use Exception;
use App\Enums\Status;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Traits\Logs\HasCustomLogs;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\Payroll\Longevity;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Suiterus\Adg\Models\Activity\Activity;

class LongevityController extends Controller
{
    use HasCustomLogs;

    public function index(Request $request){
        return response()->json([
            'data'  => Longevity::where('status', Status::INACTIVE)->paginate(5)
        ]);
    }

    public function store(Request $request) {
        $validate = Validator::make($request->all(), [
            'percentage'    =>  'required|integer',
            'effectivity_date'  =>  'required'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors' => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try{
            $longevity = Longevity::create([
                'percentage' => $request->percentage,
                'effectivity_date' => $request->effectivity_date,
                'status'    =>  Status::INACTIVE
            ]);

            $this->logCustomMessage(
                'create_longevity_pay_percentage',
                $longevity,
                Auth::user()->name . ' created a new longevity pay percentage: ' . $longevity->percentage . '%',
                $longevity,
                LongevityPayLogType::CREATE_SETUP,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => __('responses.success.create')
            ]);
        } catch(Exception $e){
            DB::rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }

    public function update(Request $request) {
        $validate = Validator::make($request->all(), [
            'id'    => 'required|integer',
            'percentage'    =>  'required|integer',
            'effectivity_date'  =>  'required'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors' => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();
        try{
            
            $longevity = Longevity::find($request->id);
            $oldData = clone $longevity;
            $longevity->whereId($request->id)->update([
                'percentage' => $request->percentage,
                'effectivity_date' => $request->effectivity_date
            ]);

            $longevity->old = collect($oldData);
            $longevity->attributes = collect($longevity);

            $this->logCustomMessage(
                'update_longevity_pay_percentage',
                $longevity,
                Auth::user()->name . ' updated the longevity pay percentage: ' . $longevity->percentage . '%',
                $longevity,
                LongevityPayLogType::UPDATE_SETUP,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => __('responses.success.update')
            ]);
        } catch(Exception $e){
            DB::rollBack();
            return response()->json([
                'errors'    => [__('responses.exception')],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);
        }
    }

    public function get_active() {
        return Longevity::where('status', Status::ACTIVE)->first();
    }

    public function set_as_active(Request $request) {
        $validate = Validator::make($request->all(), [
            'id'    => 'required|numeric'
        ]);

        if($validate->fails()) {
            return response()->json([
                'errors'    => $validate->errors()
            ], 400);
        }

        DB::beginTransaction();

        try{
            $data = Longevity::findOrFail($request->id);
            $oldData = clone $data;
            $active = Longevity::where('status', Status::ACTIVE)->first();

            $data->update(['status' => Status::ACTIVE]);
            if($active){
                $active->update(['status' => Status::INACTIVE]);
            }

            $data->old = collect($oldData);
            $data->attributes = collect($data);

            $this->logCustomMessage(
                'active_longevity_pay_percentage',
                $data,
                Auth::user()->name . ' set as active longevity pay percentage: ' . $data->percentage . '%',
                $data,
                LongevityPayLogType::ACTIVE,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  => 'Successfully set as the active longevity pay percentage'
            ]);
        } catch(Exception $e) {
            DB::rollBack();
            return response()->json([
                'errors'    => ['There was a problem in selecting the longevity pay percentage.'],
                'message'   => $e->getMessage(),
                'line'      => $e->getLine()
            ], 500);  
        }
    }


}
