<?php

namespace Suiterus\Adg\Controllers\SM;

use Exception;
use Illuminate\Http\Request;
use Suiterus\Adg\Models\SM\Loan;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use Illuminate\Database\Eloquent\ModelNotFoundException as ME;
use Suiterus\Adg\Models\Activity\Activity;
use App\Traits\Logs\HasCustomLogs;
use App\Enums\Log\UserLoanLogType;
use Illuminate\Support\Facades\Auth;

class LoanController extends Controller
{
    use HasCustomLogs;

    public function create_loan(Request $req) {

        $valid = Validator::make($req->all(),[
            'name'    =>  'required|string|unique:adg_db.loans,name,NULL,id,deleted_at,NULL',
            'status' => 'integer|required'
        ],[
            'name.unique'   =>  $req->name.' is already taken.',
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try {
            $loan = Loan::create([
                'name' => $req->name,
                'description' => $req->description,
                'status' => $req->status
            ]);

            $this->logCustomMessage(
                'create_loan',
                $loan,
                Auth::user()->name . ' created loan: ' . $loan->name,
                $loan,
                UserLoanLogType::CREATE_LOAN,
                new Activity()
            );

            DB::commit();
            return response()->json([
                'text'  =>  $req->name.' created successfully.'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01' ],
                'msg'   =>  $e->getMessage()
            ],500);
        }
    }

    public function init_list_loan(Request $req){
        $paginate = $req->page_count ? intval($req->page_count) : env('DEFAULT_PAGECOUNT');
        return response()->json([
            'data'  =>  Loan::orderBy('id', 'desc')->paginate($paginate)
        ]);
    }

    public function init_list_all_loan(){
        return Loan::all();
    }

    public function search_loan_name(Request $req){
        $paginate = $req->page_count ? intval($req->page_count) : env('DEFAULT_PAGECOUNT');
        return response()->json([
            'data'  =>  Loan::where('name', 'LIKE', '%'.$req->keyword.'%')->paginate($paginate)
        ]);
    }

    public function edit_loan(Request $req){
        $valid = Validator::make($req->all(),[
            'name'      =>  'required|string|min:4|unique:adg_db.loans,name,'.$req->id.',id,deleted_at,NULL',
            'status'    => 'integer|required'
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try{
            try{
                $perms = Loan::findOrFail($req->id);
                $perms->name = $req->name;
                $perms->description = $req->description;
                $perms->status = $req->status;
                $perms->save();

                $updatedLoan = Loan::findOrFail($req->id);

                $updatedLoan->old = collect($perms);
                $updatedLoan->attributes = collect($updatedLoan);

                $this->logCustomMessage(
                    'update_loan',
                    $updatedLoan,
                    Auth::user()->name . ' updated loan: ' . $updatedLoan->name,
                    $updatedLoan,
                    UserLoanLogType::UPDATE_LOAN,
                    new Activity()
                );

                DB::commit();
                return response()->json([
                    'text'  =>  $req->name.' has been updated.'
                ]);
            }catch(ME $ee){
                DB::rollback();
                return response()->json([
                    'errors'    =>  [ "Loan doesn't exists." ],
                ],400);
            }
        }catch(Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'There is a problem in updating a Loan.' ],
                'msg'       =>  $e->getMessage()
            ],500);
        }
    }

    public function delete_loan(Request $req){
        DB::beginTransaction();
        try{
            $perm = Loan::findOrFail($req->id);

            $this->logCustomMessage(
                'update_loan',
                $perm,
                Auth::user()->name . ' deleted loan: ' . $perm->name,
                $perm,
                UserLoanLogType::DELETE_LOAN,
                new Activity()
            );

            $perm->delete();
            
            DB::commit();
            return response()->json([
                'text'  =>  'Loan has been deleted.'
            ]);
        }catch(ME $ee){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ "Loan doesn't exists." ],
            ],400);
        }
    }
}
