<?php

namespace App\Http\Controllers;


use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Suiterus\Hrjp\Models\ItemCode;
use Suiterus\Adg\Models\Activity\Activity;
use App\Traits\Logs\HasCustomLogs;
use App\Enums\Log\ItemCodeLogType;
use Illuminate\Support\Facades\Auth;

class ItemCodeController extends Controller
{
    use HasCustomLogs;
    private $db;

    function __construct()
    {
        $this->db = DB::connection('hrjp_db');
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $paginate = isset($request->paginate) && $request->paginate !== null ? $request->paginate : env('DEFAULT_PAGECOUNT');
        try {

            $data = ItemCode::when(isset($request->search) && $request->search !== null, function ($query) use ($request) {
                $keyword = '%' . $request->search . '%';
                $query->where('item_code', 'LIKE', $keyword);
            })->paginate($paginate);

            return response()->json([
                'data' => $data
            ]);

        } catch (Exception $e) {
            return response()->json([
                'errors' => [__('responses.exception')],
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'item_code' => 'required',
            'position_id' => 'required|exists:' . env('HRJP_DB_CONNECTION') . '.positions,id',
            'status' => 'required|in:1,2,3'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors' => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $itemCode = ItemCode::create([
                'item_code' => $request->item_code,
                'position_id' => $request->position_id,
                'description' => $request->description,
                'status' => $request->status
            ]);

            $this->logCustomMessage(
                'create_item_code',
                $itemCode,
                Auth::user()->name . ' created an item code: ' . $itemCode->item_code,
                $itemCode,
                ItemCodeLogType::CREATE,
                new Activity()
            );

            $this->db->commit();
            return response()->json([
                'text' => __('responses.success.create')
            ]);

        } catch (Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors' => [__('responses.exception')],
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'id' => 'required|exists:' . env('HRJP_DB_CONNECTION') . '.item_codes,id',
            'item_code' => 'required',
            'position_id' => 'required|exists:' . env('HRJP_DB_CONNECTION') . '.positions,id',
            'status' => 'required|in:1,2,3'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors' => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $item_code = ItemCode::find($request->id);
            $item_code->update([
                'item_code' => $request->item_code,
                'position_id' => $request->position_id,
                'description' => $request->description,
                'status' => $request->status
            ]);

            $item_code->save();

            $updatedItemCode = ItemCode::find($request->id);
            $updatedItemCode->old = collect($item_code);
            $updatedItemCode->attributes = collect($updatedItemCode);

            $this->logCustomMessage(
                'update_item_code',
                $updatedItemCode,
                Auth::user()->name . ' updated the item code: ' . $updatedItemCode->item_code,
                $updatedItemCode,
                ItemCodeLogType::UPDATE,
                new Activity()
            );

            $this->db->commit();
            return response()->json([
                'text' => __('responses.success.update')
            ]);

        } catch (Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors' => [__('responses.exception')],
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'id' => 'required|exists:' . env('HRJP_DB_CONNECTION') . '.item_codes,id',
        ]);

        if ($validate->fails()) {
            return response()->json([
                'errors' => $validate->errors()
            ], 400);
        }

        $this->db->beginTransaction();
        try {

            $item_code = ItemCode::find($request->id);

            $this->logCustomMessage(
                'delete_item_code',
                $item_code,
                Auth::user()->name . ' deleted the item code: ' . $item_code->item_code,
                $item_code,
                ItemCodeLogType::DELETE,
                new Activity()
            );

            $item_code->delete();

            $this->db->commit();
            return response()->json([
                'text' => __('responses.success.update')
            ]);

        } catch (Exception $e) {
            $this->db->rollBack();
            return response()->json([
                'errors' => [__('responses.exception')],
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function getByPosition(Request $request){
        return ItemCode::where('position_id', $request->position_id)->get();
    }
}
