<?php

namespace Suiterus\Adg\Controllers\SM;

use DB;
use Exception;
use Validator;
use Illuminate\Http\Request;
use App\Enums\Log\HolidayLogType;
use App\Traits\Logs\HasCustomLogs;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\SM\HolidayType;
use Suiterus\Adg\Models\Activity\Activity;


class HolidayTypeController extends Controller
{
    use HasCustomLogs;
    
    public function create(Request $request) {

        $valid = Validator::make($request->all(),[
            'title'    => 'required|string|unique:adg_db.holiday_types,title,NULL,id,deleted_at,NULL',
            'status'   => 'integer|required'
            
        ],[
            'title.unique'   =>  $request->title.' is already taken.',
        ]);

        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }

        DB::beginTransaction();
        try {
            $holidayType = HolidayType::create([
                'title'         =>  $request->title,
                'description'   =>  $request->description,
                'status'        =>  $request->status, 
                'created_by'    =>  Auth::id()
            ]);

            $this->logCustomMessage(
                'create_holiday_type',
                $holidayType,
                Auth::user()->name . ' created a Holiday Type: ' . $request->title,
                $holidayType,
                HolidayLogType::CREATE_TYPE,
                new Activity()
            );
            
            DB::commit();
            return response()->json([
                'text'  =>  $request->title.' created successfully!'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01' ],
                'msg'       =>  $e->getMessage(),
                'line'      => $e->getLine()
            ],500);
        }
    }

    public function init_list(){
        return response()->json([
            'data'  =>  HolidayType::orderBy('id', 'desc')->paginate(5)
        ]);
    }

    public function search_type(Request $request){
        return response()->json([
            'data'  =>  HolidayType::where('title', 'LIKE', '%'.$request->keyword.'%')->paginate(5)
        ]);
    }

    public function edit(Request $request){
        $valid = Validator::make($request->all(),[
            'id'      =>  'required|integer|exists:adg_db.holiday_types,id',
            'title'   =>  'required|string|min:4|unique:adg_db.holiday_types,title,'.$request->id.',id,deleted_at,NULL',
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try{
            try{
                $perms = HolidayType::findOrFail($request->id);
                $perms->title       = $request->title;
                $perms->description = $request->description;
                $perms->status      = $request->status;

                $updated = HolidayType::find($request->id);
                $updated->old = collect($perms);
                $updated->attributes = collect($updated);

                $this->logCustomMessage(
                    'update_holiday_type',
                    $updated,
                    Auth::user()->name . ' updated the Holiday Type: ' . $request->title,
                    $updated,
                    HolidayLogType::UPDATE_TYPE,
                    new Activity()
                );

                $perms->save();
                DB::commit();
                return response()->json([
                    'text'  =>  $request->title.' has been updated.'
                ]);
            }catch(ME $ee){
                DB::rollback();
                return response()->json([
                    'errors'    =>  [ 'Holiday type doesnt exists.' ],
                ],400);
            }
        }catch(Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'There is a problem in updating a holiday type.' ],
                'msg'       =>  $e->getMessage()
            ],500);
        }
    }

    public function delete(Request $request){
        DB::beginTransaction();
        try{
            $perm = HolidayType::findOrFail($request->id);

            $this->logCustomMessage(
                'deleted_holiday_type',
                $perm,
                Auth::user()->name . ' deleted the Holiday Type: ' . $perm->title,
                $perm,
                HolidayLogType::DELETE_TYPE,
                new Activity()
            );

            $perm->delete();
            DB::commit();
            return response()->json([
                'text'  =>  'Holiday Type has been deleted.'
            ]);
        }catch(ME $ee){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Holiday Type doesnt exists.' ],
            ],400);
        }
    }
}
