<?php

namespace Suiterus\Adg\Controllers\SM;

use DB;
use Exception;
use Validator;
use App\Enums\Status;
use Illuminate\Http\Request;
use App\Enums\Log\HolidayLogType;
use App\Traits\Logs\HasCustomLogs;
use Suiterus\Adg\Models\SM\Branch;
use Suiterus\Adg\Models\SM\Holiday;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Suiterus\Adg\Models\SM\HolidayType;
use Suiterus\Adg\Models\Activity\Activity;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterLegend;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterDayEmployeeShift;
use Suiterus\Adg\Models\Timekeeping\Roster\RosterHeadEmployeeShift;

class HolidayController extends Controller
{
    use HasCustomLogs;
    
    public function create(Request $request) {

        $valid = Validator::make($request->all(),[
            'title'    => 'required|string|unique:adg_db.holidays,title,NULL,id,deleted_at,NULL',
            'status'   => 'integer|required'
            
        ],[
            'title.unique'   =>  $request->title.' is already taken.',
        ]);

        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }

        DB::beginTransaction();
        try {

           $holiday = Holiday::create([
                'title'             =>  $request->title,
                'description'       =>  $request->description,
                'holiday_type_id'   =>  $request->holiday_type_id,
                'time_effectivity'  =>  $request->time_effectivity,
                'branch_id'         =>  $request->branch_id,
                'purpose'           =>  $request->purpose,
                'date'              =>  $request->date,
                'status'            =>  $request->status, 
                'created_by'        =>  Auth::id()
            ]);
            
            if($request->status == 1) {
                $roster_head_shift = RosterHeadEmployeeShift::whereHas('rosterDay', function ($query) use ($holiday) {
                    $query->whereDate('date', $holiday->date);
                })->get();
                
                $roster_emp_shift = RosterDayEmployeeShift::whereHas('rosterDay', function ($query) use ($holiday) {
                        $query->whereDate('date', $holiday->date);
                })->get();
                
                $legend = RosterLegend::where('key','leave')->value('id');
                
                if ($roster_emp_shift->isNotEmpty()) {
                    $roster_emp_shift->toQuery()->update([
                        //'remarks' => '‎ ', //insert this remarks for frontend to see blank with color.
                        'roster_legend_id' => empty($legend) ? null : $legend
                    ]);
                }
                if ($roster_head_shift->isNotEmpty()) {
                    $roster_head_shift->toQuery()->update([
                        //'remarks' => '‎ ', 
                        'roster_legend_id' => empty($legend) ? null : $legend
                    ]);
                }
            }

            $this->logCustomMessage(
                'create_holiday',
                $holiday,
                Auth::user()->name . ' created a Holiday: ' . $request->title,
                $holiday,
                HolidayLogType::CREATE,
                new Activity()
            );
            
            DB::commit();
            return response()->json([
                'text'  =>  $request->title.' created successfully!'
            ]);
        } catch (Exception $e) {
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Can`t create your entry as of now. Contact the developer to fix it. Error Code : SM-comp-0x01' ],
                'msg'       =>  $e->getMessage(),
                'line'      => $e->getLine()
            ],500);
        }
    }

    public function init_list(Request $request){
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');

        return response()->json([
            'data'  =>  Holiday::orderBy('id', 'desc')->paginate($paginate)
        ]);
    }

    public function fetch_holidays(){

        return Holiday::where('status', Status::ACTIVE)
        ->where('date', '>=', now()->startOfDay()) // Filter for today and upcoming dates
        ->orderBy('date', 'asc')
        ->get();

    }

    public function search(Request $request){
        $paginate = $request->page_count ? intval($request->page_count) : env('DEFAULT_PAGECOUNT');

        return response()->json([
            'data'  =>  Holiday::where('title', 'LIKE', '%'.$request->keyword.'%')->paginate($paginate)
        ]);
    }

    public function edit(Request $request){
        $valid = Validator::make($request->all(),[
            'id'      =>  'required|integer|exists:adg_db.holidays,id',
            'title'   =>  'required|string|min:4|unique:adg_db.holidays,title,'.$request->id.',id,deleted_at,NULL',
        ]);
        if($valid->fails()){
            return response()->json([
                'errors'    =>  $valid->errors()
            ],400);
        }
        DB::beginTransaction();
        try{
            try{
                $perms = Holiday::findOrFail($request->id);
                
                if($request->status == 1) { //only do this block of code if the status of holiday is set to active
                    $roster_head_shift = RosterHeadEmployeeShift::whereHas('rosterDay', function ($query) use ($request) {
                        $query->whereDate('date', $request->date);
                    })->get();
                    
                    $roster_emp_shift = RosterDayEmployeeShift::whereHas('rosterDay', function ($query) use ($request) {
                            $query->whereDate('date', $request->date);
                    })->get();
                    
                    $legend = RosterLegend::where('key','leave')->value('id');
                    
                    if ($roster_emp_shift->isNotEmpty()) {
                        $roster_emp_shift->toQuery()->update([
                            //'remarks' => '‎ ',  //insert this remarks for frontend to see blank with color 
                            'roster_legend_id' => empty($legend) ? null : $legend
                        ]);
                    }
                    if ($roster_head_shift->isNotEmpty()) {
                        $roster_head_shift->toQuery()->update([
                            //'remarks' => '‎ ', 
                            'roster_legend_id' => empty($legend) ? null : $legend
                        ]);
                    }
                } elseif ($request->status == 2 && $perms->status == 1) {
                     //update roster table to remove holiday if holiday is initially active
                    RosterHeadEmployeeShift::whereHas('rosterDay', function ($query) use ($perms) {
                        $query->whereDate('date', $perms->date);
                    })->update([
                        'roster_legend_id' => null]
                    );
                    
                    RosterDayEmployeeShift::whereHas('rosterDay', function ($query) use ($perms) {
                        $query->whereDate('date', $perms->date);
                    })->update([ 
                        'roster_legend_id' => null
                    ]);
                }
               
                $perms->title = $request->title;
                $perms->description = $request->description;
                $perms->holiday_type_id = $request->holiday_type_id;
                $perms->time_effectivity = $request->time_effectivity;
                $perms->branch_id = $request->branch_id;
                $perms->purpose = $request->purpose;
                $perms->date = $request->date;
                $perms->status = $request->status;

                $updated = Holiday::find($request->id);
                $updated->old = collect($perms);
                $updated->attributes = collect($updated);

                $this->logCustomMessage(
                    'update_holiday',
                    $updated,
                    Auth::user()->name . ' updated the Holiday: ' . $request->title,
                    $updated,
                    HolidayLogType::UPDATE,
                    new Activity()
                );

                $perms->save();

                DB::commit();
                return response()->json([
                    'text'  =>  $request->title.' has been updated.'
                ]);
            }catch(ME $ee){
                DB::rollback();
                return response()->json([
                    'errors'    =>  [ 'Holiday doesnt exists.' ],
                ],400);
            }
        }catch(Exception $e){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'There is a problem in updating a Holiday.' ],
                'msg'       =>  $e->getMessage()
            ],500);
        }
    }

    public function delete(Request $request){
        DB::beginTransaction();
        try{
            $perm = Holiday::findOrFail($request->id);

            $this->logCustomMessage(
                'delete_holiday',
                $perm,
                Auth::user()->name . ' deleted the Holiday: ' . $perm->title,
                $perm,
                HolidayLogType::DELETE,
                new Activity()
            );

            $perm->delete();
            DB::commit();
            return response()->json([
                'text'  =>  'Holiday has been deleted.'
            ]);
        }catch(ME $ee){
            DB::rollback();
            return response()->json([
                'errors'    =>  [ 'Holiday doesnt exists.' ],
            ],400);
        }
    }

    public function fetch_type(){
        return response()->json([
            'data'  =>  HolidayType::orderBy('id', 'desc')->get()
        ]);
    }
}